<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Time Log Pauses Table
     * 
     * Records each pause/resume event for a time log.
     * A time log can have multiple pauses.
     * 
     * Flow: START → PAUSE → RESUME → PAUSE → RESUME → FINISH
     */
    public function up(): void
    {
        Schema::create('project_time_log_pauses', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('time_log_id')->comment('Parent time log');
            
            // Pause Event
            $table->timestamp('paused_at')->comment('When timer was paused');
            $table->decimal('pause_latitude', 10, 8)->nullable()->comment('GPS lat at pause');
            $table->decimal('pause_longitude', 11, 8)->nullable()->comment('GPS lng at pause');
            $table->decimal('pause_accuracy', 8, 2)->nullable()->comment('GPS accuracy at pause');
            $table->string('pause_ip_address', 45)->nullable();
            
            // Resume Event (null if not yet resumed)
            $table->timestamp('resumed_at')->nullable()->comment('When timer was resumed');
            $table->decimal('resume_latitude', 10, 8)->nullable()->comment('GPS lat at resume');
            $table->decimal('resume_longitude', 11, 8)->nullable()->comment('GPS lng at resume');
            $table->decimal('resume_accuracy', 8, 2)->nullable()->comment('GPS accuracy at resume');
            $table->string('resume_ip_address', 45)->nullable();
            
            // Calculated pause duration (in seconds for precision)
            $table->integer('pause_duration_seconds')->default(0)->comment('Duration of this pause');
            
            // Optional note for the pause
            $table->string('pause_reason')->nullable()->comment('Why paused (optional)');
            
            $table->timestamps();
            
            // Index for quick lookups
            $table->index('time_log_id');
            $table->index(['time_log_id', 'resumed_at']); // Find active pauses
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('project_time_log_pauses');
    }
};
