<?php

namespace Modules\ProjectManagement\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Modules\ProjectManagement\Models\Project;
use Modules\ProjectManagement\Models\Task;
use Modules\ProjectManagement\Models\TimeLog;
use Modules\ProjectManagement\Models\Milestone;
use Modules\ProjectManagement\Models\Discussion;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Exception;

class DashboardController extends AdminController
{
    /**
     * Display the project management dashboard
     */
    public function index()
    {
        try {
            // Overview Stats
            $stats = [
                'projects' => [
                    'total' => Project::count(),
                    'active' => Project::active()->count(),
                    'pinned' => Project::pinned()->count(),
                    'my_projects' => Project::byTeamMember($this->admin->id)->count(),
                ],
                'tasks' => [
                    'total' => Task::count(),
                    'my_tasks' => Task::byOwner($this->admin->id)->count(),
                    'completed' => Task::completed()->count(),
                    'overdue' => Task::overdue()->count(),
                ],
                'time' => [
                    'total_hours' => round(TimeLog::sum('duration_minutes') / 60, 1),
                    'billable_hours' => round(TimeLog::billable()->sum('duration_minutes') / 60, 1),
                    'my_hours' => round(TimeLog::byAdmin($this->admin->id)->sum('duration_minutes') / 60, 1),
                    'running_timers' => TimeLog::where('is_running', true)->count(),
                ],
                'milestones' => [
                    'total' => Milestone::count(),
                    'completed' => Milestone::completed()->count(),
                    'incomplete' => Milestone::incomplete()->count(),
                    'overdue' => Milestone::incomplete()->where('target_date', '<', now())->count(),
                ],
                'discussions' => [
                    'total' => Discussion::count(),
                    'my_discussions' => Discussion::where('created_by', $this->admin->id)->count(),
                ],
            ];

            // Recent Activity
            $recentProjects = Project::with('status')
                ->orderBy('created_at', 'desc')
                ->limit(5)
                ->get();

            $myActiveTasks = Task::with(['status', 'project'])
                ->byOwner($this->admin->id)
                ->whereHas('status', function($q) {
                    $q->where('is_completed', false);
                })
                ->orderBy('due_date', 'asc')
                ->limit(5)
                ->get();

            $runningTimers = TimeLog::with(['task.project', 'admin'])
                ->where('is_running', true)
                ->orderBy('start_time', 'desc')
                ->limit(5)
                ->get();

            return $this->View('projectmanagement::dashboard.index', compact(
                'stats',
                'recentProjects',
                'myActiveTasks',
                'runningTimers'
            ));

        } catch (Exception $e) {
            Log::error('DashboardController@index Error: ' . $e->getMessage());
            return back()->with('error', 'Failed to load dashboard');
        }
    }

    /**
     * Get chart data for AJAX
     */
    public function chartData(Request $request): JsonResponse
    {
        try {
            $chartType = $request->input('chart');

            switch ($chartType) {
                case 'project-status':
                    return $this->getProjectStatusChart();
                
                case 'task-priority':
                    return $this->getTaskPriorityChart();
                
                case 'task-status':
                    return $this->getTaskStatusChart();
                
                case 'time-tracking':
                    return $this->getTimeTrackingChart();
                
                case 'project-progress':
                    return $this->getProjectProgressChart();
                
                case 'monthly-time':
                    return $this->getMonthlyTimeChart();
                
                default:
                    return response()->json(['error' => 'Invalid chart type'], 400);
            }

        } catch (Exception $e) {
            Log::error('DashboardController@chartData Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to load chart data'], 500);
        }
    }

    /**
     * Project Status Distribution (Pie Chart)
     */
    private function getProjectStatusChart(): JsonResponse
    {
        $data = Project::select('project_statuses.label', DB::raw('count(*) as count'))
            ->join('project_statuses', 'projects.status_id', '=', 'project_statuses.id')
            ->groupBy('project_statuses.label', 'project_statuses.color')
            ->get();

        return response()->json([
            'labels' => $data->pluck('label'),
            'data' => $data->pluck('count'),
            'colors' => ['#3b82f6', '#10b981', '#f59e0b', '#ef4444', '#8b5cf6'],
        ]);
    }

    /**
     * Task Priority Distribution (Pie Chart)
     */
    private function getTaskPriorityChart(): JsonResponse
    {
        $priorities = ['low', 'medium', 'high', 'critical'];
        $data = [];
        
        foreach ($priorities as $priority) {
            $data[] = Task::where('priority', $priority)->count();
        }

        return response()->json([
            'labels' => ['Low', 'Medium', 'High', 'Critical'],
            'data' => $data,
            'colors' => ['#10b981', '#3b82f6', '#f59e0b', '#ef4444'],
        ]);
    }

    /**
     * Task Status Distribution (Donut Chart)
     */
    private function getTaskStatusChart(): JsonResponse
    {
        $data = Task::select('task_statuses.label', DB::raw('count(*) as count'))
            ->join('task_statuses', 'tasks.status_id', '=', 'task_statuses.id')
            ->groupBy('task_statuses.label')
            ->get();

        return response()->json([
            'labels' => $data->pluck('label'),
            'data' => $data->pluck('count'),
            'colors' => ['#6366f1', '#8b5cf6', '#ec4899', '#f43f5e', '#10b981'],
        ]);
    }

    /**
     * Time Tracking - Billable vs Non-billable (Bar Chart)
     */
    private function getTimeTrackingChart(): JsonResponse
    {
        $billable = round(TimeLog::where('is_billable', true)->sum('duration_minutes') / 60, 1);
        $nonBillable = round(TimeLog::where('is_billable', false)->sum('duration_minutes') / 60, 1);

        return response()->json([
            'labels' => ['Billable', 'Non-Billable'],
            'data' => [$billable, $nonBillable],
            'colors' => ['#10b981', '#94a3b8'],
        ]);
    }

    /**
     * Project Progress - Top 10 Projects (Horizontal Bar Chart)
     */
    private function getProjectProgressChart(): JsonResponse
    {
        $projects = Project::with('tasks')
            ->active()
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get()
            ->map(function ($project) {
                $totalTasks = $project->tasks->count();
                $completedTasks = $project->tasks->where('status.is_completed', true)->count();
                $progress = $totalTasks > 0 ? round(($completedTasks / $totalTasks) * 100, 1) : 0;

                return [
                    'name' => $project->title,
                    'progress' => $progress,
                ];
            });

        return response()->json([
            'labels' => $projects->pluck('name'),
            'data' => $projects->pluck('progress'),
            'colors' => array_fill(0, 10, '#3b82f6'),
        ]);
    }

    /**
     * Monthly Time Tracking Trend (Line Chart)
     */
    private function getMonthlyTimeChart(): JsonResponse
    {
        $monthlyData = TimeLog::select(
                DB::raw('DATE_FORMAT(start_time, "%Y-%m") as month'),
                DB::raw('ROUND(SUM(duration_minutes) / 60, 1) as hours')
            )
            ->where('start_time', '>=', now()->subMonths(6))
            ->groupBy('month')
            ->orderBy('month', 'asc')
            ->get();

        return response()->json([
            'labels' => $monthlyData->pluck('month')->map(function($month) {
                return \Carbon\Carbon::parse($month . '-01')->format('M Y');
            }),
            'data' => $monthlyData->pluck('hours'),
            'colors' => ['#3b82f6'],
        ]);
    }

    /**
     * Refresh dashboard stats via AJAX
     */
    public function refreshStats(): JsonResponse
    {
        try {
            return response()->json([
                'projects' => [
                    'total' => Project::count(),
                    'active' => Project::active()->count(),
                    'pinned' => Project::pinned()->count(),
                    'my_projects' => Project::byTeamMember($this->admin->id)->count(),
                ],
                'tasks' => [
                    'total' => Task::count(),
                    'my_tasks' => Task::byOwner($this->admin->id)->count(),
                    'completed' => Task::completed()->count(),
                    'overdue' => Task::overdue()->count(),
                ],
                'time' => [
                    'total_hours' => round(TimeLog::sum('duration_minutes') / 60, 1),
                    'billable_hours' => round(TimeLog::billable()->sum('duration_minutes') / 60, 1),
                    'my_hours' => round(TimeLog::byAdmin($this->admin->id)->sum('duration_minutes') / 60, 1),
                    'running_timers' => TimeLog::where('is_running', true)->count(),
                ],
                'milestones' => [
                    'total' => Milestone::count(),
                    'completed' => Milestone::completed()->count(),
                    'incomplete' => Milestone::incomplete()->count(),
                    'overdue' => Milestone::incomplete()->where('target_date', '<', now())->count(),
                ],
                'discussions' => [
                    'total' => Discussion::count(),
                    'my_discussions' => Discussion::where('created_by', $this->admin->id)->count(),
                ],
            ]);

        } catch (Exception $e) {
            Log::error('DashboardController@refreshStats Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to refresh stats'], 500);
        }
    }
}