<?php

namespace Modules\ProjectManagement\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Modules\ProjectManagement\Models\Discussion;
use Modules\ProjectManagement\Models\Project;
use Modules\ProjectManagement\Models\DiscussionReply;
use Modules\ProjectManagement\Models\DiscussionAttachment;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Exception;

class DiscussionController extends AdminController
{
    /**
     * Allowed file types for uploads
     */
    protected $allowedMimes = [
        'jpg', 'jpeg', 'png', 'gif', 'webp', 'svg',
        'pdf', 'doc', 'docx', 'xls', 'xlsx', 'csv', 'ppt', 'pptx',
        'txt', 'rtf', 'zip', 'rar', '7z',
        'html', 'css', 'js', 'json', 'xml'
    ];

    /**
     * Max file size in KB (10MB)
     */
    protected $maxFileSize = 10240;

    public function index()
    {
        $stats = [
            'total' => Discussion::count(),
            'portal_visible' => Discussion::portalVisible()->count(),
            'my_discussions' => Discussion::where('created_by', $this->admin->id)->count(),
        ];

        return $this->View('projectmanagement::discussions.index', compact('stats'));
    }

    public function dataTable(Request $request): JsonResponse
    {
        $query = Discussion::with(['project', 'creator']);

        if ($search = $request->input('search')) {
            $query->where(function($q) use ($search) {
                $q->where('title', 'LIKE', "%{$search}%")
                  ->orWhere('message', 'LIKE', "%{$search}%");
            });
        }

        if ($filters = $request->input('filters')) {
            $decoded = is_array($filters) ? $filters : json_decode($filters, true);
            foreach ($decoded ?? [] as $key => $value) {
                if ($value !== '' && $value !== null) {
                    if ($key === 'portal_visible') {
                        $query->where('portal_visible', $value);
                    } elseif ($key === 'project_id') {
                        $query->where('project_id', $value);
                    }
                }
            }
        }

        $query->orderBy(
            $request->input('sort', 'created_at'),
            $request->input('dir', 'desc')
        );

        $data = $query->paginate($request->input('per_page', 15));

        $items = collect($data->items())->map(function ($item) {
            $item->_show_url = route('admin.discussions.show', $item->id);
            $item->project_display = $item->project ? $item->project->title : '-';
            $item->creator_display = $item->creator ? $item->creator->name : 'Unknown';
            $item->created_at_display = $item->created_at 
                ? $item->created_at->format('M d, Y H:i')
                : '-';
            $item->portal_display = $item->portal_visible ? 'Visible' : 'Hidden';
            $item->replies_count = $item->replies()->count();
            $item->attachments_count = $item->attachments()->count();
            
            return $item;
        });

        return response()->json([
            'data' => $items,
            'total' => $data->total(),
            'current_page' => $data->currentPage(),
            'last_page' => $data->lastPage(),
        ]);
    }

    public function stats(): JsonResponse
    {
        try {
            $stats = [
                'total' => Discussion::count(),
                'portal_visible' => Discussion::where('portal_visible', true)->count(),
                'my_discussions' => Discussion::where('created_by', $this->admin->id)->count(),
            ];

            return response()->json($stats);

        } catch (Exception $e) {
            Log::error('DiscussionController@stats Error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to load stats'
            ], 500);
        }
    }

    public function create()
    {
        $projects = Project::active()->orderBy('title')->get();

        return $this->View('projectmanagement::discussions.create', compact('projects'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'project_id' => 'required|exists:projects,id',
            'title' => 'required|string|max:191',
            'message' => 'required|string',
            'portal_visible' => 'boolean',
            'attachments' => 'nullable|array',
            'attachments.*' => 'file|max:' . $this->maxFileSize . '|mimes:' . implode(',', $this->allowedMimes),
        ]);

        $validated['created_by'] = $this->admin->id;

        // Remove attachments from validated data before creating discussion
        $attachmentFiles = $request->file('attachments', []);
        unset($validated['attachments']);

        $discussion = Discussion::create($validated);

        // Handle file uploads
        if (!empty($attachmentFiles)) {
            $this->handleFileUploads($discussion, $attachmentFiles);
        }

        return redirect()->route('admin.discussions.show', $discussion->id)
            ->with('success', 'Discussion created successfully!');
    }

    public function show($id)
    {
        $discussion = Discussion::with(['project', 'creator', 'replies.creator', 'attachments'])
            ->findOrFail($id);

        return $this->View('projectmanagement::discussions.show', compact('discussion'));
    }

    public function destroy($id)
    {
        $discussion = Discussion::findOrFail($id);
        
        // Attachments will be deleted via model events
        $discussion->attachments()->delete();
        $discussion->delete();

        if (request()->ajax()) {
            return response()->json(['success' => true, 'message' => 'Discussion deleted successfully!']);
        }

        return redirect()->route('admin.discussions.index')
            ->with('success', 'Discussion deleted successfully!');
    }

    public function bulkDelete(Request $request): JsonResponse
    {
        $ids = $request->input('ids', []);
        if (empty($ids)) {
            return response()->json(['success' => false, 'message' => 'No discussions selected'], 400);
        }

        // Delete attachments first
        DiscussionAttachment::whereIn('discussion_id', $ids)->delete();
        
        $deleted = Discussion::whereIn('id', $ids)->delete();
        return response()->json(['success' => true, 'message' => "{$deleted} discussions deleted!"]);
    }

    public function addReply(Request $request, $id)
    {
        $discussion = Discussion::findOrFail($id);
        
        $validated = $request->validate([
            'message' => 'required|string',
        ]);

        $reply = DiscussionReply::create([
            'discussion_id' => $discussion->id,
            'message' => $validated['message'],
            'created_by' => $this->admin->id,
        ]);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Reply added!',
                'reply' => $reply->load('creator')
            ]);
        }

        return back()->with('success', 'Reply added successfully!');
    }

    public function deleteReply($id, $replyId)
    {
        $discussion = Discussion::findOrFail($id);
        $reply = DiscussionReply::where('discussion_id', $discussion->id)
            ->findOrFail($replyId);
            
        $reply->delete();

        return response()->json([
            'success' => true,
            'message' => 'Reply deleted!'
        ]);
    }

    // ==================== ATTACHMENT METHODS ====================

    /**
     * Handle file uploads for discussion
     */
    protected function handleFileUploads(Discussion $discussion, array $files): void
    {
        foreach ($files as $file) {
            if (!$file || !$file->isValid()) {
                continue;
            }

            try {
                // Generate unique filename
                $originalName = $file->getClientOriginalName();
                $extension = $file->getClientOriginalExtension();
                $fileName = pathinfo($originalName, PATHINFO_FILENAME);
                $uniqueName = $fileName . '_' . time() . '_' . uniqid() . '.' . $extension;

                // Store file
                $path = $file->storeAs(
                    'discussions/' . $discussion->id,
                    $uniqueName,
                    'public'
                );

                // Create attachment record
                DiscussionAttachment::create([
                    'discussion_id' => $discussion->id,
                    'file_name' => $originalName,
                    'file_path' => $path,
                    'file_type' => $file->getMimeType(),
                    'file_size' => $file->getSize(),
                    'uploaded_by' => $this->admin->id,
                ]);

            } catch (Exception $e) {
                Log::error('Discussion file upload error: ' . $e->getMessage());
            }
        }
    }

    /**
     * Download attachment
     */
    public function downloadAttachment($attachmentId)
    {
        $attachment = DiscussionAttachment::findOrFail($attachmentId);

        if (!Storage::disk('public')->exists($attachment->file_path)) {
            abort(404, 'File not found');
        }

        return Storage::disk('public')->download(
            $attachment->file_path,
            $attachment->file_name
        );
    }

    /**
     * Preview attachment (for images)
     */
    public function previewAttachment($attachmentId)
    {
        $attachment = DiscussionAttachment::findOrFail($attachmentId);

        if (!Storage::disk('public')->exists($attachment->file_path)) {
            abort(404, 'File not found');
        }

        $file = Storage::disk('public')->get($attachment->file_path);
        $mimeType = $attachment->file_type ?? 'application/octet-stream';

        return response($file, 200)
            ->header('Content-Type', $mimeType)
            ->header('Content-Disposition', 'inline; filename="' . $attachment->file_name . '"');
    }

    /**
     * Delete single attachment
     */
    public function deleteAttachment($attachmentId): JsonResponse
    {
        try {
            $attachment = DiscussionAttachment::findOrFail($attachmentId);
            $attachment->delete(); // This will also delete the file via model boot method

            return response()->json([
                'success' => true,
                'message' => 'Attachment deleted successfully!'
            ]);

        } catch (Exception $e) {
            Log::error('Delete attachment error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete attachment'
            ], 500);
        }
    }

    /**
     * Upload additional attachments to existing discussion
     */
    public function uploadAttachments(Request $request, $id): JsonResponse
    {
        $discussion = Discussion::findOrFail($id);

        $request->validate([
            'attachments' => 'required|array',
            'attachments.*' => 'file|max:' . $this->maxFileSize . '|mimes:' . implode(',', $this->allowedMimes),
        ]);

        $files = $request->file('attachments', []);
        $uploaded = [];

        foreach ($files as $file) {
            if (!$file || !$file->isValid()) {
                continue;
            }

            try {
                $originalName = $file->getClientOriginalName();
                $extension = $file->getClientOriginalExtension();
                $fileName = pathinfo($originalName, PATHINFO_FILENAME);
                $uniqueName = $fileName . '_' . time() . '_' . uniqid() . '.' . $extension;

                $path = $file->storeAs(
                    'discussions/' . $discussion->id,
                    $uniqueName,
                    'public'
                );

                $attachment = DiscussionAttachment::create([
                    'discussion_id' => $discussion->id,
                    'file_name' => $originalName,
                    'file_path' => $path,
                    'file_type' => $file->getMimeType(),
                    'file_size' => $file->getSize(),
                    'uploaded_by' => $this->admin->id,
                ]);

                $uploaded[] = [
                    'id' => $attachment->id,
                    'file_name' => $attachment->file_name,
                    'formatted_size' => $attachment->formatted_size,
                    'icon' => $attachment->icon,
                    'is_image' => $attachment->is_image,
                    'download_url' => $attachment->download_url,
                    'preview_url' => $attachment->preview_url,
                ];

            } catch (Exception $e) {
                Log::error('Additional attachment upload error: ' . $e->getMessage());
            }
        }

        return response()->json([
            'success' => true,
            'message' => count($uploaded) . ' file(s) uploaded successfully!',
            'attachments' => $uploaded,
        ]);
    }
}