<?php

namespace Modules\ProjectManagement\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Modules\ProjectManagement\Models\Milestone;
use Modules\ProjectManagement\Models\Project;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Exception;

class MilestoneController extends AdminController
{
    public function index()
    {
        $stats = [
            'total' => Milestone::count(),
            'completed' => Milestone::completed()->count(),
            'incomplete' => Milestone::incomplete()->count(),
            'overdue' => Milestone::incomplete()->where('target_date', '<', now())->count(),
        ];

        return $this->View('projectmanagement::milestones.index', compact('stats'));
    }

public function dataTable(Request $request): JsonResponse
{
    $query = Milestone::with(['project', 'creator']);

    // Search filter
    if ($search = $request->input('search')) {
        $query->where('name', 'LIKE', "%{$search}%");
    }

    // Handle filter_type from quick filters (All, Completed, In Progress, Overdue)
    $filterType = $request->input('filter_type', 'all');
    
    if ($filterType === 'completed') {
        $query->where('is_completed', true);
    } elseif ($filterType === 'incomplete') {
        $query->where('is_completed', false);
    } elseif ($filterType === 'overdue') {
        $query->where('is_completed', false)
              ->where('target_date', '<', now());
    }
    // 'all' - no additional filter

    // Handle advanced filters from DataTable filter dropdowns
    if ($filters = $request->input('filters')) {
        $decoded = is_array($filters) ? $filters : json_decode($filters, true);
        foreach ($decoded ?? [] as $key => $value) {
            if ($value !== '' && $value !== null) {
                if ($key === 'is_completed') {
                    $query->where('is_completed', $value);
                } elseif ($key === 'project_id') {
                    $query->where('project_id', $value);
                }
            }
        }
    }

    // Sorting
    $query->orderBy(
        $request->input('sort', 'sort_order'),
        $request->input('dir', 'asc')
    );

    // Pagination
    $data = $query->paginate($request->input('per_page', 15));

    // Map items with display fields and action URLs
    $items = collect($data->items())->map(function ($item) {
        // Add action URLs
        $item->_edit_url = route('admin.milestones.edit', $item->id);
        $item->_show_url = route('admin.milestones.show', $item->id);
        
        // Add display fields for frontend rendering
        $item->project_display = $item->project ? $item->project->title : '-';
        
        // Format target date
        $item->target_date_display = $item->target_date 
            ? \Carbon\Carbon::parse($item->target_date)->format('M d, Y')
            : '-';
        
        // Status display
        $item->status_display = $item->is_completed ? 'Completed' : 'Incomplete';
        
        // Check if overdue
        $item->is_overdue = !$item->is_completed && $item->target_date 
            && \Carbon\Carbon::parse($item->target_date)->isPast();
        
        // Days remaining/overdue
        if ($item->target_date && !$item->is_completed) {
            $targetDate = \Carbon\Carbon::parse($item->target_date);
            $now = \Carbon\Carbon::now();
            
            if ($targetDate->isPast()) {
                $item->days_status = 'Overdue by ' . $now->diffInDays($targetDate) . ' days';
            } else {
                $item->days_status = $targetDate->diffInDays($now) . ' days remaining';
            }
        } else {
            $item->days_status = null;
        }
        
        return $item;
    });

    return response()->json([
        'data' => $items,
        'total' => $data->total(),
        'current_page' => $data->currentPage(),
        'last_page' => $data->lastPage(),
    ]);
}

/**
 * Get milestone statistics for AJAX refresh
 */
public function stats(): JsonResponse
{
    try {
        $stats = [
            'total' => Milestone::count(),
            'completed' => Milestone::where('is_completed', true)->count(),
            'incomplete' => Milestone::where('is_completed', false)->count(),
            'overdue' => Milestone::where('is_completed', false)
                ->where('target_date', '<', now())
                ->count(),
        ];

        return response()->json($stats);

    } catch (\Exception $e) {
        \Log::error('MilestoneController@stats Error: ' . $e->getMessage());
        
        return response()->json([
            'success' => false,
            'message' => 'Failed to load stats'
        ], 500);
    }
}

    public function create()
    {
        $projects = Project::active()->orderBy('title')->get();

        return $this->View('projectmanagement::milestones.create', compact('projects'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'project_id' => 'required|exists:projects,id',
            'name' => 'required|string|max:191',
            'description' => 'nullable|string',
            'target_date' => 'nullable|date',
            'portal_visible' => 'boolean',
            'sort_order' => 'nullable|integer',
        ]);

        $validated['created_by'] = $this->admin->id;
        
        if (!isset($validated['sort_order'])) {
            $maxOrder = Milestone::where('project_id', $validated['project_id'])->max('sort_order') ?? 0;
            $validated['sort_order'] = $maxOrder + 1;
        }

        $milestone = Milestone::create($validated);

        return redirect()->route('admin.milestones.index')
            ->with('success', 'Milestone created successfully!');
    }

    /**
 * Display the specified milestone details
 */
    public function show($id)
    {
        $milestone = Milestone::with(['project', 'creator'])->findOrFail($id);

        return $this->View('projectmanagement::milestones.show', compact('milestone'));
    }

    public function edit($id)
    {
        $milestone = Milestone::findOrFail($id);
        $projects = Project::active()->orderBy('title')->get();

        return $this->View('projectmanagement::milestones.edit', compact('milestone', 'projects'));
    }

    public function update(Request $request, $id)
    {
        $milestone = Milestone::findOrFail($id);

        $validated = $request->validate([
            'project_id' => 'required|exists:projects,id',
            'name' => 'required|string|max:191',
            'description' => 'nullable|string',
            'target_date' => 'nullable|date',
            'portal_visible' => 'boolean',
            'sort_order' => 'nullable|integer',
            'is_completed' => 'boolean',
        ]);

        if (isset($validated['is_completed']) && $validated['is_completed'] && !$milestone->is_completed) {
            $validated['completed_at'] = now();
        } elseif (isset($validated['is_completed']) && !$validated['is_completed']) {
            $validated['completed_at'] = null;
        }

        $milestone->update($validated);

        return redirect()->route('admin.milestones.index')
            ->with('success', 'Milestone updated successfully!');
    }

    public function destroy($id)
    {
        $milestone = Milestone::findOrFail($id);
        $milestone->delete();

        if (request()->ajax()) {
            return response()->json(['success' => true, 'message' => 'Milestone deleted successfully!']);
        }

        return redirect()->route('admin.milestones.index')
            ->with('success', 'Milestone deleted successfully!');
    }

    public function bulkDelete(Request $request): JsonResponse
    {
        $ids = $request->input('ids', []);
        if (empty($ids)) {
            return response()->json(['success' => false, 'message' => 'No milestones selected'], 400);
        }

        $deleted = Milestone::whereIn('id', $ids)->delete();
        return response()->json(['success' => true, 'message' => "{$deleted} milestones deleted!"]);
    }
}
