<?php

namespace Modules\ProjectManagement\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Modules\ProjectManagement\Models\Project;
use Modules\ProjectManagement\Models\ProjectStatus;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Exception;

class ProjectController extends AdminController
{
    public function index()
    {
        $stats = [
            'total' => Project::count(),
            'active' => Project::active()->count(),
            'pinned' => Project::pinned()->count(),
            'my_projects' => Project::byTeamMember($this->admin->id)->count(),
        ];

        return $this->View('projectmanagement::projects.index', compact('stats'));
    }

public function dataTable(Request $request): JsonResponse
{
    try {
        // Query PROJECTS, not Tasks!
        $query = Project::with(['status', 'account', 'currency']);

        // Search filter
        if ($search = $request->input('search')) {
            $query->where('title', 'LIKE', "%{$search}%")
                  ->orWhere('scope_notes', 'LIKE', "%{$search}%");
        }

        // Handle filter_type from quick filters (All, Active, Pinned, My Projects)
        $filterType = $request->input('filter_type', 'all');
        
        if ($filterType === 'active') {
            $query->active();
        } elseif ($filterType === 'pinned') {
            $query->pinned();
        } elseif ($filterType === 'my') {
            $query->byTeamMember($this->admin->id);
        }
        // 'all' - no additional filter

        // Advanced filters
        if ($filters = $request->input('filters')) {
            $decoded = is_array($filters) ? $filters : json_decode($filters, true);
            foreach ($decoded ?? [] as $key => $value) {
                if ($value !== '' && $value !== null) {
                    if ($key === 'status_id') {
                        $query->where('status_id', $value);
                    } elseif ($key === 'account_id') {
                        $query->where('account_id', $value);
                    } elseif ($key === 'billing_method') {
                        $query->where('billing_method', $value);
                    }
                }
            }
        }

        // Sorting - Pinned projects go first
        $sortColumn = $request->input('sort', 'created_at');
        $sortDirection = $request->input('dir', 'desc');
        
        $query->orderBy('is_pinned', 'desc') // Pinned first
              ->orderBy($sortColumn, $sortDirection);

        // Pagination
        $data = $query->paginate($request->input('per_page', 15));

        // Map items with display fields and action URLs
        $items = collect($data->items())->map(function ($item) {
            // Add action URLs
            $item->_show_url = route('admin.projects.show', $item->id);
            $item->_edit_url = route('admin.projects.edit', $item->id);
            $item->_finance_url = route('admin.projects.finance', $item->id);
            
            // Add display fields for frontend rendering
            $item->account_display = $item->account ? $item->account->name : '-';
            $item->status_display = $item->status ? $item->status->label : 'Unknown';
            
            // Billing method display
            $billingMethods = [
                'fixed' => 'Fixed Amount',
                'project_hours' => 'Project Hours',
                'task_hours' => 'Task Hours',
            ];
            $item->billing_method_display = $billingMethods[$item->billing_method] ?? ucfirst($item->billing_method);
            
            return $item;
        });

        return response()->json([
            'data' => $items,
            'total' => $data->total(),
            'current_page' => $data->currentPage(),
            'last_page' => $data->lastPage(),
        ]);

    } catch (Exception $e) {
        Log::error('ProjectController@dataTable Error: ' . $e->getMessage(), [
            'trace' => $e->getTraceAsString()
        ]);

        return response()->json([
            'success' => false,
            'message' => 'Failed to load projects data',
            'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
        ], 500);
    }
}



    /**
 * Get project statistics for AJAX updates
 */
public function stats(): JsonResponse
{
    return response()->json([
        'total' => Project::count(),
        'active' => Project::active()->count(),
        'pinned' => Project::pinned()->count(),
        'my_projects' => Project::byTeamMember($this->admin->id)->count(),
    ]);
}

    public function create()
    {
        $statuses = ProjectStatus::where('is_active', true)->orderBy('sort_order')->get();
        
        // Fetch customers
        $customers = \App\Models\Customer::where('active', true)->orderBy('name')->get();
        
        // Check if Currency model exists
        $currencies = [];
        if (class_exists('\App\Models\Currency')) {
            $currencies = \App\Models\Currency::where('is_active', true)->get();
        }
        
        $admins = \App\Models\Admin::where('is_active', true)->orderBy('name')->get();

        return $this->View('projectmanagement::projects.create', compact('statuses', 'customers', 'currencies', 'admins'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:191',
            'account_id' => 'nullable|exists:customers,id',
            'billing_method' => 'required|in:fixed,project_hours,task_hours',
            'fixed_amount' => 'nullable|numeric|min:0',
            'hourly_rate' => 'nullable|numeric|min:0',
            'currency_id' => 'nullable|exists:currencies,id',
            'status_id' => 'required|exists:project_statuses,id',
            'planned_start_date' => 'nullable|date',
            'planned_end_date' => 'nullable|date|after_or_equal:planned_start_date',
            'scope_notes' => 'nullable|string',
            'internal_notes' => 'nullable|string',
            'is_billable' => 'boolean',
            'team' => 'array',
            'team.*' => 'exists:admins,id',
        ]);

        DB::beginTransaction();
        try {
            $validated['created_by'] = $this->admin->id;
            $validated['updated_by'] = $this->admin->id;
            
            $team = $validated['team'] ?? [];
            unset($validated['team']);
            
            $project = Project::create($validated);
            
            if (!empty($team)) {
                $project->team()->sync($team);
            }

            DB::commit();

            return redirect()->route('admin.projects.show', $project->id)
                ->with('success', 'Project created successfully!');
        } catch (\Exception $e) {
            DB::rollback();
            return back()->withInput()->with('error', 'Failed to create project: ' . $e->getMessage());
        }
    }

    public function show($id)
    {
        $with = ['status', 'team', 'tasks.status', 'tasks.owners', 'milestones', 'discussions'];
        if (class_exists('\App\Models\Account')) {
            $with[] = 'account';
        }
        if (class_exists('\App\Models\Currency')) {
            $with[] = 'currency';
        }
        $project = Project::with($with)->findOrFail($id);

        $stats = [
            'total_tasks' => $project->tasks()->count(),
            'completed_tasks' => $project->completed_tasks_count,
            'total_hours' => $project->total_time_logged,
            'billable_hours' => $project->total_billable_hours,
            'billed_hours' => $project->total_billed_hours,
        ];

        return $this->View('projectmanagement::projects.show', compact('project', 'stats'));
    }
    
    /**
 * Display project overview with comprehensive statistics
 */
public function overview($id)
{
    try {
        $project = Project::with(['status', 'team', 'tasks.status', 'tasks.owners', 'tasks.timeLogs'])
            ->findOrFail($id);

        // Calculate project progress based on completed tasks
        $totalTasks = $project->tasks->count();
        $completedTasks = $project->tasks->where('status.status_label', 'Completed')->count();
        $projectProgress = $totalTasks > 0 ? ($completedTasks / $totalTasks) * 100 : 0;

        // Calculate total hours and costs
        $totalHours = $project->tasks->sum(function ($task) {
            return $task->timeLogs->sum('hours_spent');
        });

        $estimatedHours = $project->tasks->sum('estimated_hours') ?? 0;

        // Calculate total cost based on billing method
        $totalCost = 0;
        if ($project->billing_method === 'fixed') {
            $totalCost = $project->fixed_amount ?? 0;
        } elseif ($project->billing_method === 'project_hours') {
            $totalCost = $totalHours * ($project->hourly_rate ?? 0);
        } else { // task_hours
            $totalCost = $project->tasks->sum(function ($task) {
                $hours = $task->timeLogs->sum('hours_spent');
                return $hours * ($task->hourly_rate ?? $task->project->hourly_rate ?? 0);
            });
        }

        // Calculate remaining budget
        $remainingBudget = ($project->fixed_amount ?? 0) - $totalCost;

        // Get team members
        $team = $project->team ?? collect();
        $teamCount = $team->count();

        // Get recent tasks (limit to 10)
        $tasks = $project->tasks()
            ->with(['status', 'owners', 'timeLogs'])
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return $this->View('projectmanagement::projects.overview', compact(
            'project',
            'projectProgress',
            'totalTasks',
            'completedTasks',
            'totalHours',
            'estimatedHours',
            'totalCost',
            'remainingBudget',
            'team',
            'teamCount',
            'tasks'
        ));
    } catch (\Exception $e) {
        \Log::error('Error loading project overview: ' . $e->getMessage());
        return back()->with('error', 'Failed to load project overview.');
    }
}


    public function edit($id)
    {
        $project = Project::with('team')->findOrFail($id);
        $statuses = ProjectStatus::where('is_active', true)->orderBy('sort_order')->get();
        
        // Fetch customers
        $customers = \App\Models\Customer::where('active', true)->orderBy('name')->get();
        
        $currencies = [];
        if (class_exists('\App\Models\Currency')) {
            $currencies = \App\Models\Currency::where('is_active', true)->get();
        }
        
        $admins = \App\Models\Admin::where('is_active', true)->orderBy('name')->get();

        return $this->View('projectmanagement::projects.edit', compact('project', 'statuses', 'customers', 'currencies', 'admins'));
    }

    public function update(Request $request, $id)
    {
        $project = Project::findOrFail($id);

        $validated = $request->validate([
            'title' => 'required|string|max:191',
            'account_id' => 'nullable|exists:customers,id',
            'billing_method' => 'required|in:fixed,project_hours,task_hours',
            'fixed_amount' => 'nullable|numeric|min:0',
            'hourly_rate' => 'nullable|numeric|min:0',
            'currency_id' => 'nullable|exists:currencies,id',
            'status_id' => 'required|exists:project_statuses,id',
            'planned_start_date' => 'nullable|date',
            'planned_end_date' => 'nullable|date|after_or_equal:planned_start_date',
            'scope_notes' => 'nullable|string',
            'internal_notes' => 'nullable|string',
            'is_billable' => 'boolean',
            'team' => 'array',
            'team.*' => 'exists:admins,id',
        ]);

        DB::beginTransaction();
        try {
            $validated['updated_by'] = $this->admin->id;
            
            $team = $validated['team'] ?? [];
            unset($validated['team']);
            
            $project->update($validated);
            
            if (!empty($team)) {
                $project->team()->sync($team);
            }

            DB::commit();

            return redirect()->route('admin.projects.show', $project->id)
                ->with('success', 'Project updated successfully!');
        } catch (\Exception $e) {
            DB::rollback();
            return back()->withInput()->with('error', 'Failed to update project: ' . $e->getMessage());
        }
    }

    public function destroy($id)
    {
        DB::beginTransaction();
        try {
            $project = Project::with(['tasks', 'milestones', 'discussions'])->findOrFail($id);
            
            \Log::info('Deleting project', [
                'project_id' => $id,
                'title' => $project->title,
                'deleted_by' => $this->admin->id
            ]);
            
            // Count relationships for logging
            $stats = [
                'tasks' => $project->tasks()->count(),
                'milestones' => $project->milestones()->count(),
                'discussions' => $project->discussions()->count(),
                'time_logs' => $project->timeLogs()->count(),
            ];
            
            // Delete all tasks (will cascade delete time logs, comments, attachments via model events)
            $project->tasks()->each(function($task) {
                // Delete task comments
                $task->comments()->delete();
                // Delete task attachments (also delete physical files)
                $task->attachments()->each(function($attachment) {
                    \Storage::disk('public')->delete($attachment->file_path);
                    $attachment->delete();
                });
                // Delete time logs
                $task->timeLogs()->delete();
                // Detach owners and followers
                $task->owners()->detach();
                $task->followers()->detach();
                // Delete the task
                $task->delete();
            });
            
            // Delete milestones
            $project->milestones()->delete();
            
            // Delete discussions and their replies
            $project->discussions()->each(function($discussion) {
                $discussion->replies()->delete();
                $discussion->delete();
            });
            
            // Detach team members (pivot table)
            $project->team()->detach();
            
            // Finally, delete the project itself
            $project->delete();
            
            DB::commit();
            
            \Log::info('Project deleted successfully', [
                'project_id' => $id,
                'stats' => $stats
            ]);
            
            if (request()->ajax()) {
                return response()->json([
                    'success' => true, 
                    'message' => 'Project and all related data deleted successfully!',
                    'stats' => $stats
                ]);
            }
            
            return redirect()->route('admin.projects.index')
                ->with('success', 'Project deleted successfully!');
                
        } catch (\Exception $e) {
            DB::rollback();
            
            \Log::error('Project deletion failed', [
                'project_id' => $id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            if (request()->ajax()) {
                return response()->json([
                    'success' => false, 
                    'message' => 'Failed to delete project: ' . $e->getMessage()
                ], 500);
            }
            
            return back()->with('error', 'Failed to delete project');
        }
    }

    public function bulkDelete(Request $request): JsonResponse
    {
        $ids = $request->input('ids', []);
        if (empty($ids)) {
            return response()->json(['success' => false, 'message' => 'No projects selected'], 400);
        }

        $deleted = Project::whereIn('id', $ids)->delete();
        return response()->json(['success' => true, 'message' => "{$deleted} projects deleted!"]);
    }

    public function pin($id)
    {
        $project = Project::findOrFail($id);
        $project->is_pinned = !$project->is_pinned;
        $project->save();

        return response()->json([
            'success' => true,
            'is_pinned' => $project->is_pinned,
            'message' => $project->is_pinned ? 'Project pinned!' : 'Project unpinned!'
        ]);
    }

    public function duplicate($id)
    {
        $original = Project::with(['milestones', 'tasks'])->findOrFail($id);

        DB::beginTransaction();
        try {
            $project = $original->replicate();
            $project->title = $original->title . ' (Copy)';
            $project->created_by = $this->admin->id;
            $project->updated_by = $this->admin->id;
            $project->is_pinned = false;
            $project->save();

            // Copy team members
            $project->team()->sync($original->team->pluck('id'));

            // Copy milestones
            foreach ($original->milestones as $milestone) {
                $newMilestone = $milestone->replicate();
                $newMilestone->project_id = $project->id;
                $newMilestone->is_completed = false;
                $newMilestone->completed_at = null;
                $newMilestone->save();
            }

            // Copy tasks
            foreach ($original->tasks as $task) {
                $newTask = $task->replicate();
                $newTask->project_id = $project->id;
                $newTask->completed_at = null;
                $newTask->save();

                // Copy task owners and followers
                $newTask->owners()->sync($task->owners->pluck('id'));
                $newTask->followers()->sync($task->followers->pluck('id'));
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Project duplicated successfully!',
                'project_id' => $project->id,
                'url' => route('admin.projects.show', $project->id)
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(['success' => false, 'message' => 'Failed to duplicate project'], 500);
        }
    }

    public function finance($id)
    {
        $project = Project::with(['tasks.timeLogs', 'currency'])->findOrFail($id);

        $timeLogs = $project->timeLogs()->with(['task', 'admin'])->get();
        
        $unbilledLogs = $timeLogs->where('is_billed', false)->where('is_billable', true);
        $billedLogs = $timeLogs->where('is_billed', true);

        $stats = [
            'total_hours' => $timeLogs->sum('duration_minutes') / 60,
            'billable_hours' => $timeLogs->where('is_billable', true)->sum('duration_minutes') / 60,
            'unbilled_hours' => $unbilledLogs->sum('duration_minutes') / 60,
            'billed_hours' => $billedLogs->sum('duration_minutes') / 60,
            'unbilled_amount' => $unbilledLogs->sum(function($log) {
                return ($log->duration_minutes / 60) * ($log->hourly_rate ?? 0);
            }),
            'billed_amount' => $billedLogs->sum(function($log) {
                return ($log->duration_minutes / 60) * ($log->hourly_rate ?? 0);
            }),
        ];

        return $this->View('projectmanagement::projects.finance', compact('project', 'stats', 'unbilledLogs', 'billedLogs'));
    }

    public function generateInvoice(Request $request, $id)
    {
        // This would integrate with your invoice module
        // For now, just return a placeholder
        return response()->json([
            'success' => true,
            'message' => 'Invoice generation functionality to be integrated with Invoice module'
        ]);
    }
}