# Project Management Module - Installation & Setup Guide

## Pre-Installation Checklist

Before installing this module, ensure you have:

1. **EchoPx Laravel Application** running and accessible
2. **Admin Panel Access** with appropriate permissions
3. **Database Access** - MySQL or compatible database
4. **PHP 8.0+** and required extensions
5. **Composer** installed (for manual installation)

## Recommended Installation Method: Admin Panel Upload

### Step 1: Prepare the Module

1. Download `ProjectManagement.zip`
2. **Do NOT unzip** the file - upload it as-is

### Step 2: Upload via Admin Panel

1. Login to your admin panel at: `https://your-domain.com/admin`
2. Navigate to: **Settings** → **Modules** (or **System** → **Modules**)
3. Look for an **"Upload Module"** or **"Install Module"** button
4. Click it and select `ProjectManagement.zip`
5. Wait for upload to complete

### Step 3: Enable the Module

1. After upload, you'll see **ProjectManagement** in the modules list
2. Click **"Enable"** button next to ProjectManagement
3. The system will automatically:
   - Extract files to `Modules/ProjectManagement/`
   - Run database migrations
   - Register routes and views
   - Add menu items

### Step 4: Verify Installation

1. **Refresh the Admin Panel** (hard refresh: Ctrl+Shift+R or Cmd+Shift+R)
2. You should see **"Projects"** in the sidebar menu
3. Click on **Projects** → **All Projects**
4. If you see the projects list page, installation succeeded!

### Step 5: Clear Caches (If Needed)

If the module doesn't appear, SSH into your server and run:

```bash
cd /path/to/your/application
php artisan route:clear
php artisan config:clear
php artisan view:clear
php artisan optimize:clear
```

Then refresh the admin panel.

## Alternative: Manual Installation via SSH

If you have SSH access and prefer manual installation:

### Step 1: Extract Module Files

```bash
# Navigate to your application's Modules directory
cd /path/to/your/application/Modules

# Extract the ZIP file
unzip /path/to/ProjectManagement.zip

# Verify extraction
ls -la ProjectManagement/
# You should see: Config, Database, Http, Models, Providers, Resources, Routes, etc.
```

### Step 2: Set Permissions

```bash
# Make sure web server can read the files
chown -R www-data:www-data ProjectManagement
chmod -R 755 ProjectManagement

# Or if using different user:
chown -R your-web-user:your-web-group ProjectManagement
```

### Step 3: Run Migrations

```bash
cd /path/to/your/application

# Run migrations to create database tables
php artisan migrate

# Output should show:
# Migrating: 2024_01_01_000001_create_project_statuses_table
# Migrated:  2024_01_01_000001_create_project_statuses_table
# ... (9 migrations total)
```

### Step 4: Enable Module

```bash
# Enable the module
php artisan module:enable ProjectManagement

# Verify it's enabled
php artisan module:list

# Output should show:
# ┌────────────────────┬─────────┬─────────┐
# │ Name               │ Status  │ Order   │
# ├────────────────────┼─────────┼─────────┤
# │ ProjectManagement  │ Enabled │ 15      │
# └────────────────────┴─────────┴─────────┘
```

### Step 5: Clear All Caches

```bash
# Clear route cache
php artisan route:clear

# Clear config cache
php artisan config:clear

# Clear view cache
php artisan view:clear

# Clear application cache
php artisan cache:clear

# Or clear everything at once
php artisan optimize:clear
```

### Step 6: Verify Routes

```bash
# Check that project routes are registered
php artisan route:list | grep projects

# You should see output like:
# GET|HEAD  admin/projects ...................... admin.projects.index
# POST      admin/projects ...................... admin.projects.store
# GET|HEAD  admin/projects/create ............... admin.projects.create
# GET|HEAD  admin/projects/data ................. admin.projects.data
# ... (many more routes)
```

## Post-Installation Configuration

### 1. Configure Module Settings

Edit `Modules/ProjectManagement/Config/config.php`:

```php
return [
    'per_page' => 15, // Default pagination
    
    // Billing methods available
    'billing_methods' => [
        'fixed' => 'Fixed Amount',
        'project_hours' => 'Project Hours',
        'task_hours' => 'Task Hours',
    ],
    
    // Task priorities
    'priorities' => [
        'low' => 'Low',
        'medium' => 'Medium',
        'high' => 'High',
        'critical' => 'Critical',
    ],
    
    // Timer settings
    'timer' => [
        'auto_stop_hours' => 12, // Auto-stop timers after X hours
        'round_minutes' => 15,   // Round time to nearest X minutes
    ],
    
    // Default portal visibility (can be overridden per project)
    'portal_defaults' => [
        'show_tasks' => false,
        'show_time_logs' => false,
        'show_files' => false,
        'show_discussions' => false,
        'show_milestones' => false,
        'show_activity' => false,
        'show_team' => false,
        'show_finance' => false,
    ],
];
```

### 2. Customize Statuses (Optional)

Default statuses are automatically created, but you can customize them:

**Project Statuses** (database: `project_statuses`):
- Planned (Gray)
- Active (Blue)
- On Hold (Orange)
- Completed (Green)
- Cancelled (Red)

**Task Statuses** (database: `task_statuses`):
- Backlog (Gray)
- To Do (Dark Gray)
- In Progress (Blue)
- Review (Orange)
- Done (Green) - marks task as completed
- Blocked (Red)

To modify:
```sql
-- Example: Add new project status
INSERT INTO project_statuses (label, color, sort_order, show_by_default, is_active)
VALUES ('Under Review', '#9333ea', 6, true, true);

-- Example: Change status color
UPDATE project_statuses SET color = '#10b981' WHERE label = 'Completed';
```

### 3. Set Up Permissions (If Using Role-Based Access)

If your application uses Spatie Laravel Permission or similar:

```sql
-- Example permissions
INSERT INTO permissions (name, guard_name, created_at, updated_at)
VALUES
    ('view_projects', 'admin', NOW(), NOW()),
    ('create_projects', 'admin', NOW(), NOW()),
    ('edit_projects', 'admin', NOW(), NOW()),
    ('delete_projects', 'admin', NOW(), NOW()),
    ('view_tasks', 'admin', NOW(), NOW()),
    ('create_tasks', 'admin', NOW(), NOW()),
    ('edit_tasks', 'admin', NOW(), NOW()),
    ('delete_tasks', 'admin', NOW(), NOW()),
    ('view_time_logs', 'admin', NOW(), NOW()),
    ('create_time_logs', 'admin', NOW(), NOW()),
    ('edit_time_logs', 'admin', NOW(), NOW()),
    ('delete_time_logs', 'admin', NOW(), NOW());
```

Then assign to roles via admin panel or:

```php
$role = Role::findByName('project-manager', 'admin');
$role->givePermissionTo([
    'view_projects',
    'create_projects',
    'edit_projects',
    'view_tasks',
    'create_tasks',
    'edit_tasks',
    'view_time_logs',
]);
```

## Troubleshooting Installation Issues

### Issue: Module Not Appearing in List

**Solution:**
```bash
# Check if module.json exists and is valid
cat Modules/ProjectManagement/module.json

# Should show valid JSON with "active": 1

# If file is missing or corrupted, reinstall module
```

### Issue: "Class Not Found" Errors

**Solution:**
```bash
# Regenerate autoload files
composer dump-autoload

# Clear all caches
php artisan optimize:clear

# Restart PHP-FPM (if applicable)
sudo systemctl restart php8.1-fpm
```

### Issue: Migration Errors

**Common Scenarios:**

**Scenario 1: Table Already Exists**
```
SQLSTATE[42S01]: Base table or view already exists
```

**Solution:** This is usually safe. The migrations check for table existence before creating. If you get this error, the table already exists and you can continue. To verify:

```bash
# Check if tables were created
mysql -u your_user -p your_database -e "SHOW TABLES LIKE 'project%';"

# You should see:
# project_discussions
# project_milestones
# project_statuses
# project_tasks
# project_team
# project_time_logs
# projects
```

**Scenario 2: Column Already Exists**
```
SQLSTATE[42S21]: Column already exists
```

**Solution:** The column exists from a previous installation attempt. Safe to ignore or:

```bash
# Rollback last migration batch
php artisan migrate:rollback --step=1

# Re-run migrations
php artisan migrate
```

**Scenario 3: Foreign Key Constraint Fails**

Our migrations use indexed columns WITHOUT foreign key constraints for maximum compatibility. If you still see foreign key errors:

```bash
# Check MySQL/MariaDB version
mysql --version

# Ensure InnoDB engine (supports transactions)
# Check table engines:
mysql -u your_user -p your_database -e "
SELECT TABLE_NAME, ENGINE 
FROM information_schema.TABLES 
WHERE TABLE_SCHEMA = 'your_database' 
AND TABLE_NAME LIKE 'project%';
"
```

### Issue: Routes Not Working (404 Errors)

**Solution:**
```bash
# Clear route cache
php artisan route:clear

# Verify routes exist
php artisan route:list | grep admin.projects

# If empty, module routes not loaded. Enable module:
php artisan module:enable ProjectManagement

# Clear config cache
php artisan config:clear

# Restart web server
sudo systemctl restart nginx
# OR
sudo systemctl restart apache2
```

### Issue: Views Not Found

**Solution:**
```bash
# Clear view cache
php artisan view:clear

# Verify views exist
ls -la Modules/ProjectManagement/Resources/views/

# Should show:
# menu.blade.php
# projects/
# tasks/
# timelogs/
# milestones/
# discussions/

# If files missing, re-extract module ZIP
```

### Issue: Permissions Denied (File System)

**Solution:**
```bash
# Fix ownership
sudo chown -R www-data:www-data Modules/ProjectManagement

# Fix permissions
sudo chmod -R 755 Modules/ProjectManagement

# Make storage writable
sudo chmod -R 775 storage
sudo chmod -R 775 bootstrap/cache
```

### Issue: Menu Not Showing

**Check 1:** Verify menu.blade.php exists:
```bash
cat Modules/ProjectManagement/Resources/views/menu.blade.php
```

**Check 2:** Clear view cache:
```bash
php artisan view:clear
php artisan config:clear
```

**Check 3:** Verify module is enabled:
```bash
php artisan module:list
# ProjectManagement should show "Enabled"
```

**Check 4:** Hard refresh browser:
- Chrome/Firefox: Ctrl+Shift+R (Windows) or Cmd+Shift+R (Mac)
- Clear browser cache if needed

## Database Schema Verification

After successful installation, verify tables were created:

```sql
-- Check project statuses
SELECT * FROM project_statuses;
-- Should return 5 rows (Planned, Active, On Hold, Completed, Cancelled)

-- Check task statuses
SELECT * FROM task_statuses;
-- Should return 6 rows (Backlog, To Do, In Progress, Review, Done, Blocked)

-- Check main tables exist
SHOW TABLES LIKE 'project%';
-- Should return 8 tables

-- Check pivot tables exist
SHOW TABLES LIKE 'task_%' OR LIKE '%_team';
-- Should return 5 tables (task_owners, task_followers, task_comments, task_attachments, task_checklists, project_team)
```

## First-Time Usage

### 1. Create Your First Project

1. Navigate to **Projects** → **Add Project**
2. Fill in:
   - Title: "Test Project"
   - Account: Select a customer
   - Billing Method: "Fixed" or "Project Hours"
   - Status: "Active"
   - Planned dates
3. Add team members
4. Set portal visibility (what client can see)
5. Click **Create**

### 2. Create a Task

1. Open the project you just created
2. Click **Add Task** or go to **Tasks** → **Add Task**
3. Fill in:
   - Title: "Setup Development Environment"
   - Status: "To Do"
   - Priority: "High"
   - Assign owners
4. Click **Create**

### 3. Track Time

1. Open the task
2. Click **Start Timer**
3. Work on the task
4. Click **Stop Timer** when done
5. Time is automatically logged

### 4. View Finance

1. Open your project
2. Click **Finance** tab
3. Review billable hours and amounts
4. Generate invoice (integrates with Invoice module)

## Uninstallation (If Needed)

To completely remove the module:

```bash
# 1. Disable module
php artisan module:disable ProjectManagement

# 2. Backup data (IMPORTANT!)
mysqldump -u your_user -p your_database \
  project_statuses task_statuses projects project_milestones \
  project_tasks project_time_logs project_discussions \
  discussion_replies task_comments task_attachments task_checklists \
  project_team task_owners task_followers \
  > projectmanagement_backup.sql

# 3. Rollback migrations (removes all tables and data)
php artisan migrate:rollback --path=Modules/ProjectManagement/Database/Migrations

# 4. Remove module files
rm -rf Modules/ProjectManagement

# 5. Clear caches
php artisan optimize:clear
```

## Support & Documentation

- **README.md** - Module overview and features
- **This File** - Detailed installation instructions
- **Views README** - View structure and form fields
- **Controllers README** - Controller implementation guide

For additional help:
1. Check Laravel logs: `storage/logs/laravel.log`
2. Enable debug mode temporarily: Set `APP_DEBUG=true` in `.env`
3. Check PHP error logs
4. Review module configuration: `Modules/ProjectManagement/Config/config.php`

---

**Installation Complete!** You're ready to start managing projects.
