<?php

namespace Modules\ProjectManagement\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;

class Discussion extends Model
{
    protected $table = 'project_discussions';

    protected $fillable = [
        'project_id',
        'title',
        'message',
        'portal_visible',
        'created_by',
    ];

    protected $casts = [
        'portal_visible' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    // ==================== RELATIONSHIPS ====================

    public function project()
    {
        return $this->belongsTo(Project::class, 'project_id');
    }

    public function creator()
    {
        return $this->belongsTo('App\Models\Admin', 'created_by');
    }

    public function replies()
    {
        return $this->hasMany(DiscussionReply::class, 'discussion_id')->orderBy('created_at', 'asc');
    }

    /**
     * Get all attachments for this discussion
     */
    public function attachments()
    {
        return $this->hasMany(DiscussionAttachment::class, 'discussion_id');
    }

    // ==================== SCOPES ====================

    public function scopePortalVisible(Builder $query): Builder
    {
        return $query->where('portal_visible', true);
    }

    public function scopeByProject(Builder $query, $projectId): Builder
    {
        return $query->where('project_id', $projectId);
    }

    // ==================== ACCESSORS ====================

    public function getRepliesCountAttribute(): int
    {
        return $this->replies()->count();
    }

    public function getAttachmentsCountAttribute(): int
    {
        return $this->attachments()->count();
    }

    public function getHasAttachmentsAttribute(): bool
    {
        return $this->attachments()->exists();
    }
}