<?php

namespace Modules\ProjectManagement\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;

class DiscussionAttachment extends Model
{
    protected $table = 'discussion_attachments';

    protected $fillable = [
        'discussion_id',
        'file_name',
        'file_path',
        'file_type',
        'file_size',
        'uploaded_by',
    ];

    protected $casts = [
        'file_size' => 'integer',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    // ==================== RELATIONSHIPS ====================

    public function discussion()
    {
        return $this->belongsTo(Discussion::class, 'discussion_id');
    }

    public function uploader()
    {
        return $this->belongsTo('App\Models\Admin', 'uploaded_by');
    }

    // ==================== ACCESSORS ====================

    /**
     * Get formatted file size
     */
    public function getFormattedSizeAttribute(): string
    {
        $bytes = $this->file_size;
        
        if ($bytes >= 1073741824) {
            return number_format($bytes / 1073741824, 2) . ' GB';
        } elseif ($bytes >= 1048576) {
            return number_format($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            return number_format($bytes / 1024, 2) . ' KB';
        } else {
            return $bytes . ' bytes';
        }
    }

    /**
     * Get file extension
     */
    public function getExtensionAttribute(): string
    {
        return strtolower(pathinfo($this->file_name, PATHINFO_EXTENSION));
    }

    /**
     * Get file icon based on type
     */
    public function getIconAttribute(): string
    {
        $ext = $this->extension;
        
        $icons = [
            // Images
            'jpg' => 'image', 'jpeg' => 'image', 'png' => 'image', 
            'gif' => 'image', 'webp' => 'image', 'svg' => 'image',
            // Documents
            'pdf' => 'file-text', 'doc' => 'file-text', 'docx' => 'file-text',
            'xls' => 'file-spreadsheet', 'xlsx' => 'file-spreadsheet', 'csv' => 'file-spreadsheet',
            'ppt' => 'file-presentation', 'pptx' => 'file-presentation',
            'txt' => 'file-text', 'rtf' => 'file-text',
            // Archives
            'zip' => 'archive', 'rar' => 'archive', '7z' => 'archive', 'tar' => 'archive', 'gz' => 'archive',
            // Code
            'html' => 'code', 'css' => 'code', 'js' => 'code', 'php' => 'code', 'json' => 'code',
        ];

        return $icons[$ext] ?? 'file';
    }

    /**
     * Check if file is an image
     */
    public function getIsImageAttribute(): bool
    {
        return in_array($this->extension, ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg']);
    }

    /**
     * Get download URL
     */
    public function getDownloadUrlAttribute(): string
    {
        return route('admin.discussions.attachment.download', $this->id);
    }

    /**
     * Get preview URL for images
     */
    public function getPreviewUrlAttribute(): ?string
    {
        if (!$this->is_image) {
            return null;
        }
        return route('admin.discussions.attachment.preview', $this->id);
    }

    // ==================== METHODS ====================

    /**
     * Delete file from storage
     */
    public function deleteFile(): bool
    {
        if ($this->file_path && Storage::disk('public')->exists($this->file_path)) {
            return Storage::disk('public')->delete($this->file_path);
        }
        return true;
    }

    /**
     * Boot method for model events
     */
    protected static function boot()
    {
        parent::boot();

        // Delete file when model is deleted
        static::deleting(function ($attachment) {
            $attachment->deleteFile();
        });
    }
}
