<?php

namespace Modules\ProjectManagement\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;

class Milestone extends Model
{
    protected $table = 'project_milestones';

    protected $fillable = [
        'project_id',
        'name',
        'description',
        'target_date',
        'completed_at',
        'is_completed',
        'portal_visible',
        'sort_order',
        'created_by',
    ];

    protected $casts = [
        'target_date' => 'date',
        'completed_at' => 'datetime',
        'is_completed' => 'boolean',
        'portal_visible' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    // ==================== RELATIONSHIPS ====================

    public function project()
    {
        return $this->belongsTo(Project::class, 'project_id');
    }

    public function creator()
    {
        return $this->belongsTo('App\Models\Admin', 'created_by');
    }

    public function tasks()
    {
        return $this->hasMany(Task::class, 'milestone_id');
    }

    // ==================== SCOPES ====================

    public function scopeCompleted(Builder $query): Builder
    {
        return $query->where('is_completed', true);
    }

    public function scopeIncomplete(Builder $query): Builder
    {
        return $query->where('is_completed', false);
    }

    public function scopePortalVisible(Builder $query): Builder
    {
        return $query->where('portal_visible', true);
    }

    public function scopeByProject(Builder $query, $projectId): Builder
    {
        return $query->where('project_id', $projectId);
    }

    // ==================== ACCESSORS ====================

    public function getIsOverdueAttribute(): bool
    {
        if (!$this->target_date || $this->is_completed) {
            return false;
        }
        return $this->target_date->isPast();
    }

    public function getCompletionPercentageAttribute(): float
    {
        $total = $this->tasks()->count();
        if ($total == 0) {
            return 0;
        }
        $completed = $this->tasks()->whereHas('status', function ($q) {
            $q->where('is_completed', true);
        })->count();
        return round(($completed / $total) * 100, 2);
    }
}
