<?php

namespace Modules\ProjectManagement\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;

class Project extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'title',
        'account_id',
        'billing_method',
        'fixed_amount',
        'hourly_rate',
        'currency_id',
        'status_id',
        'planned_start_date',
        'planned_end_date',
        'actual_start_date',
        'actual_end_date',
        'scope_notes',
        'internal_notes',
        'is_pinned',
        'is_billable',
        // Portal visibility flags
        'portal_show_tasks',
        'portal_allow_task_comments',
        'portal_show_task_conversations',
        'portal_show_task_files',
        'portal_show_checklist',
        'portal_allow_task_uploads',
        'portal_show_time_logs',
        'portal_show_finance',
        'portal_allow_file_uploads',
        'portal_show_discussions',
        'portal_show_milestones',
        'portal_show_gantt',
        'portal_show_activity',
        'portal_show_team',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'fixed_amount' => 'decimal:2',
        'hourly_rate' => 'decimal:2',
        'planned_start_date' => 'date',
        'planned_end_date' => 'date',
        'actual_start_date' => 'date',
        'actual_end_date' => 'date',
        'is_pinned' => 'boolean',
        'is_billable' => 'boolean',
        'portal_show_tasks' => 'boolean',
        'portal_allow_task_comments' => 'boolean',
        'portal_show_task_conversations' => 'boolean',
        'portal_show_task_files' => 'boolean',
        'portal_show_checklist' => 'boolean',
        'portal_allow_task_uploads' => 'boolean',
        'portal_show_time_logs' => 'boolean',
        'portal_show_finance' => 'boolean',
        'portal_allow_file_uploads' => 'boolean',
        'portal_show_discussions' => 'boolean',
        'portal_show_milestones' => 'boolean',
        'portal_show_gantt' => 'boolean',
        'portal_show_activity' => 'boolean',
        'portal_show_team' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime',
    ];

    // ==================== RELATIONSHIPS ====================

public function account()
{
    // Only define relationship if Customer model exists
    if (class_exists('App\Models\Customer')) {
        return $this->belongsTo('App\Models\Customer', 'account_id');
    }
    // Return empty relation if model doesn't exist
    return $this->belongsTo(Project::class, 'account_id')->whereRaw('1 = 0');
}

public function currency()
{
    // Only define relationship if Currency model exists
    if (class_exists('App\Models\Currency')) {
        return $this->belongsTo('App\Models\Currency', 'currency_id');
    }
    // Return empty relation if model doesn't exist
    return $this->belongsTo(Project::class, 'currency_id')->whereRaw('1 = 0');
}

public function status()
{
    return $this->belongsTo(ProjectStatus::class, 'status_id');
}

public function creator()
{
    return $this->belongsTo('App\Models\Admin', 'created_by');
}

public function updater()
{
    return $this->belongsTo('App\Models\Admin', 'updated_by');
}

    public function tasks()
    {
        return $this->hasMany(Task::class, 'project_id');
    }

    public function milestones()
    {
        return $this->hasMany(Milestone::class, 'project_id')->orderBy('sort_order');
    }

    public function timeLogs()
    {
        return $this->hasManyThrough(TimeLog::class, Task::class, 'project_id', 'task_id');
    }

    public function discussions()
    {
        return $this->hasMany(Discussion::class, 'project_id')->orderBy('created_at', 'desc');
    }

    public function team()
{
    return $this->belongsToMany('App\Models\Admin', 'project_team', 'project_id', 'admin_id')
        ->withTimestamps();
}

    // ==================== SCOPES ====================

    public function scopeActive(Builder $query): Builder
    {
        return $query->whereHas('status', function ($q) {
            $q->where('is_active', true);
        });
    }

    public function scopePinned(Builder $query): Builder
    {
        return $query->where('is_pinned', true);
    }

    public function scopeBillable(Builder $query): Builder
    {
        return $query->where('is_billable', true);
    }

    public function scopeSearch(Builder $query, ?string $search): Builder
    {
        if (empty($search)) {
            return $query;
        }

        return $query->where(function ($q) use ($search) {
            $q->where('title', 'LIKE', "%{$search}%")
                ->orWhere('scope_notes', 'LIKE', "%{$search}%")
                ->orWhere('internal_notes', 'LIKE', "%{$search}%");
        });
    }

    public function scopeByAccount(Builder $query, $accountId): Builder
    {
        return $query->where('account_id', $accountId);
    }

    public function scopeByStatus(Builder $query, $statusId): Builder
    {
        return $query->where('status_id', $statusId);
    }

    public function scopeByTeamMember(Builder $query, $adminId): Builder
    {
        return $query->whereHas('team', function ($q) use ($adminId) {
            $q->where('admin_id', $adminId);
        });
    }

    // ==================== ACCESSORS ====================

    public function getStatusLabelAttribute(): string
    {
        return $this->status ? $this->status->label : 'Unknown';
    }

    public function getStatusColorAttribute(): string
    {
        return $this->status ? $this->status->color : '#9ca3af';
    }

    public function getBillingMethodLabelAttribute(): string
    {
        $methods = config('projectmanagement.billing_methods');
        return $methods[$this->billing_method] ?? ucfirst($this->billing_method);
    }

    public function getTotalTimeLoggedAttribute(): float
    {
        return $this->timeLogs()->sum('duration_minutes') / 60;
    }

    public function getTotalBillableHoursAttribute(): float
{
    return $this->timeLogs()->where('project_time_logs.is_billable', true)->sum('duration_minutes') / 60;
}

    public function getTotalBilledHoursAttribute(): float
{
    return $this->timeLogs()->where('project_time_logs.is_billed', true)->sum('duration_minutes') / 60;
}

    public function getCompletedTasksCountAttribute(): int
    {
        return $this->tasks()->whereHas('status', function ($q) {
            $q->where('is_completed', true);
        })->count();
    }

    public function getTotalTasksCountAttribute(): int
    {
        return $this->tasks()->count();
    }

    public function getProgressPercentageAttribute(): float
    {
        $total = $this->total_tasks_count;
        if ($total == 0) {
            return 0;
        }
        return round(($this->completed_tasks_count / $total) * 100, 2);
    }
}
