<?php

namespace Modules\ProjectManagement\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;

class Task extends Model
{
    use SoftDeletes;

    protected $table = 'project_tasks';

    protected $fillable = [
        'project_id',
        'milestone_id',
        'title',
        'description',
        'status_id',
        'priority',
        'planned_start_date',
        'due_date',
        'completed_at',
        'estimated_hours',
        'hourly_rate',
        'is_billable',
        'is_recurring',
        'recurring_frequency',
        'recurring_until',
        'parent_task_id',
        'sort_order',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'planned_start_date' => 'date',
        'due_date' => 'date',
        'completed_at' => 'datetime',
        'estimated_hours' => 'decimal:2',
        'hourly_rate' => 'decimal:2',
        'is_billable' => 'boolean',
        'is_recurring' => 'boolean',
        'recurring_until' => 'date',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime',
    ];

    // ==================== RELATIONSHIPS ====================

    public function project()
    {
        return $this->belongsTo(Project::class, 'project_id');
    }

    public function milestone()
    {
        return $this->belongsTo(Milestone::class, 'milestone_id');
    }

    public function status()
    {
        return $this->belongsTo(TaskStatus::class, 'status_id');
    }

    public function creator()
    {
        return $this->belongsTo('App\Models\Admin', 'created_by');
    }

    public function updater()
    {
        return $this->belongsTo('App\Models\Admin', 'updated_by');
    }

    public function owners()
    {
        return $this->belongsToMany('App\Models\Admin', 'task_owners', 'task_id', 'admin_id')
            ->withTimestamps();
    }

    public function followers()
    {
        return $this->belongsToMany('App\Models\Admin', 'task_followers', 'task_id', 'admin_id')
            ->withTimestamps();
    }

    public function timeLogs()
    {
        return $this->hasMany(TimeLog::class, 'task_id');
    }

    public function comments()
    {
        return $this->hasMany(TaskComment::class, 'task_id')->orderBy('created_at', 'asc');
    }

    public function attachments()
    {
        return $this->hasMany(TaskAttachment::class, 'task_id');
    }

    public function items()
    {
        return $this->hasMany(TaskItem::class, 'task_id');
    }

    public function checklists()
    {
        return $this->hasMany(TaskChecklist::class, 'task_id')->orderBy('sort_order');
    }

    public function parentTask()
    {
        return $this->belongsTo(Task::class, 'parent_task_id');
    }

    public function subtasks()
    {
        return $this->hasMany(Task::class, 'parent_task_id');
    }

    // ==================== SCOPES ====================

    public function scopeActive(Builder $query): Builder
    {
        return $query->whereHas('status', function ($q) {
            $q->where('show_by_default', true);
        });
    }

    public function scopeCompleted(Builder $query): Builder
    {
        return $query->whereHas('status', function ($q) {
            $q->where('is_completed', true);
        });
    }

    public function scopeIncomplete(Builder $query): Builder
    {
        return $query->whereHas('status', function ($q) {
            $q->where('is_completed', false);
        });
    }

    public function scopeBillable(Builder $query): Builder
    {
        return $query->where('is_billable', true);
    }

    public function scopeOverdue(Builder $query): Builder
    {
        return $query->where('due_date', '<', now())
            ->whereHas('status', function ($q) {
                $q->where('is_completed', false);
            });
    }

    public function scopeSearch(Builder $query, ?string $search): Builder
    {
        if (empty($search)) {
            return $query;
        }

        return $query->where(function ($q) use ($search) {
            $q->where('title', 'LIKE', "%{$search}%")
                ->orWhere('description', 'LIKE', "%{$search}%");
        });
    }

    public function scopeByProject(Builder $query, $projectId): Builder
    {
        return $query->where('project_id', $projectId);
    }

    public function scopeByMilestone(Builder $query, $milestoneId): Builder
    {
        return $query->where('milestone_id', $milestoneId);
    }

    public function scopeByStatus(Builder $query, $statusId): Builder
    {
        return $query->where('status_id', $statusId);
    }

    public function scopeByPriority(Builder $query, $priority): Builder
    {
        return $query->where('priority', $priority);
    }

    public function scopeByOwner(Builder $query, $adminId): Builder
    {
        return $query->whereHas('owners', function ($q) use ($adminId) {
            $q->where('admin_id', $adminId);
        });
    }

    public function scopeByFollower(Builder $query, $adminId): Builder
    {
        return $query->whereHas('followers', function ($q) use ($adminId) {
            $q->where('admin_id', $adminId);
        });
    }

    // ==================== ACCESSORS ====================

    public function getStatusLabelAttribute(): string
    {
        return $this->status ? $this->status->label : 'Unknown';
    }

    public function getStatusColorAttribute(): string
    {
        return $this->status ? $this->status->color : '#9ca3af';
    }

    public function getPriorityLabelAttribute(): string
    {
        $priorities = config('projectmanagement.priorities');
        return $priorities[$this->priority] ?? ucfirst($this->priority);
    }

    public function getPriorityColorAttribute(): string
    {
        $colors = [
            'low' => '#10b981',
            'medium' => '#3b82f6',
            'high' => '#f59e0b',
            'critical' => '#ef4444',
        ];
        return $colors[$this->priority] ?? '#9ca3af';
    }

    public function getIsOverdueAttribute(): bool
    {
        if (!$this->due_date) {
            return false;
        }
        return $this->due_date->isPast() && !$this->completed_at;
    }

    public function getIsCompletedAttribute(): bool
    {
        return $this->status && $this->status->is_completed;
    }

    public function getTotalTimeLoggedAttribute(): float
    {
        return $this->timeLogs()->sum('duration_minutes') / 60;
    }

    public function getTotalBillableTimeAttribute(): float
    {
        return $this->timeLogs()->where('is_billable', true)->sum('duration_minutes') / 60;
    }

    /**
     * Check if this task has an active timer running
     */
    public function getHasActiveTimerAttribute(): bool
    {
        try {
            $adminId = null;
            if (auth()->guard('admin')->check()) {
                $adminId = auth()->guard('admin')->id();
            } elseif (auth()->check()) {
                $adminId = auth()->id();
            }
            
            if (!$adminId) {
                return false;
            }
            
            return $this->timeLogs()
                ->where('admin_id', $adminId)
                ->whereNull('end_time')
                ->exists();
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Check if this task's timer has been completed (finished)
     */
    public function getHasCompletedTimerAttribute(): bool
    {
        try {
            $adminId = null;
            if (auth()->guard('admin')->check()) {
                $adminId = auth()->guard('admin')->id();
            } elseif (auth()->check()) {
                $adminId = auth()->id();
            }
            
            if (!$adminId) {
                return false;
            }
            
            return $this->timeLogs()
                ->where('admin_id', $adminId)
                ->whereNotNull('end_time')
                ->exists();
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Get the active timer for this task
     */
    public function getActiveTimerAttribute()
    {
        try {
            $adminId = null;
            if (auth()->guard('admin')->check()) {
                $adminId = auth()->guard('admin')->id();
            } elseif (auth()->check()) {
                $adminId = auth()->id();
            }
            
            if (!$adminId) {
                return null;
            }
            
            return $this->timeLogs()
                ->where('admin_id', $adminId)
                ->whereNull('end_time')
                ->first();
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Get the completed timer for this task
     */
    public function getCompletedTimerAttribute()
    {
        try {
            $adminId = null;
            if (auth()->guard('admin')->check()) {
                $adminId = auth()->guard('admin')->id();
            } elseif (auth()->check()) {
                $adminId = auth()->id();
            }
            
            if (!$adminId) {
                return null;
            }
            
            return $this->timeLogs()
                ->where('admin_id', $adminId)
                ->whereNotNull('end_time')
                ->first();
        } catch (\Exception $e) {
            return null;
        }
    }
}