<?php

namespace Modules\ProjectManagement\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class TaskItem extends Model
{
    protected $table = 'task_items';

    protected $fillable = [
        'task_id',
        'product_id',
        'quantity',
        'unit_price',
        'total_price',
        'tax_ids',
        'tax_amount',
        'notes',
    ];

    protected $casts = [
        'quantity' => 'decimal:2',
        'unit_price' => 'decimal:2',
        'total_price' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'tax_ids' => 'array',
    ];

    // ==================== RELATIONSHIPS ====================

    /**
     * Get the task that owns this item
     */
    public function task(): BelongsTo
    {
        return $this->belongsTo(Task::class, 'task_id');
    }

    /**
     * Get the product for this item
     */
    public function product(): BelongsTo
    {
        // Product model from main app
        return $this->belongsTo(\App\Models\Product::class, 'product_id');
    }

    // ==================== ACCESSORS ====================

    /**
     * Get formatted quantity
     */
    public function getFormattedQuantityAttribute(): string
    {
        return rtrim(rtrim(number_format($this->quantity, 2), '0'), '.');
    }

    /**
     * Get formatted unit price
     */
    public function getFormattedUnitPriceAttribute(): string
    {
        return $this->unit_price ? number_format($this->unit_price, 2) : '0.00';
    }

    /**
     * Get formatted total price
     */
    public function getFormattedTotalPriceAttribute(): string
    {
        return $this->total_price ? number_format($this->total_price, 2) : '0.00';
    }

    /**
     * Get formatted tax amount
     */
    public function getFormattedTaxAmountAttribute(): string
    {
        return $this->tax_amount ? number_format($this->tax_amount, 2) : '0.00';
    }

    // ==================== MUTATORS ====================

    /**
     * Auto-calculate total price when quantity or unit_price changes
     */
    protected static function boot()
    {
        parent::boot();

        static::saving(function ($item) {
            if ($item->quantity && $item->unit_price) {
                $item->total_price = $item->quantity * $item->unit_price;
            }
        });
    }
}