<?php

namespace Modules\ProjectManagement\Models;

use Illuminate\Database\Eloquent\Model;

class TimeLogPause extends Model
{
    protected $table = 'project_time_log_pauses';

    protected $fillable = [
        'time_log_id',
        // Pause event
        'paused_at',
        'pause_latitude',
        'pause_longitude',
        'pause_accuracy',
        'pause_ip_address',
        // Resume event
        'resumed_at',
        'resume_latitude',
        'resume_longitude',
        'resume_accuracy',
        'resume_ip_address',
        // Duration & reason
        'pause_duration_seconds',
        'pause_reason',
    ];

    protected $casts = [
        'paused_at' => 'datetime',
        'resumed_at' => 'datetime',
        'pause_latitude' => 'decimal:8',
        'pause_longitude' => 'decimal:8',
        'pause_accuracy' => 'decimal:2',
        'resume_latitude' => 'decimal:8',
        'resume_longitude' => 'decimal:8',
        'resume_accuracy' => 'decimal:2',
        'pause_duration_seconds' => 'integer',
    ];

    // ==================== RELATIONSHIPS ====================

    public function timeLog()
    {
        return $this->belongsTo(TimeLog::class, 'time_log_id');
    }

    // ==================== ACCESSORS ====================

    /**
     * Check if this pause is still active (not resumed)
     */
    public function getIsActiveAttribute(): bool
    {
        return is_null($this->resumed_at);
    }

    /**
     * Get pause duration in minutes
     */
    public function getPauseDurationMinutesAttribute(): float
    {
        return round($this->pause_duration_seconds / 60, 2);
    }

    /**
     * Get formatted pause duration (HH:MM:SS)
     */
    public function getFormattedPauseDurationAttribute(): string
    {
        $seconds = $this->pause_duration_seconds;
        $hours = floor($seconds / 3600);
        $minutes = floor(($seconds % 3600) / 60);
        $secs = $seconds % 60;
        return sprintf('%02d:%02d:%02d', $hours, $minutes, $secs);
    }

    /**
     * Google Maps URL for pause location
     */
    public function getPauseGoogleMapsUrlAttribute(): string
    {
        if (!$this->pause_latitude || !$this->pause_longitude) {
            return '#';
        }
        return "https://www.google.com/maps?q={$this->pause_latitude},{$this->pause_longitude}";
    }

    /**
     * Google Maps URL for resume location
     */
    public function getResumeGoogleMapsUrlAttribute(): string
    {
        if (!$this->resume_latitude || !$this->resume_longitude) {
            return '#';
        }
        return "https://www.google.com/maps?q={$this->resume_latitude},{$this->resume_longitude}";
    }

    // ==================== METHODS ====================

    /**
     * Resume this pause and calculate duration
     */
    public function resume(array $location = []): void
    {
        $this->resumed_at = now();
        
        if (!empty($location)) {
            $this->resume_latitude = $location['latitude'] ?? null;
            $this->resume_longitude = $location['longitude'] ?? null;
            $this->resume_accuracy = $location['accuracy'] ?? null;
            $this->resume_ip_address = $location['ip_address'] ?? null;
        }
        
        // Calculate pause duration
        $this->pause_duration_seconds = $this->paused_at->diffInSeconds($this->resumed_at);
        
        $this->save();
    }
}
