<style>
/* ========================================
   Project Management Dashboard - Theme-Aware
   ======================================== */

/* Page Animation */
@keyframes fadeIn {
    from { opacity: 0; transform: translateY(10px); }
    to { opacity: 1; transform: translateY(0); }
}

@keyframes slideIn {
    from { opacity: 0; transform: translateX(-20px); }
    to { opacity: 1; transform: translateX(0); }
}

@keyframes pulse {
    0%, 100% { transform: scale(1); opacity: 1; }
    50% { transform: scale(1.05); opacity: 0.9; }
}

@keyframes spin {
    from { transform: rotate(0deg); }
    to { transform: rotate(360deg); }
}

.dashboard-container {
    padding: var(--space-xl);
    animation: fadeIn 0.4s ease;
}

/* ===== PAGE HEADER ===== */
.dashboard-header {
    background: linear-gradient(135deg, var(--primary) 0%, #764ba2 100%);
    padding: var(--space-lg) var(--space-xl);
    border-radius: var(--radius-lg);
    margin-bottom: var(--space-lg);
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    animation: slideIn 0.3s ease;
}

.header-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: var(--space-lg);
}

.header-left h1 {
    color: white;
    font-size: var(--font-xl);
    font-weight: 700;
    margin: 0 0 var(--space-xs) 0;
    display: flex;
    align-items: center;
    gap: var(--space-sm);
}

.header-left h1 svg {
    width: 20px;
    height: 20px;
    animation: pulse 2s ease-in-out infinite;
}

.header-left p {
    color: rgba(255, 255, 255, 0.9);
    font-size: var(--font-sm);
    margin: 0;
}

.header-actions {
    display: flex;
    gap: var(--space-sm);
}

.btn {
    padding: var(--space-sm) var(--space-md);
    border-radius: var(--radius-md);
    font-weight: 600;
    font-size: var(--font-sm);
    border: none;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    gap: var(--space-xs);
    text-decoration: none;
}

.btn svg {
    width: 14px;
    height: 14px;
}

.btn-white {
    background: white;
    color: var(--primary);
    box-shadow: 0 4px 15px rgba(255, 255, 255, 0.3);
}

.btn-white:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(255, 255, 255, 0.4);
}

/* ===== OVERVIEW STATS GRID ===== */
.overview-stats {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(240px, 1fr));
    gap: var(--space-md);
    margin-bottom: var(--space-lg);
}

.stat-card {
    background: var(--card-bg);
    border: 1px solid var(--card-border);
    border-radius: var(--radius-lg);
    padding: var(--space-lg);
    display: flex;
    align-items: center;
    gap: var(--space-md);
    transition: all 0.3s ease;
    animation: slideIn 0.3s ease;
    position: relative;
    overflow: hidden;
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 4px;
    height: 100%;
    background: var(--primary);
    transform: scaleY(0);
    transform-origin: top;
    transition: transform 0.3s ease;
}

.stat-card:hover::before {
    transform: scaleY(1);
}

.stat-card:hover {
    transform: translateY(-4px) scale(1.01);
    box-shadow: 0 8px 30px rgba(0, 0, 0, 0.12);
}

.stat-card:nth-child(1) { animation-delay: 0.05s; }
.stat-card:nth-child(2) { animation-delay: 0.1s; }
.stat-card:nth-child(3) { animation-delay: 0.15s; }
.stat-card:nth-child(4) { animation-delay: 0.2s; }
.stat-card:nth-child(5) { animation-delay: 0.25s; }
.stat-card:nth-child(6) { animation-delay: 0.3s; }

.stat-icon {
    width: 48px;
    height: 48px;
    border-radius: var(--radius-md);
    display: flex;
    align-items: center;
    justify-content: center;
    flex-shrink: 0;
    transition: transform 0.3s ease;
}

.stat-card:hover .stat-icon {
    animation: pulse 1s ease infinite;
}

.stat-icon svg {
    width: 20px;
    height: 20px;
    color: white;
}

.stat-icon-primary { background: var(--primary); }
.stat-icon-success { background: var(--success); }
.stat-icon-warning { background: var(--warning); }
.stat-icon-danger { background: var(--danger); }
.stat-icon-info { background: #3b82f6; }
.stat-icon-purple { background: #8b5cf6; }

.stat-content h3 {
    font-size: 28px;
    font-weight: 700;
    color: var(--text-primary);
    margin: 0 0 var(--space-xs) 0;
}

.stat-content p {
    font-size: var(--font-xs);
    color: var(--text-muted);
    margin: 0;
    text-transform: uppercase;
    letter-spacing: 0.05em;
    font-weight: 600;
}

.stat-content small {
    font-size: 10px;
    color: var(--text-muted);
    display: block;
    margin-top: 2px;
}

/* ===== CHARTS GRID ===== */
.charts-grid {
    display: grid;
    grid-template-columns: repeat(2, 1fr);
    gap: var(--space-lg);
    margin-bottom: var(--space-lg);
}

.chart-card {
    background: var(--card-bg);
    border: 1px solid var(--card-border);
    border-radius: var(--radius-lg);
    padding: var(--space-xl);
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
    animation: fadeIn 0.4s ease;
}

.chart-card:nth-child(1) { animation-delay: 0.35s; }
.chart-card:nth-child(2) { animation-delay: 0.4s; }
.chart-card:nth-child(3) { animation-delay: 0.45s; }
.chart-card:nth-child(4) { animation-delay: 0.5s; }

.chart-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: var(--space-lg);
}

.chart-title {
    font-size: var(--font-lg);
    font-weight: 700;
    color: var(--text-primary);
    display: flex;
    align-items: center;
    gap: var(--space-sm);
}

.chart-title svg {
    width: 18px;
    height: 18px;
    color: var(--primary);
}

.chart-canvas {
    position: relative;
    height: 280px;
}

.chart-canvas canvas {
    max-height: 100%;
}

/* ===== RECENT ACTIVITY SECTIONS ===== */
.activity-grid {
    display: grid;
    grid-template-columns: repeat(3, 1fr);
    gap: var(--space-lg);
    margin-bottom: var(--space-lg);
}

.activity-card {
    background: var(--card-bg);
    border: 1px solid var(--card-border);
    border-radius: var(--radius-lg);
    padding: var(--space-xl);
    animation: fadeIn 0.4s ease;
}

.activity-card:nth-child(1) { animation-delay: 0.55s; }
.activity-card:nth-child(2) { animation-delay: 0.6s; }
.activity-card:nth-child(3) { animation-delay: 0.65s; }

.activity-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: var(--space-lg);
    padding-bottom: var(--space-md);
    border-bottom: 2px solid var(--card-border);
}

.activity-title {
    font-size: var(--font-base);
    font-weight: 700;
    color: var(--text-primary);
    display: flex;
    align-items: center;
    gap: var(--space-sm);
}

.activity-title svg {
    width: 16px;
    height: 16px;
    color: var(--primary);
}

.activity-list {
    display: flex;
    flex-direction: column;
    gap: var(--space-md);
}

.activity-item {
    padding: var(--space-md);
    background: var(--input-bg);
    border-radius: var(--radius-md);
    transition: all 0.3s ease;
    cursor: pointer;
}

.activity-item:hover {
    background: var(--card-border);
    transform: translateX(4px);
}

.activity-item-header {
    display: flex;
    justify-content: space-between;
    align-items: start;
    margin-bottom: var(--space-xs);
}

.activity-item-title {
    font-size: var(--font-sm);
    font-weight: 600;
    color: var(--text-primary);
    margin: 0;
}

.activity-item-meta {
    font-size: var(--font-xs);
    color: var(--text-muted);
}

.badge {
    padding: 3px 8px;
    border-radius: 12px;
    font-size: 10px;
    font-weight: 600;
    text-transform: uppercase;
}

.badge-success { background: var(--success-light); color: var(--success); }
.badge-warning { background: var(--warning-light); color: var(--warning); }
.badge-danger { background: var(--danger-light); color: var(--danger); }
.badge-info { background: #dbeafe; color: #1e40af; }

/* Empty State */
.empty-state {
    text-align: center;
    padding: var(--space-xl);
    color: var(--text-muted);
}

.empty-state svg {
    width: 48px;
    height: 48px;
    margin-bottom: var(--space-md);
    opacity: 0.5;
}

/* Loading */
.chart-loading {
    display: flex;
    align-items: center;
    justify-content: center;
    height: 280px;
}

.spinner {
    width: 40px;
    height: 40px;
    border: 4px solid var(--card-border);
    border-top: 4px solid var(--primary);
    border-radius: 50%;
    animation: spin 1s linear infinite;
}

/* Responsive */
@media (max-width: 1200px) {
    .charts-grid {
        grid-template-columns: 1fr;
    }
    
    .activity-grid {
        grid-template-columns: 1fr;
    }
}

@media (max-width: 768px) {
    .dashboard-container {
        padding: var(--space-md);
    }
    
    .overview-stats {
        grid-template-columns: 1fr;
    }
    
    .header-content {
        flex-direction: column;
        align-items: stretch;
    }
}
</style>

<div class="dashboard-container">
    <!-- Dashboard Header -->
    <div class="dashboard-header">
        <div class="header-content">
            <div class="header-left">
                <h1>
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z"/>
                    </svg>
                    Project Management Dashboard
                </h1>
                <p>Overview of all projects, tasks, and time tracking</p>
            </div>
            
            <div class="header-actions">
                <button class="btn btn-white" onclick="refreshDashboard()">
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"/>
                    </svg>
                    Refresh
                </button>
            </div>
        </div>
    </div>

    <!-- Overview Stats -->
    <div class="overview-stats">
        <!-- Projects -->
        <div class="stat-card">
            <div class="stat-icon stat-icon-primary">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M3 7v10a2 2 0 002 2h14a2 2 0 002-2V9a2 2 0 00-2-2h-6l-2-2H5a2 2 0 00-2 2z"/>
                </svg>
            </div>
            <div class="stat-content">
                <h3 id="statProjectsTotal">{{ $stats['projects']['total'] }}</h3>
                <p>Total Projects</p>
                <small>{{ $stats['projects']['active'] }} active</small>
            </div>
        </div>

        <!-- My Projects -->
        <div class="stat-card">
            <div class="stat-icon stat-icon-success">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"/>
                </svg>
            </div>
            <div class="stat-content">
                <h3 id="statMyProjects">{{ $stats['projects']['my_projects'] }}</h3>
                <p>My Projects</p>
                <small>{{ $stats['projects']['pinned'] }} pinned</small>
            </div>
        </div>

        <!-- Total Tasks -->
        <div class="stat-card">
            <div class="stat-icon stat-icon-info">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2m-6 9l2 2 4-4"/>
                </svg>
            </div>
            <div class="stat-content">
                <h3 id="statTasksTotal">{{ $stats['tasks']['total'] }}</h3>
                <p>Total Tasks</p>
                <small>{{ $stats['tasks']['completed'] }} completed</small>
            </div>
        </div>

        <!-- My Tasks -->
        <div class="stat-card">
            <div class="stat-icon stat-icon-warning">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                </svg>
            </div>
            <div class="stat-content">
                <h3 id="statMyTasks">{{ $stats['tasks']['my_tasks'] }}</h3>
                <p>My Tasks</p>
                <small>{{ $stats['tasks']['overdue'] }} overdue</small>
            </div>
        </div>

        <!-- Total Hours -->
        <div class="stat-card">
            <div class="stat-icon stat-icon-purple">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                </svg>
            </div>
            <div class="stat-content">
                <h3 id="statTotalHours">{{ $stats['time']['total_hours'] }}h</h3>
                <p>Total Hours</p>
                <small>{{ $stats['time']['billable_hours'] }}h billable</small>
            </div>
        </div>

        <!-- Milestones -->
        <div class="stat-card">
            <div class="stat-icon stat-icon-danger">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M5 3v4M3 5h4M6 17v4m-2-2h4m5-16l2.286 6.857L21 12l-5.714 2.143L13 21l-2.286-6.857L5 12l5.714-2.143L13 3z"/>
                </svg>
            </div>
            <div class="stat-content">
                <h3 id="statMilestonesTotal">{{ $stats['milestones']['total'] }}</h3>
                <p>Milestones</p>
                <small>{{ $stats['milestones']['completed'] }} completed</small>
            </div>
        </div>
    </div>

    <!-- Charts Grid - Row 1 -->
    <div class="charts-grid">
        <!-- Project Status Distribution -->
        <div class="chart-card">
            <div class="chart-header">
                <h3 class="chart-title">
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M11 3.055A9.001 9.001 0 1020.945 13H11V3.055z"/>
                        <path stroke-linecap="round" stroke-linejoin="round" d="M20.488 9H15V3.512A9.025 9.025 0 0120.488 9z"/>
                    </svg>
                    Project Status Distribution
                </h3>
            </div>
            <div class="chart-canvas">
                <div class="chart-loading" id="loadingProjectStatus">
                    <div class="spinner"></div>
                </div>
                <canvas id="chartProjectStatus" style="display: none;"></canvas>
            </div>
        </div>

        <!-- Task Priority Distribution -->
        <div class="chart-card">
            <div class="chart-header">
                <h3 class="chart-title">
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M7 12l3-3 3 3 4-4M8 21l4-4 4 4M3 4h18M4 4h16v12a1 1 0 01-1 1H5a1 1 0 01-1-1V4z"/>
                    </svg>
                    Task Priority Distribution
                </h3>
            </div>
            <div class="chart-canvas">
                <div class="chart-loading" id="loadingTaskPriority">
                    <div class="spinner"></div>
                </div>
                <canvas id="chartTaskPriority" style="display: none;"></canvas>
            </div>
        </div>
    </div>

    <!-- Charts Grid - Row 2 -->
    <div class="charts-grid">
        <!-- Task Status Distribution -->
        <div class="chart-card">
            <div class="chart-header">
                <h3 class="chart-title">
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z"/>
                    </svg>
                    Task Status Distribution
                </h3>
            </div>
            <div class="chart-canvas">
                <div class="chart-loading" id="loadingTaskStatus">
                    <div class="spinner"></div>
                </div>
                <canvas id="chartTaskStatus" style="display: none;"></canvas>
            </div>
        </div>

        <!-- Time Tracking - Billable vs Non-Billable -->
        <div class="chart-card">
            <div class="chart-header">
                <h3 class="chart-title">
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                    </svg>
                    Time Tracking Overview
                </h3>
            </div>
            <div class="chart-canvas">
                <div class="chart-loading" id="loadingTimeTracking">
                    <div class="spinner"></div>
                </div>
                <canvas id="chartTimeTracking" style="display: none;"></canvas>
            </div>
        </div>
    </div>

    <!-- Charts Grid - Row 3 (Full Width Charts) -->
    <div class="charts-grid">
        <!-- Project Progress - Top 10 -->
        <div class="chart-card">
            <div class="chart-header">
                <h3 class="chart-title">
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M13 7h8m0 0v8m0-8l-8 8-4-4-6 6"/>
                    </svg>
                    Project Progress - Top 10
                </h3>
            </div>
            <div class="chart-canvas">
                <div class="chart-loading" id="loadingProjectProgress">
                    <div class="spinner"></div>
                </div>
                <canvas id="chartProjectProgress" style="display: none;"></canvas>
            </div>
        </div>

        <!-- Monthly Time Tracking Trend -->
        <div class="chart-card">
            <div class="chart-header">
                <h3 class="chart-title">
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M7 12l3-3 3 3 4-4M8 21l4-4 4 4M3 4h18M4 4h16v12a1 1 0 01-1 1H5a1 1 0 01-1-1V4z"/>
                    </svg>
                    Monthly Time Trend (Last 6 Months)
                </h3>
            </div>
            <div class="chart-canvas">
                <div class="chart-loading" id="loadingMonthlyTime">
                    <div class="spinner"></div>
                </div>
                <canvas id="chartMonthlyTime" style="display: none;"></canvas>
            </div>
        </div>
    </div>

    <!-- Recent Activity Grid -->
    <div class="activity-grid">
        <!-- Recent Projects -->
        <div class="activity-card">
            <div class="activity-header">
                <h3 class="activity-title">
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M3 7v10a2 2 0 002 2h14a2 2 0 002-2V9a2 2 0 00-2-2h-6l-2-2H5a2 2 0 00-2 2z"/>
                    </svg>
                    Recent Projects
                </h3>
                <a href="{{ route('admin.projects.index') }}" style="font-size: 12px; color: var(--primary); text-decoration: none;">View All</a>
            </div>
            <div class="activity-list">
                @forelse($recentProjects as $project)
                <a href="{{ route('admin.projects.show', $project->id) }}" class="activity-item" style="text-decoration: none; color: inherit;">
                    <div class="activity-item-header">
                        <h4 class="activity-item-title">{{ Str::limit($project->title, 30) }}</h4>
                        <span class="badge badge-info">{{ $project->status->label ?? 'N/A' }}</span>
                    </div>
                    <p class="activity-item-meta">Created {{ $project->created_at->diffForHumans() }}</p>
                </a>
                @empty
                <div class="empty-state">
                    <svg fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M3 7v10a2 2 0 002 2h14a2 2 0 002-2V9a2 2 0 00-2-2h-6l-2-2H5a2 2 0 00-2 2z"/>
                    </svg>
                    <p>No recent projects</p>
                </div>
                @endforelse
            </div>
        </div>

        <!-- My Active Tasks -->
        <div class="activity-card">
            <div class="activity-header">
                <h3 class="activity-title">
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"/>
                    </svg>
                    My Active Tasks
                </h3>
                <a href="{{ route('admin.tasks.index') }}" style="font-size: 12px; color: var(--primary); text-decoration: none;">View All</a>
            </div>
            <div class="activity-list">
                @forelse($myActiveTasks as $task)
                <a href="{{ route('admin.tasks.show', $task->id) }}" class="activity-item" style="text-decoration: none; color: inherit;">
                    <div class="activity-item-header">
                        <h4 class="activity-item-title">{{ Str::limit($task->title, 30) }}</h4>
                        @if($task->is_overdue)
                        <span class="badge badge-danger">Overdue</span>
                        @else
                        <span class="badge badge-success">{{ $task->status->label ?? 'N/A' }}</span>
                        @endif
                    </div>
                    <p class="activity-item-meta">
                        {{ $task->project->title ?? 'No Project' }} • 
                        @if($task->due_date)
                            Due {{ $task->due_date->format('M d') }}
                        @else
                            No due date
                        @endif
                    </p>
                </a>
                @empty
                <div class="empty-state">
                    <svg fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"/>
                    </svg>
                    <p>No active tasks</p>
                </div>
                @endforelse
            </div>
        </div>

        <!-- Running Timers -->
        <div class="activity-card">
            <div class="activity-header">
                <h3 class="activity-title">
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                    </svg>
                    Running Timers
                </h3>
                <a href="{{ route('admin.timelogs.index') }}" style="font-size: 12px; color: var(--primary); text-decoration: none;">View All</a>
            </div>
            <div class="activity-list">
                @forelse($runningTimers as $timer)
                <div class="activity-item">
                    <div class="activity-item-header">
                        <h4 class="activity-item-title">{{ Str::limit($timer->task->title ?? 'Unknown Task', 30) }}</h4>
                        <span class="badge badge-warning">Running</span>
                    </div>
                    <p class="activity-item-meta">
                        {{ $timer->admin->name ?? 'Unknown' }} • 
                        Started {{ $timer->start_time->diffForHumans() }}
                    </p>
                </div>
                @empty
                <div class="empty-state">
                    <svg fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                    </svg>
                    <p>No running timers</p>
                </div>
                @endforelse
            </div>
        </div>
    </div>
</div>

<!-- Chart.js Library -->
<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>

<script>
// Chart instances storage
const chartInstances = {};

// Chart default configuration
Chart.defaults.font.family = '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif';
Chart.defaults.color = getComputedStyle(document.documentElement).getPropertyValue('--text-secondary').trim();

// Initialize dashboard
$(document).ready(function() {
    // Load all charts
    loadChart('project-status', 'chartProjectStatus', 'pie');
    loadChart('task-priority', 'chartTaskPriority', 'doughnut');
    loadChart('task-status', 'chartTaskStatus', 'doughnut');
    loadChart('time-tracking', 'chartTimeTracking', 'bar');
    loadChart('project-progress', 'chartProjectProgress', 'horizontalBar');
    loadChart('monthly-time', 'chartMonthlyTime', 'line');
});

/**
 * Load chart data via AJAX and render
 */
function loadChart(chartType, canvasId, chartFormat) {
    const loadingId = `loading${canvasId.charAt(5).toUpperCase() + canvasId.slice(6)}`;
    
    $.ajax({
        url: '{{ route("admin.project-management.chart-data") }}',
        type: 'GET',
        data: { chart: chartType },
        success: function(response) {
            // Hide loading, show canvas
            $(`#${loadingId}`).hide();
            $(`#${canvasId}`).show();
            
            // Render chart
            renderChart(canvasId, chartFormat, response);
        },
        error: function(xhr) {
            $(`#${loadingId}`).html(`
                <div style="text-align: center; color: var(--text-muted);">
                    <svg style="width: 48px; height: 48px; margin-bottom: 12px; opacity: 0.5;" fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M12 9v3.75m9-.75a9 9 0 11-18 0 9 9 0 0118 0zm-9 3.75h.008v.008H12v-.008z"/>
                    </svg>
                    <p>Failed to load chart</p>
                </div>
            `);
            console.error(`Error loading ${chartType}:`, xhr);
        }
    });
}

/**
 * Render Chart.js chart
 */
function renderChart(canvasId, type, data) {
    const ctx = document.getElementById(canvasId).getContext('2d');
    
    // Destroy existing chart if it exists
    if (chartInstances[canvasId]) {
        chartInstances[canvasId].destroy();
    }
    
    let chartConfig = {};
    
    // Configure based on chart type
    switch(type) {
        case 'pie':
            chartConfig = {
                type: 'pie',
                data: {
                    labels: data.labels,
                    datasets: [{
                        data: data.data,
                        backgroundColor: data.colors,
                        borderWidth: 2,
                        borderColor: getComputedStyle(document.documentElement).getPropertyValue('--card-bg').trim(),
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'bottom',
                            labels: {
                                padding: 15,
                                font: { size: 12, weight: '600' }
                            }
                        },
                        tooltip: {
                            backgroundColor: 'rgba(0, 0, 0, 0.8)',
                            padding: 12,
                            titleFont: { size: 13, weight: '600' },
                            bodyFont: { size: 12 },
                            cornerRadius: 8,
                        }
                    }
                }
            };
            break;
            
        case 'doughnut':
            chartConfig = {
                type: 'doughnut',
                data: {
                    labels: data.labels,
                    datasets: [{
                        data: data.data,
                        backgroundColor: data.colors,
                        borderWidth: 2,
                        borderColor: getComputedStyle(document.documentElement).getPropertyValue('--card-bg').trim(),
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    cutout: '65%',
                    plugins: {
                        legend: {
                            position: 'bottom',
                            labels: {
                                padding: 15,
                                font: { size: 12, weight: '600' }
                            }
                        },
                        tooltip: {
                            backgroundColor: 'rgba(0, 0, 0, 0.8)',
                            padding: 12,
                            titleFont: { size: 13, weight: '600' },
                            bodyFont: { size: 12 },
                            cornerRadius: 8,
                        }
                    }
                }
            };
            break;
            
        case 'bar':
            chartConfig = {
                type: 'bar',
                data: {
                    labels: data.labels,
                    datasets: [{
                        label: 'Hours',
                        data: data.data,
                        backgroundColor: data.colors,
                        borderRadius: 8,
                        borderWidth: 0,
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { display: false },
                        tooltip: {
                            backgroundColor: 'rgba(0, 0, 0, 0.8)',
                            padding: 12,
                            titleFont: { size: 13, weight: '600' },
                            bodyFont: { size: 12 },
                            cornerRadius: 8,
                            callbacks: {
                                label: function(context) {
                                    return context.parsed.y + ' hours';
                                }
                            }
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            grid: {
                                color: getComputedStyle(document.documentElement).getPropertyValue('--card-border').trim(),
                                drawBorder: false,
                            },
                            ticks: {
                                font: { size: 11 },
                                callback: function(value) {
                                    return value + 'h';
                                }
                            }
                        },
                        x: {
                            grid: { display: false },
                            ticks: { font: { size: 11, weight: '600' } }
                        }
                    }
                }
            };
            break;
            
        case 'horizontalBar':
            chartConfig = {
                type: 'bar',
                data: {
                    labels: data.labels,
                    datasets: [{
                        label: 'Progress (%)',
                        data: data.data,
                        backgroundColor: data.colors,
                        borderRadius: 6,
                        borderWidth: 0,
                    }]
                },
                options: {
                    indexAxis: 'y',
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { display: false },
                        tooltip: {
                            backgroundColor: 'rgba(0, 0, 0, 0.8)',
                            padding: 12,
                            titleFont: { size: 13, weight: '600' },
                            bodyFont: { size: 12 },
                            cornerRadius: 8,
                            callbacks: {
                                label: function(context) {
                                    return context.parsed.x + '% complete';
                                }
                            }
                        }
                    },
                    scales: {
                        x: {
                            beginAtZero: true,
                            max: 100,
                            grid: {
                                color: getComputedStyle(document.documentElement).getPropertyValue('--card-border').trim(),
                                drawBorder: false,
                            },
                            ticks: {
                                font: { size: 11 },
                                callback: function(value) {
                                    return value + '%';
                                }
                            }
                        },
                        y: {
                            grid: { display: false },
                            ticks: { 
                                font: { size: 10, weight: '600' },
                                callback: function(value, index) {
                                    const label = this.getLabelForValue(value);
                                    return label.length > 20 ? label.substring(0, 20) + '...' : label;
                                }
                            }
                        }
                    }
                }
            };
            break;
            
        case 'line':
            chartConfig = {
                type: 'line',
                data: {
                    labels: data.labels,
                    datasets: [{
                        label: 'Hours Logged',
                        data: data.data,
                        borderColor: data.colors[0],
                        backgroundColor: data.colors[0] + '20',
                        borderWidth: 3,
                        fill: true,
                        tension: 0.4,
                        pointRadius: 5,
                        pointHoverRadius: 7,
                        pointBackgroundColor: data.colors[0],
                        pointBorderColor: '#fff',
                        pointBorderWidth: 2,
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { display: false },
                        tooltip: {
                            backgroundColor: 'rgba(0, 0, 0, 0.8)',
                            padding: 12,
                            titleFont: { size: 13, weight: '600' },
                            bodyFont: { size: 12 },
                            cornerRadius: 8,
                            callbacks: {
                                label: function(context) {
                                    return context.parsed.y + ' hours';
                                }
                            }
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            grid: {
                                color: getComputedStyle(document.documentElement).getPropertyValue('--card-border').trim(),
                                drawBorder: false,
                            },
                            ticks: {
                                font: { size: 11 },
                                callback: function(value) {
                                    return value + 'h';
                                }
                            }
                        },
                        x: {
                            grid: { display: false },
                            ticks: { font: { size: 11, weight: '600' } }
                        }
                    }
                }
            };
            break;
    }
    
    // Create and store chart instance
    chartInstances[canvasId] = new Chart(ctx, chartConfig);
}

/**
 * Refresh dashboard stats and charts
 */
function refreshDashboard() {
    // Show loading state on button
    const btn = event.target.closest('.btn');
    const originalHTML = btn.innerHTML;
    btn.innerHTML = `
        <svg style="width: 14px; height: 14px; animation: spin 1s linear infinite;" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"/>
        </svg>
        Refreshing...
    `;
    btn.disabled = true;
    
    // Refresh stats
    $.ajax({
        url: '{{ route("admin.project-management.refresh-stats") }}',
        type: 'GET',
        success: function(stats) {
            // Update stat cards
            $('#statProjectsTotal').text(stats.projects.total);
            $('#statMyProjects').text(stats.projects.my_projects);
            $('#statTasksTotal').text(stats.tasks.total);
            $('#statMyTasks').text(stats.tasks.my_tasks);
            $('#statTotalHours').text(stats.time.total_hours + 'h');
            $('#statMilestonesTotal').text(stats.milestones.total);
            
            // Reload all charts
            loadChart('project-status', 'chartProjectStatus', 'pie');
            loadChart('task-priority', 'chartTaskPriority', 'doughnut');
            loadChart('task-status', 'chartTaskStatus', 'doughnut');
            loadChart('time-tracking', 'chartTimeTracking', 'bar');
            loadChart('project-progress', 'chartProjectProgress', 'horizontalBar');
            loadChart('monthly-time', 'chartMonthlyTime', 'line');
            
            // Success feedback
            showNotification('Dashboard refreshed successfully!', 'success');
        },
        error: function() {
            showNotification('Failed to refresh dashboard', 'error');
        },
        complete: function() {
            // Restore button
            setTimeout(() => {
                btn.innerHTML = originalHTML;
                btn.disabled = false;
            }, 1000);
        }
    });
}

/**
 * Show notification toast
 */
function showNotification(message, type) {
    const colors = {
        success: 'linear-gradient(135deg, #10b981 0%, #059669 100%)',
        error: 'linear-gradient(135deg, #ef4444 0%, #dc2626 100%)',
        info: 'linear-gradient(135deg, #3b82f6 0%, #2563eb 100%)',
    };
    
    const notification = $(`
        <div style="
            position: fixed;
            top: 20px;
            right: 20px;
            background: ${colors[type] || colors.info};
            color: white;
            padding: 16px 24px;
            border-radius: 10px;
            box-shadow: 0 8px 30px rgba(0, 0, 0, 0.3);
            z-index: 10000;
            animation: slideIn 0.3s ease;
            font-size: 14px;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 10px;
        ">
            <svg style="width: 18px; height: 18px;" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
            </svg>
            ${message}
        </div>
    `);
    
    $('body').append(notification);
    
    setTimeout(() => {
        notification.fadeOut(300, function() {
            $(this).remove();
        });
    }, 3000);
}

// Animation CSS for notifications
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from { opacity: 0; transform: translateX(50px); }
        to { opacity: 1; transform: translateX(0); }
    }
`;
document.head.appendChild(style);
</script>