<style>
/* ========================================
   Milestones - Theme-Aware (GUIDE COMPLIANT)
   ======================================== */

/* Page Animation */
.milestones-page {
    padding: var(--space-xl);
    animation: fadeIn 0.4s ease;
}

@keyframes fadeIn {
    from { opacity: 0; transform: translateY(10px); }
    to { opacity: 1; transform: translateY(0); }
}

/* Slide In Animation */
@keyframes slideIn {
    from { opacity: 0; transform: translateX(-20px); }
    to { opacity: 1; transform: translateX(0); }
}

/* Pulse Animation */
@keyframes pulse {
    0%, 100% { transform: scale(1); opacity: 1; }
    50% { transform: scale(1.05); opacity: 0.8; }
}

/* ===== PAGE HEADER ===== */
.page-header {
    background: var(--primary);
    padding: var(--space-lg) var(--space-xl);
    border-radius: var(--radius-lg);
    margin-bottom: var(--space-lg);
    box-shadow: var(--shadow-md);
    transition: all 0.3s ease;
}

.header-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: var(--space-lg);
}

.header-left h1 {
    color: white;
    font-size: var(--font-xl);
    font-weight: 700;
    margin: 0 0 4px 0;
    display: flex;
    align-items: center;
    gap: var(--space-sm);
    letter-spacing: -0.02em;
}

.header-left h1 svg {
    width: 20px;
    height: 20px;
}

.header-left p {
    color: rgba(255, 255, 255, 0.9);
    font-size: var(--font-sm);
    margin: 0;
}

/* Search Bar */
.header-search {
    flex: 1;
    max-width: 320px;
    position: relative;
}

.header-search svg {
    position: absolute;
    left: 12px;
    top: 50%;
    transform: translateY(-50%);
    width: 14px;
    height: 14px;
    color: rgba(255, 255, 255, 0.7);
    pointer-events: none;
    transition: all 0.3s ease;
}

.header-search input {
    width: 100%;
    padding: 8px 12px 8px 32px;
    border-radius: var(--radius-md);
    border: 1px solid rgba(255, 255, 255, 0.3);
    font-size: var(--font-sm);
    background: rgba(255, 255, 255, 0.15);
    color: white;
    transition: all 0.3s ease;
}

.header-search input::placeholder {
    color: rgba(255, 255, 255, 0.7);
}

.header-search input:focus {
    outline: none;
    background: white;
    color: #1a202c !important;
    border-color: white;
}

.header-search input:focus + svg {
    color: var(--primary);
}

/* Header Button */
.btn-primary {
    padding: 8px 16px;
    background: white;
    color: var(--primary);
    border: none;
    border-radius: var(--radius-md);
    font-size: var(--font-sm);
    font-weight: 600;
    text-decoration: none;
    display: flex;
    align-items: center;
    gap: 6px;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-primary:hover {
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
}

.btn-primary svg {
    width: 14px;
    height: 14px;
}

/* ===== STATS CARDS ===== */
.stats-grid {
    display: grid;
    grid-template-columns: repeat(4, 1fr);
    gap: var(--space-md);
    margin-bottom: var(--space-lg);
}

.stat-card {
    background: var(--card-bg);
    border: 1px solid var(--card-border);
    border-radius: var(--radius-lg);
    padding: var(--space-md) var(--space-lg);
    display: flex;
    align-items: center;
    gap: var(--space-md);
    transition: all 0.3s ease;
    animation: slideIn 0.3s ease;
}

.stat-card:nth-child(1) { animation-delay: 0.1s; }
.stat-card:nth-child(2) { animation-delay: 0.2s; }
.stat-card:nth-child(3) { animation-delay: 0.3s; }
.stat-card:nth-child(4) { animation-delay: 0.4s; }

.stat-card:hover {
    transform: translateY(-2px) scale(1.02);
    box-shadow: var(--shadow-md);
}

.stat-icon {
    width: 36px;
    height: 36px;
    border-radius: var(--radius-md);
    display: flex;
    align-items: center;
    justify-content: center;
    flex-shrink: 0;
    transition: all 0.3s ease;
}

.stat-card:hover .stat-icon {
    animation: pulse 1s ease infinite;
}

.stat-icon svg {
    width: 18px;
    height: 18px;
    color: white;
}

.stat-icon-primary { background: var(--primary); }
.stat-icon-success { background: var(--success); }
.stat-icon-warning { background: var(--warning); }
.stat-icon-danger { background: var(--danger); }

.stat-content h3 {
    font-size: var(--font-xl);
    font-weight: 700;
    color: var(--text-primary);
    margin: 0;
    transition: all 0.3s ease;
}

.stat-card:hover .stat-content h3 {
    transform: scale(1.05);
}

.stat-content p {
    font-size: var(--font-xs);
    color: var(--text-muted);
    margin: 2px 0 0 0;
    text-transform: uppercase;
    letter-spacing: 0.05em;
    font-weight: 600;
}

/* ===== FILTERS BAR ===== */
.filters-card {
    background: var(--card-bg);
    border: 1px solid var(--card-border);
    padding: var(--space-lg);
    border-radius: var(--radius-lg);
    margin-bottom: var(--space-lg);
    box-shadow: var(--shadow-sm);
    animation: fadeIn 0.4s ease 0.5s backwards;
}

.filters-row {
    display: flex;
    gap: var(--space-md);
    align-items: center;
    flex-wrap: wrap;
}

.quick-filters {
    display: flex;
    gap: var(--space-sm);
}

.btn-view {
    background: var(--primary-light);
    color: var(--primary);
}

.btn-view:hover {
    background: var(--primary);
    color: white;
}

.filter-btn {
    padding: 8px 14px;
    border: 1px solid var(--card-border);
    background: var(--card-bg);
    color: var(--text-secondary);
    border-radius: var(--radius-md);
    font-size: var(--font-sm);
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
}

.filter-btn:hover {
    border-color: var(--primary);
    color: var(--primary);
    transform: translateY(-1px);
}

.filter-btn.active {
    background: var(--primary);
    color: white;
    border-color: var(--primary);
    box-shadow: var(--shadow-sm);
}

.per-page {
    display: flex;
    align-items: center;
    gap: var(--space-sm);
    margin-left: auto;
}

.per-page label {
    font-size: var(--font-sm);
    color: var(--text-secondary);
    font-weight: 600;
}

.per-page select {
    padding: 8px 12px;
    border: 1px solid var(--input-border);
    background: var(--input-bg);
    color: var(--input-text);
    border-radius: var(--radius-md);
    font-size: var(--font-sm);
    cursor: pointer;
    transition: all 0.3s ease;
}

/* ===== TABLE CARD ===== */
.table-card {
    background: var(--card-bg);
    border: 1px solid var(--card-border);
    border-radius: var(--radius-lg);
    padding: var(--space-xl);
    box-shadow: var(--shadow-sm);
    animation: fadeIn 0.4s ease 0.6s backwards;
}

.table-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: var(--space-lg);
    padding-bottom: var(--space-md);
    border-bottom: 1px solid var(--card-border);
}

.table-title {
    font-size: var(--font-lg);
    font-weight: 700;
    color: var(--text-primary);
    display: flex;
    align-items: center;
    gap: var(--space-sm);
}

.table-title svg {
    width: 18px;
    height: 18px;
    color: var(--primary);
}

.pagination-info {
    font-size: var(--font-sm);
    color: var(--text-muted);
    font-weight: 600;
}

/* ===== DATA TABLE ===== */
.data-table {
    width: 100%;
    border-collapse: separate;
    border-spacing: 0;
}

.data-table thead th {
    background: var(--input-bg);
    padding: 12px var(--space-md);
    text-align: left;
    font-size: var(--font-sm);
    font-weight: 600;
    color: var(--text-secondary);
    border-bottom: 2px solid var(--card-border);
    white-space: nowrap;
    cursor: pointer;
    transition: all 0.3s ease;
}

.data-table thead th:hover {
    background: var(--card-bg);
    color: var(--primary);
}

.data-table thead th.sortable::after {
    content: '⇅';
    margin-left: 4px;
    opacity: 0.3;
    font-size: 10px;
}

.data-table thead th.sorting-asc::after {
    content: '↑';
    opacity: 1;
    color: var(--primary);
}

.data-table thead th.sorting-desc::after {
    content: '↓';
    opacity: 1;
    color: var(--primary);
}

.data-table tbody tr {
    transition: all 0.3s ease;
    border-bottom: 1px solid var(--card-border);
    animation: fadeIn 0.3s ease;
}

.data-table tbody tr:hover {
    background: var(--input-bg);
    transform: scale(1.005);
}

.data-table tbody td {
    padding: 14px var(--space-md);
    font-size: var(--font-base);
    color: var(--text-primary);
}

/* ===== BADGES ===== */
.badge {
    padding: 4px 10px;
    border-radius: 12px;
    font-size: var(--font-xs);
    font-weight: 600;
    text-transform: uppercase;
    display: inline-flex;
    align-items: center;
    gap: 4px;
    transition: all 0.3s ease;
}

.badge:hover {
    transform: scale(1.05);
}

.badge svg {
    width: 12px;
    height: 12px;
}

.badge-success {
    background: var(--success-light);
    color: var(--success);
}

.badge-warning {
    background: var(--warning-light);
    color: var(--warning);
}

.badge-danger {
    background: var(--danger-light);
    color: var(--danger);
}

/* Days Status */
.days-status {
    font-size: var(--font-xs);
    color: var(--text-muted);
    margin-top: 2px;
}

.days-status.overdue {
    color: var(--danger);
    font-weight: 600;
}

.days-status.upcoming {
    color: var(--warning);
    font-weight: 600;
}

/* ===== ACTION BUTTONS ===== */
.actions {
    display: flex;
    gap: 4px;
}

.btn-icon {
    width: 30px;
    height: 30px;
    border: none;
    border-radius: var(--radius-md);
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-icon svg {
    width: 14px;
    height: 14px;
}

.btn-icon:hover {
    transform: translateY(-1px) scale(1.1);
}

.btn-edit {
    background: var(--warning-light);
    color: var(--warning);
}

.btn-edit:hover {
    background: var(--warning);
    color: white;
}

.btn-delete {
    background: var(--danger-light);
    color: var(--danger);
}

.btn-delete:hover {
    background: var(--danger);
    color: white;
}

/* ===== PAGINATION ===== */
.pagination {
    display: flex;
    justify-content: center;
    align-items: center;
    gap: var(--space-sm);
    margin-top: var(--space-xl);
    padding-top: var(--space-lg);
    border-top: 1px solid var(--card-border);
}

.page-btn {
    padding: 8px 14px;
    border: 1px solid var(--card-border);
    background: var(--card-bg);
    color: var(--text-secondary);
    border-radius: var(--radius-md);
    font-size: var(--font-sm);
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
}

.page-btn:hover:not(:disabled) {
    border-color: var(--primary);
    color: var(--primary);
    transform: translateY(-1px);
}

.page-btn:disabled {
    opacity: 0.4;
    cursor: not-allowed;
}

.page-btn.active {
    background: var(--primary);
    color: white;
    border-color: var(--primary);
    box-shadow: var(--shadow-sm);
}

/* ===== EMPTY STATE ===== */
.empty-state {
    text-align: center;
    padding: 64px 20px;
    animation: fadeIn 0.4s ease;
}

.empty-state svg {
    width: 64px;
    height: 64px;
    color: var(--text-muted);
    margin-bottom: 16px;
    animation: pulse 2s ease infinite;
}

.empty-state h3 {
    font-size: var(--font-lg);
    color: var(--text-secondary);
    margin-bottom: 8px;
    font-weight: 600;
}

/* ===== LOADING ===== */
.loading-overlay {
    display: none;
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.5);
    z-index: 9999;
    align-items: center;
    justify-content: center;
}

.loading-overlay.active {
    display: flex;
}

@keyframes spin {
    from { transform: rotate(0deg); }
    to { transform: rotate(360deg); }
}

.spinner {
    width: 40px;
    height: 40px;
    border: 4px solid var(--card-bg);
    border-top: 4px solid var(--primary);
    border-radius: 50%;
    animation: spin 1s linear infinite;
}

/* ===== RESPONSIVE ===== */
@media (max-width: 992px) {
    .header-content {
        flex-direction: column;
        align-items: stretch;
    }
    
    .header-search {
        max-width: 100%;
    }
    
    .stats-grid {
        grid-template-columns: repeat(2, 1fr);
    }
}

@media (max-width: 768px) {
    .milestones-page {
        padding: var(--space-md);
    }
    
    .stats-grid {
        grid-template-columns: 1fr;
    }
}
</style>

<div class="milestones-page">
    <!-- Page Header -->
    <div class="page-header">
        <div class="header-content">
            <div class="header-left">
                <h1>
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M9 12l2 2 4-4M7.835 4.697a3.42 3.42 0 001.946-.806 3.42 3.42 0 014.438 0 3.42 3.42 0 001.946.806 3.42 3.42 0 013.138 3.138 3.42 3.42 0 00.806 1.946 3.42 3.42 0 010 4.438 3.42 3.42 0 00-.806 1.946 3.42 3.42 0 01-3.138 3.138 3.42 3.42 0 00-1.946.806 3.42 3.42 0 01-4.438 0 3.42 3.42 0 00-1.946-.806 3.42 3.42 0 01-3.138-3.138 3.42 3.42 0 00-.806-1.946 3.42 3.42 0 010-4.438 3.42 3.42 0 00.806-1.946 3.42 3.42 0 013.138-3.138z"/>
                    </svg>
                    Milestones
                </h1>
                <p>Project milestones and checkpoints</p>
            </div>
            
            <div class="header-search">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/>
                </svg>
                <input type="text" id="searchInput" placeholder="Search milestones...">
            </div>
            
            <a href="{{ route('admin.milestones.create') }}" class="btn-primary">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M12 4v16m8-8H4"/>
                </svg>
                Add Milestone
            </a>
        </div>
    </div>

    <!-- Stats Cards -->
    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-icon stat-icon-primary">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M9 12l2 2 4-4M7.835 4.697a3.42 3.42 0 001.946-.806 3.42 3.42 0 014.438 0 3.42 3.42 0 001.946.806 3.42 3.42 0 013.138 3.138 3.42 3.42 0 00.806 1.946 3.42 3.42 0 010 4.438 3.42 3.42 0 00-.806 1.946 3.42 3.42 0 01-3.138 3.138 3.42 3.42 0 00-1.946.806 3.42 3.42 0 01-4.438 0 3.42 3.42 0 00-1.946-.806 3.42 3.42 0 01-3.138-3.138 3.42 3.42 0 00-.806-1.946 3.42 3.42 0 010-4.438 3.42 3.42 0 00.806-1.946 3.42 3.42 0 013.138-3.138z"/>
                </svg>
            </div>
            <div class="stat-content">
                <h3 id="totalMilestones">{{ number_format($stats['total']) }}</h3>
                <p>Total Milestones</p>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon stat-icon-success">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                </svg>
            </div>
            <div class="stat-content">
                <h3 id="completedMilestones">{{ number_format($stats['completed']) }}</h3>
                <p>Completed</p>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon stat-icon-warning">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                </svg>
            </div>
            <div class="stat-content">
                <h3 id="incompleteMilestones">{{ number_format($stats['incomplete']) }}</h3>
                <p>In Progress</p>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon stat-icon-danger">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"/>
                </svg>
            </div>
            <div class="stat-content">
                <h3 id="overdueMilestones">{{ number_format($stats['overdue']) }}</h3>
                <p>Overdue</p>
            </div>
        </div>
    </div>

    <!-- Filters -->
    <div class="filters-card">
        <div class="filters-row">
            <div class="quick-filters">
                <button class="filter-btn active" data-filter="all" onclick="applyFilter(this, 'all')">All</button>
                <button class="filter-btn" data-filter="completed" onclick="applyFilter(this, 'completed')">Completed</button>
                <button class="filter-btn" data-filter="incomplete" onclick="applyFilter(this, 'incomplete')">In Progress</button>
                <button class="filter-btn" data-filter="overdue" onclick="applyFilter(this, 'overdue')">Overdue</button>
            </div>

            <div class="per-page">
                <label>Show:</label>
                <select id="perPageSelect" onchange="changePerPage(this.value)">
                    <option value="15" selected>15</option>
                    <option value="25">25</option>
                    <option value="50">50</option>
                    <option value="100">100</option>
                </select>
            </div>
        </div>
    </div>

    <!-- Milestones Table -->
    <div class="table-card">
        <div class="table-header">
            <h3 class="table-title">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"/>
                </svg>
                All Milestones
            </h3>
            <div class="pagination-info" id="paginationInfo">Showing 0-0 of 0 milestones</div>
        </div>

        <table class="data-table">
            <thead>
                <tr>
                    <th style="width: 40px;">
                        <input type="checkbox" id="selectAll" onchange="toggleSelectAll(this)">
                    </th>
                    <th class="sortable" data-column="id">ID</th>
                    <th class="sortable" data-column="name">Name</th>
                    <th class="sortable" data-column="project_id">Project</th>
                    <th class="sortable" data-column="target_date">Target Date</th>
                    <th class="sortable" data-column="is_completed">Status</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody id="milestonesBody">
                <!-- Populated via AJAX -->
            </tbody>
        </table>

        <!-- Pagination -->
        <div class="pagination" id="pagination" style="display: none;">
            <!-- Populated dynamically -->
        </div>
    </div>
</div>

<!-- Loading Overlay -->
<div class="loading-overlay" id="loadingOverlay">
    <div class="spinner"></div>
</div>

<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script>
let currentPage = 1;
let perPage = 15;
let totalRecords = 0;
let lastPage = 1;
let sortColumn = 'sort_order';
let sortDirection = 'asc';
let selectedIds = [];
let currentFilter = 'all';

$(document).ready(function() {
    loadMilestones();

    // Search with debounce
    let searchTimeout;
    $('#searchInput').on('keyup', function() {
        clearTimeout(searchTimeout);
        searchTimeout = setTimeout(() => {
            currentPage = 1;
            loadMilestones();
        }, 500);
    });

    // Column sorting
    $('.sortable').on('click', function() {
        const column = $(this).data('column');
        
        if (sortColumn === column) {
            sortDirection = sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            sortColumn = column;
            sortDirection = 'asc';
        }

        $('.sortable').removeClass('sorting-asc sorting-desc');
        $(this).addClass('sorting-' + sortDirection);

        loadMilestones();
    });
});

function loadMilestones() {
    showLoading();

    const filters = {
        search: $('#searchInput').val(),
        filter_type: currentFilter,
        page: currentPage,
        per_page: perPage,
        sort: sortColumn,
        dir: sortDirection
    };

    $.ajax({
        url: '{{ route("admin.milestones.data") }}',
        type: 'GET',
        data: filters,
        success: function(response) {
            renderMilestones(response.data);
            updatePagination(response);
            hideLoading();
        },
        error: function(xhr) {
            hideLoading();
            alert('Error: ' + (xhr.responseJSON?.message || 'Unknown error'));
        }
    });
}

function renderMilestones(milestones) {
    const tbody = $('#milestonesBody');
    tbody.empty();

    if (milestones.length === 0) {
        tbody.append(`
            <tr>
                <td colspan="7">
                    <div class="empty-state">
                        <svg fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M9 12l2 2 4-4M7.835 4.697a3.42 3.42 0 001.946-.806 3.42 3.42 0 014.438 0 3.42 3.42 0 001.946.806 3.42 3.42 0 013.138 3.138 3.42 3.42 0 00.806 1.946 3.42 3.42 0 010 4.438 3.42 3.42 0 00-.806 1.946 3.42 3.42 0 01-3.138 3.138 3.42 3.42 0 00-1.946.806 3.42 3.42 0 01-4.438 0 3.42 3.42 0 00-1.946-.806 3.42 3.42 0 01-3.138-3.138 3.42 3.42 0 00-.806-1.946 3.42 3.42 0 010-4.438 3.42 3.42 0 00.806-1.946 3.42 3.42 0 013.138-3.138z"/>
                        </svg>
                        <h3>No Milestones Found</h3>
                        <p>No milestones match your criteria</p>
                    </div>
                </td>
            </tr>
        `);
        return;
    }

    milestones.forEach(milestone => {
        // Status badge
        let statusBadge;
        if (milestone.is_completed) {
            statusBadge = `
                <span class="badge badge-success">
                    <svg fill="currentColor" viewBox="0 0 20 20">
                        <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"/>
                    </svg>
                    Completed
                </span>
            `;
        } else if (milestone.is_overdue) {
            statusBadge = `
                <span class="badge badge-danger">
                    <svg fill="currentColor" viewBox="0 0 20 20">
                        <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clip-rule="evenodd"/>
                    </svg>
                    Overdue
                </span>
            `;
        } else {
            statusBadge = `
                <span class="badge badge-warning">
                    <svg fill="currentColor" viewBox="0 0 20 20">
                        <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-12a1 1 0 10-2 0v4a1 1 0 00.293.707l2.828 2.829a1 1 0 101.415-1.415L11 9.586V6z" clip-rule="evenodd"/>
                    </svg>
                    In Progress
                </span>
            `;
        }

        // Days status
        let daysDisplay = '';
        if (milestone.days_status) {
            const className = milestone.is_overdue ? 'overdue' : 'upcoming';
            daysDisplay = `<div class="days-status ${className}">${milestone.days_status}</div>`;
        }

        const row = `
            <tr>
                <td><input type="checkbox" class="milestone-checkbox" value="${milestone.id}" onchange="toggleSelection(this)"></td>
                <td><strong>#${milestone.id}</strong></td>
                <td><strong>${milestone.name}</strong></td>
                <td>${milestone.project_display}</td>
                <td>
                    ${milestone.target_date_display}
                    ${daysDisplay}
                </td>
                <td>${statusBadge}</td>
                <td>
                    <div class="actions">
                        <a href="${milestone._show_url}" class="btn-icon btn-view" title="View" style="background: var(--primary-light); color: var(--primary);">
                            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/>
                                <path stroke-linecap="round" stroke-linejoin="round" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"/>
                            </svg>
                        </a>
                        <a href="${milestone._edit_url}" class="btn-icon btn-edit" title="Edit">
                            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"/>
                            </svg>
                        </a>
                        <button class="btn-icon btn-delete" onclick="deleteMilestone(${milestone.id})" title="Delete">
                            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/>
                            </svg>
                        </button>
                    </div>
                </td>
            </tr>
        `;
        tbody.append(row);
    });
}

function updatePagination(response) {
    totalRecords = response.total || 0;
    currentPage = response.current_page || 1;
    lastPage = response.last_page || 1;

    const startRecord = totalRecords === 0 ? 0 : ((currentPage - 1) * perPage) + 1;
    const endRecord = Math.min(currentPage * perPage, totalRecords);
    $('#paginationInfo').text(`Showing ${startRecord}-${endRecord} of ${totalRecords} milestones`);

    const pagination = $('#pagination');
    
    if (lastPage <= 1) {
        pagination.hide();
        return;
    }

    pagination.show().empty();

    // Previous button
    pagination.append(`
        <button class="page-btn" onclick="goToPage(${currentPage - 1})" ${currentPage === 1 ? 'disabled' : ''}>
            Previous
        </button>
    `);

    // Page numbers
    const maxPages = 7;
    let startPage = Math.max(1, currentPage - Math.floor(maxPages / 2));
    let endPage = Math.min(lastPage, startPage + maxPages - 1);

    if (endPage - startPage < maxPages - 1) {
        startPage = Math.max(1, endPage - maxPages + 1);
    }

    if (startPage > 1) {
        pagination.append(`<button class="page-btn" onclick="goToPage(1)">1</button>`);
        if (startPage > 2) pagination.append(`<span style="padding: 0 8px;">...</span>`);
    }

    for (let i = startPage; i <= endPage; i++) {
        pagination.append(`
            <button class="page-btn ${i === currentPage ? 'active' : ''}" onclick="goToPage(${i})">
                ${i}
            </button>
        `);
    }

    if (endPage < lastPage) {
        if (endPage < lastPage - 1) pagination.append(`<span style="padding: 0 8px;">...</span>`);
        pagination.append(`<button class="page-btn" onclick="goToPage(${lastPage})">${lastPage}</button>`);
    }

    // Next button
    pagination.append(`
        <button class="page-btn" onclick="goToPage(${currentPage + 1})" ${currentPage === lastPage ? 'disabled' : ''}>
            Next
        </button>
    `);
}

function goToPage(page) {
    if (page < 1 || page > lastPage) return;
    currentPage = page;
    loadMilestones();
}

function changePerPage(value) {
    perPage = parseInt(value);
    currentPage = 1;
    loadMilestones();
}

function applyFilter(button, type) {
    $('.filter-btn').removeClass('active');
    $(button).addClass('active');
    
    currentFilter = type;
    currentPage = 1;
    loadMilestones();
}

function toggleSelection(checkbox) {
    const id = parseInt(checkbox.value);
    if (checkbox.checked) {
        selectedIds.push(id);
    } else {
        selectedIds = selectedIds.filter(item => item !== id);
    }
}

function toggleSelectAll(checkbox) {
    $('.milestone-checkbox').prop('checked', checkbox.checked);
    selectedIds = checkbox.checked ? 
        $('.milestone-checkbox').map(function() { return parseInt(this.value); }).get() : [];
}

function deleteMilestone(milestoneId) {
    if (!confirm('Delete this milestone?')) return;

    $.ajax({
        url: `/admin/milestones/${milestoneId}`,
        type: 'DELETE',
        headers: {'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')},
        success: function(response) {
            if (response.success) {
                alert(response.message);
                loadMilestones();
                updateStats();
            }
        },
        error: function(xhr) {
            alert('Delete failed: ' + (xhr.responseJSON?.message || 'Error'));
        }
    });
}

function updateStats() {
    $.ajax({
        url: '{{ route("admin.milestones.stats") }}',
        type: 'GET',
        success: function(stats) {
            $('#totalMilestones').text(stats.total.toLocaleString());
            $('#completedMilestones').text(stats.completed.toLocaleString());
            $('#incompleteMilestones').text(stats.incomplete.toLocaleString());
            $('#overdueMilestones').text(stats.overdue.toLocaleString());
        }
    });
}

function showLoading() {
    $('#loadingOverlay').addClass('active');
}

function hideLoading() {
    $('#loadingOverlay').removeClass('active');
}
</script>