<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    public function up(): void
    {
        // =============================================
        // 1. VENDORS
        // =============================================
        Schema::create('vendors', function (Blueprint $table) {
            $table->id();
            $table->string('vendor_code', 50)->unique();
            $table->string('name');
            $table->string('display_name')->nullable();
            $table->string('contact_person')->nullable();
            $table->string('email')->nullable();
            $table->string('phone', 20)->nullable();
            $table->string('mobile', 20)->nullable();
            $table->string('website')->nullable();
            $table->enum('gst_type', ['REGISTERED', 'UNREGISTERED', 'COMPOSITION', 'SEZ'])->default('REGISTERED');
            $table->string('gst_number', 50)->nullable();
            $table->string('pan_number', 20)->nullable();
            $table->text('billing_address')->nullable();
            $table->string('billing_city', 100)->nullable();
            $table->string('billing_state', 100)->nullable();
            $table->string('billing_pincode', 10)->nullable();
            $table->string('billing_country', 100)->default('India');
            $table->text('shipping_address')->nullable();
            $table->string('shipping_city', 100)->nullable();
            $table->string('shipping_state', 100)->nullable();
            $table->string('shipping_pincode', 10)->nullable();
            $table->string('payment_terms', 100)->nullable();
            $table->integer('credit_days')->default(30);
            $table->decimal('credit_limit', 15, 2)->default(0);
            $table->decimal('opening_balance', 15, 2)->default(0);
            $table->enum('status', ['ACTIVE', 'INACTIVE', 'BLOCKED'])->default('ACTIVE');
            $table->text('notes')->nullable();
            $table->foreignId('created_by')->nullable()->constrained('admins')->nullOnDelete();
            $table->timestamps();
            $table->softDeletes();
        });

        // =============================================
        // 2. PURCHASE REQUESTS
        // =============================================
        Schema::create('purchase_requests', function (Blueprint $table) {
            $table->id();
            $table->string('pr_number', 50)->unique();
            $table->date('pr_date');
            $table->date('required_date')->nullable();
            $table->string('department', 100)->nullable();
            $table->enum('priority', ['LOW', 'NORMAL', 'HIGH', 'URGENT'])->default('NORMAL');
            $table->enum('status', ['DRAFT', 'PENDING', 'APPROVED', 'REJECTED', 'CANCELLED', 'CONVERTED'])->default('DRAFT');
            $table->string('purpose', 500)->nullable();
            $table->text('notes')->nullable();
            $table->text('rejection_reason')->nullable();
            $table->foreignId('requested_by')->nullable()->constrained('admins')->nullOnDelete();
            $table->foreignId('approved_by')->nullable()->constrained('admins')->nullOnDelete();
            $table->timestamp('approved_at')->nullable();
            $table->foreignId('created_by')->nullable()->constrained('admins')->nullOnDelete();
            $table->timestamps();
            $table->softDeletes();
        });

        // =============================================
        // 3. PURCHASE REQUEST ITEMS
        // =============================================
        Schema::create('purchase_request_items', function (Blueprint $table) {
            $table->id();
            $table->foreignId('purchase_request_id')->constrained()->cascadeOnDelete();
            $table->foreignId('product_id')->nullable();
            $table->foreignId('variation_id')->nullable();
            $table->foreignId('unit_id')->nullable();
            $table->decimal('qty', 15, 3);
            $table->decimal('ordered_qty', 15, 3)->default(0);
            $table->decimal('estimated_price', 15, 2)->nullable();
            $table->text('specifications')->nullable();
            $table->timestamps();
        });

        // =============================================
        // 4. PURCHASE ORDERS
        // =============================================
        Schema::create('purchase_orders', function (Blueprint $table) {
            $table->id();
            $table->string('po_number', 50)->unique();
            $table->foreignId('vendor_id')->constrained('vendors');
            $table->foreignId('purchase_request_id')->nullable()->constrained('purchase_requests')->nullOnDelete();
            $table->date('po_date');
            $table->date('expected_date')->nullable();
            $table->date('delivery_date')->nullable();
            $table->enum('status', ['DRAFT', 'SENT', 'CONFIRMED', 'PARTIALLY_RECEIVED', 'RECEIVED', 'CANCELLED'])->default('DRAFT');
            $table->text('shipping_address')->nullable();
            $table->string('shipping_city', 100)->nullable();
            $table->string('shipping_state', 100)->nullable();
            $table->string('shipping_pincode', 10)->nullable();
            $table->decimal('subtotal', 15, 2)->default(0);
            $table->decimal('tax_amount', 15, 2)->default(0);
            $table->decimal('discount_amount', 15, 2)->default(0);
            $table->decimal('shipping_charge', 15, 2)->default(0);
            $table->decimal('total_amount', 15, 2)->default(0);
            $table->string('payment_terms', 100)->nullable();
            $table->text('terms_conditions')->nullable();
            $table->text('notes')->nullable();
            $table->timestamp('sent_at')->nullable();
            $table->timestamp('confirmed_at')->nullable();
            $table->foreignId('created_by')->nullable()->constrained('admins')->nullOnDelete();
            $table->timestamps();
            $table->softDeletes();
        });

        // =============================================
        // 5. PURCHASE ORDER ITEMS (with tax_1, tax_2)
        // =============================================
        Schema::create('purchase_order_items', function (Blueprint $table) {
            $table->id();
            $table->foreignId('purchase_order_id')->constrained()->cascadeOnDelete();
            $table->foreignId('purchase_request_item_id')->nullable();
            $table->foreignId('product_id')->nullable();
            $table->foreignId('variation_id')->nullable();
            $table->foreignId('unit_id')->nullable();
            $table->decimal('qty', 15, 3);
            $table->decimal('received_qty', 15, 3)->default(0);
            $table->decimal('rate', 15, 2)->default(0);
            $table->decimal('discount_percent', 5, 2)->default(0);
            $table->decimal('discount_amount', 15, 2)->default(0);
            // Legacy tax columns
            $table->decimal('tax_percent', 5, 2)->default(0);
            $table->decimal('tax_amount', 15, 2)->default(0);
            // Tax 1
            $table->unsignedBigInteger('tax_1_id')->nullable();
            $table->string('tax_1_name', 100)->nullable();
            $table->decimal('tax_1_rate', 8, 2)->default(0);
            $table->decimal('tax_1_amount', 15, 2)->default(0);
            // Tax 2
            $table->unsignedBigInteger('tax_2_id')->nullable();
            $table->string('tax_2_name', 100)->nullable();
            $table->decimal('tax_2_rate', 8, 2)->default(0);
            $table->decimal('tax_2_amount', 15, 2)->default(0);
            // Total & Description
            $table->decimal('total', 15, 2)->default(0);
            $table->text('description')->nullable();
            $table->timestamps();
        });

        // =============================================
        // 6. PURCHASE SETTINGS
        // =============================================
        Schema::create('purchase_settings', function (Blueprint $table) {
            $table->id();
            $table->string('key')->unique();
            $table->text('value')->nullable();
            $table->string('group', 50)->default('general');
            $table->timestamps();
        });

        // =============================================
        // 7. GOODS RECEIPT NOTES (GRN)
        // =============================================
        Schema::create('goods_receipt_notes', function (Blueprint $table) {
            $table->id();
            $table->string('grn_number', 50)->unique();
            $table->foreignId('purchase_order_id')->constrained('purchase_orders');
            $table->foreignId('vendor_id')->constrained('vendors');
            $table->date('grn_date');
            $table->string('invoice_number', 100)->nullable();
            $table->date('invoice_date')->nullable();
            $table->string('lr_number', 100)->nullable();
            $table->string('vehicle_number', 50)->nullable();
            $table->enum('status', ['DRAFT', 'INSPECTING', 'APPROVED', 'REJECTED', 'CANCELLED'])->default('DRAFT');
            $table->foreignId('warehouse_id')->nullable();
            $table->foreignId('rack_id')->nullable();
            $table->decimal('total_qty', 15, 3)->default(0);
            $table->decimal('accepted_qty', 15, 3)->default(0);
            $table->decimal('rejected_qty', 15, 3)->default(0);
            $table->text('rejection_reason')->nullable();
            $table->text('notes')->nullable();
            $table->boolean('stock_updated')->default(false);
            $table->timestamp('approved_at')->nullable();
            $table->foreignId('approved_by')->nullable()->constrained('admins')->nullOnDelete();
            $table->foreignId('received_by')->nullable()->constrained('admins')->nullOnDelete();
            $table->foreignId('created_by')->nullable()->constrained('admins')->nullOnDelete();
            $table->timestamps();
            $table->softDeletes();

            $table->index(['purchase_order_id', 'status']);
            $table->index('grn_date');
        });

        // =============================================
        // 8. GOODS RECEIPT NOTE ITEMS (with tax_1, tax_2)
        // =============================================
        Schema::create('goods_receipt_note_items', function (Blueprint $table) {
            $table->id();
            $table->foreignId('goods_receipt_note_id')->constrained()->cascadeOnDelete();
            $table->foreignId('purchase_order_item_id')->nullable();
            $table->foreignId('product_id')->nullable();
            $table->foreignId('variation_id')->nullable();
            $table->foreignId('unit_id')->nullable();
            $table->decimal('ordered_qty', 15, 3)->default(0);
            $table->decimal('received_qty', 15, 3)->default(0);
            $table->decimal('accepted_qty', 15, 3)->default(0);
            $table->decimal('rejected_qty', 15, 3)->default(0);
            $table->decimal('rate', 15, 2)->default(0);
            $table->decimal('discount_percent', 8, 2)->default(0);
            // Tax 1
            $table->unsignedBigInteger('tax_1_id')->nullable();
            $table->string('tax_1_name', 100)->nullable();
            $table->decimal('tax_1_rate', 8, 2)->default(0);
            // Tax 2
            $table->unsignedBigInteger('tax_2_id')->nullable();
            $table->string('tax_2_name', 100)->nullable();
            $table->decimal('tax_2_rate', 8, 2)->default(0);
            // Rejection
            $table->string('rejection_reason', 255)->nullable();
            // Lot/Batch tracking
            $table->string('lot_no', 100)->nullable();
            $table->string('batch_no', 100)->nullable();
            $table->date('manufacturing_date')->nullable();
            $table->date('expiry_date')->nullable();
            // Stock references
            $table->unsignedBigInteger('stock_movement_id')->nullable();
            $table->foreignId('lot_id')->nullable();
            $table->text('notes')->nullable();
            $table->timestamps();

            $table->index(['goods_receipt_note_id', 'product_id']);
        });

        // =============================================
        // 9. PURCHASE BILLS
        // =============================================
        Schema::create('purchase_bills', function (Blueprint $table) {
            $table->id();
            $table->string('bill_number', 50)->unique();
            $table->foreignId('vendor_id')->constrained('vendors')->cascadeOnDelete();
            $table->foreignId('purchase_order_id')->nullable()->constrained('purchase_orders')->nullOnDelete();
            $table->foreignId('grn_id')->nullable()->constrained('goods_receipt_notes')->nullOnDelete();
            // Vendor Invoice Details
            $table->string('vendor_invoice_no', 100)->nullable();
            $table->date('vendor_invoice_date')->nullable();
            $table->date('bill_date');
            $table->date('due_date')->nullable();
            // Warehouse
            $table->foreignId('warehouse_id')->nullable();
            // Status
            $table->enum('status', ['DRAFT', 'PENDING', 'APPROVED', 'REJECTED', 'CANCELLED'])->default('DRAFT');
            $table->enum('payment_status', ['UNPAID', 'PARTIALLY_PAID', 'PAID'])->default('UNPAID');
            // Amounts
            $table->decimal('subtotal', 15, 2)->default(0);
            $table->decimal('tax_amount', 15, 2)->default(0);
            $table->decimal('discount_amount', 15, 2)->default(0);
            $table->decimal('shipping_charge', 15, 2)->default(0);
            $table->decimal('adjustment', 15, 2)->default(0);
            $table->decimal('grand_total', 15, 2)->default(0);
            $table->decimal('paid_amount', 15, 2)->default(0);
            $table->decimal('balance_due', 15, 2)->default(0);
            // Notes
            $table->text('notes')->nullable();
            $table->text('terms_conditions')->nullable();
            // Approval
            $table->timestamp('approved_at')->nullable();
            $table->unsignedBigInteger('approved_by')->nullable();
            $table->text('rejection_reason')->nullable();
            // Audit
            $table->unsignedBigInteger('created_by')->nullable();
            $table->timestamps();
            $table->softDeletes();

            $table->index(['vendor_id', 'status']);
            $table->index('payment_status');
        });

        // =============================================
        // 10. PURCHASE BILL ITEMS (with tax_1, tax_2)
        // =============================================
        Schema::create('purchase_bill_items', function (Blueprint $table) {
            $table->id();
            $table->foreignId('purchase_bill_id')->constrained('purchase_bills')->cascadeOnDelete();
            $table->foreignId('grn_item_id')->nullable();
            $table->foreignId('product_id')->nullable();
            $table->foreignId('variation_id')->nullable();
            $table->foreignId('unit_id')->nullable();
            $table->string('description')->nullable();
            $table->decimal('qty', 12, 3);
            $table->decimal('rate', 12, 2);
            // Legacy tax columns
            $table->decimal('tax_percent', 5, 2)->default(0);
            $table->decimal('tax_amount', 12, 2)->default(0);
            // Tax 1
            $table->unsignedBigInteger('tax_1_id')->nullable();
            $table->string('tax_1_name', 100)->nullable();
            $table->decimal('tax_1_rate', 8, 2)->default(0);
            $table->decimal('tax_1_amount', 15, 2)->default(0);
            // Tax 2
            $table->unsignedBigInteger('tax_2_id')->nullable();
            $table->string('tax_2_name', 100)->nullable();
            $table->decimal('tax_2_rate', 8, 2)->default(0);
            $table->decimal('tax_2_amount', 15, 2)->default(0);
            // Discount & Total
            $table->decimal('discount_percent', 5, 2)->default(0);
            $table->decimal('discount_amount', 12, 2)->default(0);
            $table->decimal('total', 15, 2)->default(0);
            $table->timestamps();

            $table->index('purchase_bill_id');
        });

        // =============================================
        // 11. PURCHASE PAYMENTS
        // =============================================
        Schema::create('purchase_payments', function (Blueprint $table) {
            $table->id();
            $table->string('payment_number', 50)->unique();
            $table->foreignId('purchase_bill_id')->constrained('purchase_bills')->cascadeOnDelete();
            $table->foreignId('vendor_id')->constrained('vendors')->cascadeOnDelete();
            $table->date('payment_date');
            $table->decimal('amount', 15, 2);
            $table->unsignedBigInteger('payment_method_id')->nullable();
            $table->string('reference_no', 100)->nullable();
            $table->string('bank_name', 100)->nullable();
            $table->string('cheque_no', 50)->nullable();
            $table->date('cheque_date')->nullable();
            $table->text('notes')->nullable();
            $table->enum('status', ['PENDING', 'COMPLETED', 'CANCELLED'])->default('COMPLETED');
            $table->unsignedBigInteger('created_by')->nullable();
            $table->timestamps();
            $table->softDeletes();

            $table->index(['purchase_bill_id', 'status']);
        });

        // =============================================
        // 12. INSERT DEFAULT SETTINGS
        // =============================================
        DB::table('purchase_settings')->insert([
            // General Settings
            ['key' => 'default_payment_terms', 'value' => 'Net 30', 'group' => 'general', 'created_at' => now(), 'updated_at' => now()],
            ['key' => 'default_tax_percent', 'value' => '18', 'group' => 'general', 'created_at' => now(), 'updated_at' => now()],
            // Vendor Settings
            ['key' => 'vendor_prefix', 'value' => 'VND-', 'group' => 'vendor', 'created_at' => now(), 'updated_at' => now()],
            // PR Settings
            ['key' => 'pr_prefix', 'value' => 'PR-', 'group' => 'pr', 'created_at' => now(), 'updated_at' => now()],
            ['key' => 'pr_approval_required', 'value' => '1', 'group' => 'pr', 'created_at' => now(), 'updated_at' => now()],
            // PO Settings
            ['key' => 'po_prefix', 'value' => 'PO-', 'group' => 'po', 'created_at' => now(), 'updated_at' => now()],
            ['key' => 'po_terms', 'value' => "1. Goods once sold will not be taken back.\n2. Delivery within specified time.\n3. Payment as per agreed terms.", 'group' => 'po', 'created_at' => now(), 'updated_at' => now()],
            // GRN Settings
            ['key' => 'grn_prefix', 'value' => 'GRN-', 'group' => 'grn', 'created_at' => now(), 'updated_at' => now()],
            // Bill Settings
            ['key' => 'bill_prefix', 'value' => 'BILL-', 'group' => 'bill', 'created_at' => now(), 'updated_at' => now()],
            // Payment Settings
            ['key' => 'payment_prefix', 'value' => 'PAY-', 'group' => 'payment', 'created_at' => now(), 'updated_at' => now()],
            // PDF Settings
            ['key' => 'pdf_primary_color', 'value' => '#1e40af', 'group' => 'pdf', 'created_at' => now(), 'updated_at' => now()],
            ['key' => 'pdf_secondary_color', 'value' => '#f3f4f6', 'group' => 'pdf', 'created_at' => now(), 'updated_at' => now()],
            ['key' => 'pdf_show_logo', 'value' => '1', 'group' => 'pdf', 'created_at' => now(), 'updated_at' => now()],
            ['key' => 'pdf_show_gst', 'value' => '1', 'group' => 'pdf', 'created_at' => now(), 'updated_at' => now()],
            ['key' => 'pdf_show_terms', 'value' => '1', 'group' => 'pdf', 'created_at' => now(), 'updated_at' => now()],
            ['key' => 'pdf_show_signature', 'value' => '1', 'group' => 'pdf', 'created_at' => now(), 'updated_at' => now()],
            ['key' => 'pdf_show_notes', 'value' => '1', 'group' => 'pdf', 'created_at' => now(), 'updated_at' => now()],
            ['key' => 'pdf_compact_mode', 'value' => '1', 'group' => 'pdf', 'created_at' => now(), 'updated_at' => now()],
            ['key' => 'pdf_font_size', 'value' => '9', 'group' => 'pdf', 'created_at' => now(), 'updated_at' => now()],
        ]);
    }

    public function down(): void
    {
        Schema::dropIfExists('purchase_payments');
        Schema::dropIfExists('purchase_bill_items');
        Schema::dropIfExists('purchase_bills');
        Schema::dropIfExists('goods_receipt_note_items');
        Schema::dropIfExists('goods_receipt_notes');
        Schema::dropIfExists('purchase_settings');
        Schema::dropIfExists('purchase_order_items');
        Schema::dropIfExists('purchase_orders');
        Schema::dropIfExists('purchase_request_items');
        Schema::dropIfExists('purchase_requests');
        Schema::dropIfExists('vendors');
    }
};