<?php

namespace Modules\Purchase\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\Purchase\Models\GoodsReceiptNote;
use Modules\Purchase\Models\GoodsReceiptNoteItem;
use Modules\Purchase\Models\PurchaseOrder;
use Modules\Purchase\Models\PurchaseSetting;
use App\Traits\DataTable;
use Modules\Purchase\Models\Vendor;

class GoodsReceiptNoteController extends AdminController
{
    use DataTable;

    public function __construct()
    {
        // Check if Inventory module is available
        if (!class_exists('\Modules\Inventory\Models\Warehouse')) {
            abort(503, 'GRN functionality requires the Inventory module to be installed.');
        }
    }
    
    // DataTable Configuration
    protected $model = GoodsReceiptNote::class;
    protected $with = ['vendor', 'purchaseOrder', 'warehouse', 'creator'];
    protected $searchable = ['grn_number', 'invoice_number', 'vendor.name', 'purchaseOrder.po_number'];
    protected $sortable = ['id', 'grn_number', 'grn_date', 'status', 'created_at'];
    protected $filterable = ['status', 'warehouse_id', 'vendor_id'];
    protected $exportTitle = 'Goods Receipt Notes Export';

    // Permission Configuration
    protected $viewPermission = 'purchase.grn.read';
    protected $createPermission = 'purchase.grn.create';
    protected $editPermission = 'purchase.grn.edit';
    protected $deletePermission = 'purchase.grn.delete';
    protected $approvePermission = 'purchase.grn.approve';

    /**
     * Display GRN listing
     */
    public function index()
    {
        $this->authorize('purchase.grn.read');
        
        $stats = [
            'total' => GoodsReceiptNote::count(),
            'draft' => GoodsReceiptNote::where('status', 'DRAFT')->count(),
            'inspecting' => GoodsReceiptNote::where('status', 'INSPECTING')->count(),
            'approved' => GoodsReceiptNote::where('status', 'APPROVED')->count(),
        ];

        $vendors = Vendor::where('status', 'ACTIVE')->orderBy('name')->pluck('name', 'id');
        
        $warehouses = collect();
        if (class_exists('\Modules\Inventory\Models\Warehouse')) {
            $warehouses = \Modules\Inventory\Models\Warehouse::orderBy('name')->pluck('name', 'id');
        }

        return view('purchase::grn.index', compact('stats', 'vendors', 'warehouses'));
    }

    /**
     * DataTable row mapping for list view
     */
    protected function mapRow($item)
    {
        $row = [
            'id' => $item->id,
            'grn_number' => $item->grn_number,
            'grn_date' => $item->grn_date->format('d M Y'),
            'po_number' => $item->purchaseOrder->po_number ?? '-',
            'po_id' => $item->purchase_order_id,
            'vendor_name' => $item->vendor->name ?? '-',
            'vendor_id' => $item->vendor_id,
            'warehouse_name' => $item->warehouse->name ?? '-',
            'invoice_number' => $item->invoice_number ?? '-',
            'total_qty' => number_format($item->total_qty, 2),
            'accepted_qty' => number_format($item->accepted_qty, 2),
            'rejected_qty' => number_format($item->rejected_qty, 2),
            'status' => $item->status,
            'status_badge' => $item->status_badge ?? $item->status,
            'stock_updated' => $item->stock_updated,
            'created_by' => $item->creator->name ?? '-',
            'created_at' => $item->created_at->format('d M Y'),
            '_show_url' => route('admin.purchase.grn.show', $item->id),
            '_edit_url' => route('admin.purchase.grn.edit', $item->id),
        ];
        
        // Quick action URLs for list view
        if (in_array($item->status, ['DRAFT', 'INSPECTING'])) {
            $row['_approve_url'] = route('admin.purchase.grn.approve', $item->id);
            $row['_reject_url'] = route('admin.purchase.grn.reject', $item->id);
        }
        if ($item->status === 'APPROVED') {
            $row['_bill_url'] = route('admin.purchase.bills.create') . '?grn_id=' . $item->id;
        }
        
        return $row;
    }

    /**
     * DataTable row mapping for export
     */
    protected function mapExportRow($item)
    {
        return [
            'ID' => $item->id,
            'GRN Number' => $item->grn_number,
            'Date' => $item->grn_date->format('Y-m-d'),
            'PO Number' => $item->purchaseOrder->po_number ?? '',
            'Vendor' => $item->vendor->name ?? '',
            'Warehouse' => $item->warehouse->name ?? '',
            'Invoice No' => $item->invoice_number ?? '',
            'Total Qty' => $item->total_qty,
            'Accepted Qty' => $item->accepted_qty,
            'Rejected Qty' => $item->rejected_qty,
            'Stock Updated' => $item->stock_updated ? 'Yes' : 'No',
            'Status' => $item->status,
        ];
    }

    /**
     * DataTable endpoint — intercepts ALL exports for GSTR2 format
     * DataTable sends ?export=xlsx/csv/pdf with filters (from_date, to_date)
     */
    public function dataTable(Request $request)
    {
        if ($request->has('export')) {
            $format = strtolower($request->get('export', 'csv'));
            if (in_array($format, ['csv', 'xlsx', 'pdf', 'excel'])) {
                $request->merge([
                    'from_date' => $request->get('from_date', date('Y-m-01')),
                    'to_date'   => $request->get('to_date', date('Y-m-t')),
                    'format'    => ($format === 'excel') ? 'xlsx' : $format,
                ]);
                return $this->exportGstr2($request);
            }
        }

        // Date range filter — inject as global scope so handleData picks it up
        if ($request->filled('from_date')) {
            GoodsReceiptNote::addGlobalScope('from_date', function ($query) use ($request) {
                $query->where('grn_date', '>=', $request->from_date);
            });
        }
        if ($request->filled('to_date')) {
            GoodsReceiptNote::addGlobalScope('to_date', function ($query) use ($request) {
                $query->where('grn_date', '<=', $request->to_date);
            });
        }

        return $this->handleData($request);
    }

    /**
     * Show create form - Select PO first
     */
  public function create(Request $request)
{
    $this->authorize('purchase.grn.create');
    
    // Get confirmed POs that are not fully received
    $purchaseOrders = PurchaseOrder::with('vendor')
        ->whereIn('status', ['CONFIRMED', 'PARTIALLY_RECEIVED'])
        ->orderBy('po_date', 'desc')
        ->get();

    $warehouses = \Modules\Inventory\Models\Warehouse::where('is_active', true)->orderBy('name')->get();
    
    // Get all active vendors that have vendor items
    $vendors = Vendor::where('status', 'ACTIVE')
        ->whereHas('activeVendorItems')
        ->orderBy('name')
        ->get();

    $selectedPO = null;
    if ($request->has('po_id')) {
        $selectedPO = PurchaseOrder::with(['vendor', 'items.product', 'items.unit'])
            ->find($request->po_id);
    }

    return view('purchase::grn.create', compact('purchaseOrders', 'warehouses', 'selectedPO', 'vendors'));
}

    /**
     * Store new GRN
     */
public function store(Request $request)
{
    $this->authorize('purchase.grn.create');
    
    $receiptMode = $request->input('receipt_mode', 'po');
    
    if ($receiptMode === 'vendor') {
        // ========== VENDOR DIRECT MODE ==========
        $validated = $request->validate([
            'direct_vendor_id' => 'required|exists:vendors,id',
            'grn_date' => 'required|date',
            'warehouse_id' => 'required|exists:warehouses,id',
            'rack_id' => 'nullable|exists:racks,id',
            'invoice_number' => 'nullable|string|max:100',
            'invoice_date' => 'nullable|date',
            'lr_number' => 'nullable|string|max:100',
            'vehicle_number' => 'nullable|string|max:50',
            'notes' => 'nullable|string|max:1000',
            'items' => 'required|array|min:1',
            'items.*.vendor_item_id' => 'nullable|exists:vendor_items,id',
            'items.*.product_id' => 'required|integer',
            'items.*.variation_id' => 'nullable',
            'items.*.unit_id' => 'nullable',
            'items.*.rate' => 'nullable|numeric|min:0',
            'items.*.selling_price' => 'nullable|numeric|min:0',
            'items.*.mrp' => 'nullable|numeric|min:0',
            'items.*.hsn_code' => 'nullable|string|max:20',
            'items.*.gst_rate' => 'nullable|numeric|min:0',
            'items.*.received_qty' => 'required|numeric|min:0',
            'items.*.accepted_qty' => 'required|numeric|min:0',
            'items.*.rejected_qty' => 'nullable|numeric|min:0',
            'items.*.lot_no' => 'nullable|string|max:100',
            'items.*.batch_no' => 'nullable|string|max:100',
            'items.*.manufacturing_date' => 'nullable|date',
            'items.*.expiry_date' => 'nullable|date',
        ]);

        DB::beginTransaction();
        try {
            $grn = GoodsReceiptNote::create([
                'grn_number' => GoodsReceiptNote::generateGrnNumber(),
                'purchase_order_id' => null, // No PO for vendor direct mode
                'vendor_id' => $validated['direct_vendor_id'],
                'grn_date' => $validated['grn_date'],
                'warehouse_id' => $validated['warehouse_id'],
                'rack_id' => $validated['rack_id'] ?? null,
                'invoice_number' => $validated['invoice_number'] ?? null,
                'invoice_date' => $validated['invoice_date'] ?? null,
                'lr_number' => $validated['lr_number'] ?? null,
                'vehicle_number' => $validated['vehicle_number'] ?? null,
                'notes' => $validated['notes'] ?? null,
                'status' => 'DRAFT',
                'received_by' => auth()->id(),
                'created_by' => auth()->id(),
            ]);

            foreach ($validated['items'] as $itemData) {
                $variationId = !empty($itemData['variation_id']) ? $itemData['variation_id'] : null;
                $gstRate = !empty($itemData['gst_rate']) ? (float)$itemData['gst_rate'] : 0;
                
                // Resolve tax name from GST rate
                $taxName = null;
                $taxId = null;
                if ($gstRate > 0) {
                    $tax = DB::table('taxes')->where('rate', $gstRate)->where('is_active', 1)->first();
                    if ($tax) {
                        $taxId = $tax->id;
                        $taxName = $tax->name;
                    } else {
                        $taxName = "GST {$gstRate}%";
                    }
                }

                $grn->items()->create([
                    'purchase_order_item_id' => null,
                    'product_id' => $itemData['product_id'],
                    'variation_id' => $variationId,
                    'unit_id' => !empty($itemData['unit_id']) ? $itemData['unit_id'] : null,
                    'ordered_qty' => $itemData['received_qty'] ?? 0,
                    'received_qty' => $itemData['received_qty'] ?? 0,
                    'accepted_qty' => $itemData['accepted_qty'] ?? 0,
                    'rejected_qty' => max(0, ($itemData['received_qty'] ?? 0) - ($itemData['accepted_qty'] ?? 0)),
                    'rate' => $itemData['rate'] ?? 0,
                    'tax_1_id' => $taxId,
                    'tax_1_name' => $taxName,
                    'tax_1_rate' => $gstRate,
                    'lot_no' => $itemData['lot_no'] ?? null,
                    'batch_no' => $itemData['batch_no'] ?? null,
                    'manufacturing_date' => $itemData['manufacturing_date'] ?? null,
                    'expiry_date' => $itemData['expiry_date'] ?? null,
                ]);
                
                // Update product with selling_price, mrp, hsn_code if provided
                if (class_exists('\Modules\Inventory\Models\Product')) {
                    $updateData = [];
                    if (!empty($itemData['selling_price'])) $updateData['sale_price'] = $itemData['selling_price'];
                    if (!empty($itemData['mrp'])) $updateData['mrp'] = $itemData['mrp'];
                    if (!empty($itemData['hsn_code'])) $updateData['hsn_code'] = $itemData['hsn_code'];
                    if (!empty($updateData)) {
                        DB::table("products")->where("id", $itemData["product_id"])->update($updateData);
                    }
                }
            }

            $grn->calculateTotals();

            DB::commit();
            return redirect()->route('admin.purchase.grn.show', $grn->id)
                ->with('success', 'GRN created from vendor items! Submit for inspection when ready.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withInput()->with('error', 'Error creating GRN: ' . $e->getMessage());
        }
        
    } else {
        // ========== PO MODE (original flow) ==========
        $validated = $request->validate([
            'purchase_order_id' => 'required|exists:purchase_orders,id',
            'grn_date' => 'required|date',
            'warehouse_id' => 'required|exists:warehouses,id',
            'rack_id' => 'nullable|exists:racks,id',
            'invoice_number' => 'nullable|string|max:100',
            'invoice_date' => 'nullable|date',
            'lr_number' => 'nullable|string|max:100',
            'vehicle_number' => 'nullable|string|max:50',
            'notes' => 'nullable|string|max:1000',
            'items' => 'required|array|min:1',
            'items.*.po_item_id' => 'required|exists:purchase_order_items,id',
            'items.*.product_id' => 'required|integer',
            'items.*.variation_id' => 'nullable',
            'items.*.unit_id' => 'nullable',
            'items.*.rate' => 'nullable|numeric|min:0',
            'items.*.selling_price' => 'nullable|numeric|min:0',
            'items.*.mrp' => 'nullable|numeric|min:0',
            'items.*.hsn_code' => 'nullable|string|max:20',
            'items.*.gst_rate' => 'nullable|numeric|min:0',
            'items.*.received_qty' => 'required|numeric|min:0',
            'items.*.accepted_qty' => 'required|numeric|min:0',
            'items.*.rejected_qty' => 'nullable|numeric|min:0',
            'items.*.lot_no' => 'nullable|string|max:100',
            'items.*.batch_no' => 'nullable|string|max:100',
            'items.*.manufacturing_date' => 'nullable|date',
            'items.*.expiry_date' => 'nullable|date',
        ]);

        $po = PurchaseOrder::with('vendor')->findOrFail($validated['purchase_order_id']);

        DB::beginTransaction();
        try {
            $grn = GoodsReceiptNote::create([
                'grn_number' => GoodsReceiptNote::generateGrnNumber(),
                'purchase_order_id' => $po->id,
                'vendor_id' => $po->vendor_id,
                'grn_date' => $validated['grn_date'],
                'warehouse_id' => $validated['warehouse_id'],
                'rack_id' => $validated['rack_id'] ?? null,
                'invoice_number' => $validated['invoice_number'] ?? null,
                'invoice_date' => $validated['invoice_date'] ?? null,
                'lr_number' => $validated['lr_number'] ?? null,
                'vehicle_number' => $validated['vehicle_number'] ?? null,
                'notes' => $validated['notes'] ?? null,
                'status' => 'DRAFT',
                'received_by' => auth()->id(),
                'created_by' => auth()->id(),
            ]);

            foreach ($validated['items'] as $itemData) {
                if (($itemData['received_qty'] ?? 0) <= 0) continue;

                $poItem = $po->items()->find($itemData['po_item_id']);
                if (!$poItem) continue;

                $variationId = !empty($itemData['variation_id']) ? $itemData['variation_id'] : $poItem->variation_id;
                
                // Tax from form gst_rate, fallback to PO item tax
                $gstRate = isset($itemData['gst_rate']) && $itemData['gst_rate'] !== '' ? (float)$itemData['gst_rate'] : ($poItem->tax_1_rate ?? 0);
                $taxName = $poItem->tax_1_name ?? null;
                $taxId = $poItem->tax_1_id ?? null;
                if ($gstRate > 0 && !$taxName) {
                    $tax = DB::table('taxes')->where('rate', $gstRate)->where('is_active', 1)->first();
                    $taxId = $tax ? $tax->id : null;
                    $taxName = $tax ? $tax->name : "GST {$gstRate}%";
                }

                $grn->items()->create([
                    'purchase_order_item_id' => $poItem->id,
                    'product_id' => $itemData['product_id'],
                    'variation_id' => $variationId,
                    'unit_id' => $itemData['unit_id'] ?? $poItem->unit_id,
                    'ordered_qty' => $poItem->qty,
                    'received_qty' => $itemData['received_qty'],
                    'accepted_qty' => $itemData['accepted_qty'],
                    'rejected_qty' => max(0, ($itemData['received_qty'] ?? 0) - ($itemData['accepted_qty'] ?? 0)),
                    'rate' => $itemData['rate'] ?? $poItem->rate,
                    'tax_1_id' => $taxId,
                    'tax_1_name' => $taxName,
                    'tax_1_rate' => $gstRate,
                    'tax_2_id' => $poItem->tax_2_id ?? null,
                    'tax_2_name' => $poItem->tax_2_name ?? null,
                    'tax_2_rate' => $poItem->tax_2_rate ?? 0,
                    'lot_no' => $itemData['lot_no'] ?? null,
                    'batch_no' => $itemData['batch_no'] ?? null,
                    'manufacturing_date' => $itemData['manufacturing_date'] ?? null,
                    'expiry_date' => $itemData['expiry_date'] ?? null,
                ]);
                
                // Update product with selling_price, mrp, hsn_code if provided
                if (class_exists('\Modules\Inventory\Models\Product')) {
                    $updateData = [];
                    if (!empty($itemData['selling_price'])) $updateData['sale_price'] = $itemData['selling_price'];
                    if (!empty($itemData['mrp'])) $updateData['mrp'] = $itemData['mrp'];
                    if (!empty($itemData['hsn_code'])) $updateData['hsn_code'] = $itemData['hsn_code'];
                    if (!empty($updateData)) {
                        DB::table("products")->where("id", $itemData["product_id"])->update($updateData);
                    }
                }
            }

            $grn->calculateTotals();

            DB::commit();
            return redirect()->route('admin.purchase.grn.show', $grn->id)
                ->with('success', 'GRN created successfully! Submit for inspection when ready.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withInput()->with('error', 'Error creating GRN: ' . $e->getMessage());
        }
    }
}


    /**
     * Show GRN details
     */
    public function show(Request $request, $id)
    {
        $this->authorize('purchase.grn.read');
        
        $grn = GoodsReceiptNote::with([
            'vendor',
            'purchaseOrder',
            'warehouse',
            'rack',
            'items.product',
            'items.variation',
            'items.unit',
            'items.lot',
            'items.purchaseOrderItem',
            'creator',
            'receiver',
            'approver',
        ])->findOrFail($id);

        // Return JSON for AJAX requests
        if ($request->wantsJson() || $request->input('format') === 'json') {
            return response()->json([
                'id' => $grn->id,
                'grn_number' => $grn->grn_number,
                'vendor_id' => $grn->vendor_id,
                'vendor' => $grn->vendor,
                'warehouse_id' => $grn->warehouse_id,
                'items' => $grn->items->map(function($item) {
                    // Get tax - first from GRN item, then fallback to PO item
                    $tax1Id = $item->tax_1_id;
                    $tax1Name = $item->tax_1_name;
                    $tax1Rate = $item->tax_1_rate ?? 0;
                    $tax2Id = $item->tax_2_id;
                    $tax2Name = $item->tax_2_name;
                    $tax2Rate = $item->tax_2_rate ?? 0;
                    
                    // Fallback to PO item if GRN doesn't have tax
                    if (empty($tax1Id) && empty($tax1Rate) && $item->purchaseOrderItem) {
                        $poItem = $item->purchaseOrderItem;
                        $tax1Id = $poItem->tax_1_id;
                        $tax1Name = $poItem->tax_1_name;
                        $tax1Rate = $poItem->tax_1_rate ?? 0;
                        $tax2Id = $poItem->tax_2_id;
                        $tax2Name = $poItem->tax_2_name;
                        $tax2Rate = $poItem->tax_2_rate ?? 0;
                    }
                    
                    // Get product - with DB fallback if Eloquent returns null (SoftDeletes)
                    $product = $item->product;
                    if (!$product && $item->product_id) {
                        $rawProduct = DB::table('products')->where('id', $item->product_id)->first();
                        $product = $rawProduct ? (object)[
                            'name' => $rawProduct->name,
                            'sku' => $rawProduct->sku ?? '',
                            'hsn_code' => $rawProduct->hsn_code ?? '',
                        ] : null;
                    }
                    
                    return [
                        'id' => $item->id,
                        'product_id' => $item->product_id,
                        'product' => $product,
                        'unit_id' => $item->unit_id,
                        'unit' => $item->unit,
                        'variation_id' => $item->variation_id,
                        'ordered_qty' => $item->ordered_qty,
                        'received_qty' => $item->received_qty,
                        'accepted_qty' => $item->accepted_qty,
                        'rejected_qty' => $item->rejected_qty,
                        'rate' => $item->rate,
                        'discount_percent' => $item->discount_percent ?? 0,
                        // Tax fields
                        'tax_1_id' => $tax1Id,
                        'tax_1_name' => $tax1Name,
                        'tax_1_rate' => $tax1Rate,
                        'tax_2_id' => $tax2Id,
                        'tax_2_name' => $tax2Name,
                        'tax_2_rate' => $tax2Rate,
                    ];
                }),
            ]);
        }

        return view('purchase::grn.show', compact('grn'));
    }

    /**
     * Edit GRN (only Draft/Inspecting)
     */
    public function edit($id)
    {
        $this->authorize('purchase.grn.edit');
        
        $grn = GoodsReceiptNote::with([
            'items.product',
            'items.unit',
            'purchaseOrder.items',
        ])->findOrFail($id);

        if (!$grn->can_edit) {
            return redirect()
                ->route('admin.purchase.grn.show', $grn->id)
                ->with('error', 'Cannot edit GRN in current status.');
        }

        $warehouses = \Modules\Inventory\Models\Warehouse::where('is_active', true)->orderBy('name')->get();
        $racks = $grn->warehouse_id 
            ? \Modules\Inventory\Models\Rack::where('warehouse_id', $grn->warehouse_id)->where('is_active', true)->get() 
            : collect();

        return view('purchase::grn.edit', compact('grn', 'warehouses', 'racks'));
    }

    /**
     * Update GRN
     */
    public function update(Request $request, $id)
    {
        $this->authorize('purchase.grn.edit');
        
        $grn = GoodsReceiptNote::findOrFail($id);

        if (!$grn->can_edit) {
            return back()->with('error', 'Cannot update GRN in current status.');
        }

        $validated = $request->validate([
            'grn_date' => 'required|date',
            'warehouse_id' => 'required|exists:warehouses,id',
            'rack_id' => 'nullable|exists:racks,id',
            'invoice_number' => 'nullable|string|max:100',
            'invoice_date' => 'nullable|date',
            'lr_number' => 'nullable|string|max:100',
            'vehicle_number' => 'nullable|string|max:50',
            'notes' => 'nullable|string|max:1000',
            'items' => 'required|array|min:1',
            'items.*.id' => 'required|exists:goods_receipt_note_items,id',
            'items.*.received_qty' => 'nullable|numeric|min:0',
            'items.*.accepted_qty' => 'nullable|numeric|min:0',
            'items.*.rejected_qty' => 'nullable|numeric|min:0',
            'items.*.lot_no' => 'nullable|string|max:100',
            'items.*.batch_no' => 'nullable|string|max:100',
            'items.*.manufacturing_date' => 'nullable|date',
            'items.*.expiry_date' => 'nullable|date',
        ]);

        DB::beginTransaction();
        try {
            $grn->update([
                'grn_date' => $validated['grn_date'],
                'warehouse_id' => $validated['warehouse_id'],
                'rack_id' => $validated['rack_id'] ?? null,
                'invoice_number' => $validated['invoice_number'] ?? null,
                'invoice_date' => $validated['invoice_date'] ?? null,
                'lr_number' => $validated['lr_number'] ?? null,
                'vehicle_number' => $validated['vehicle_number'] ?? null,
                'notes' => $validated['notes'] ?? null,
            ]);

            // Update items
            foreach ($validated['items'] as $itemData) {
                $grnItem = $grn->items()->find($itemData['id']);
                if ($grnItem) {
                    $grnItem->update([
                        'received_qty' => $itemData['received_qty'] ?? $grnItem->received_qty,
                        'accepted_qty' => $itemData['accepted_qty'] ?? $grnItem->accepted_qty,
                        'rejected_qty' => $itemData['rejected_qty'] ?? 0,
                        'lot_no' => $itemData['lot_no'] ?? null,
                        'batch_no' => $itemData['batch_no'] ?? null,
                        'manufacturing_date' => $itemData['manufacturing_date'] ?? null,
                        'expiry_date' => $itemData['expiry_date'] ?? null,
                    ]);
                }
            }

            $grn->calculateTotals();

            DB::commit();

            return redirect()
                ->route('admin.purchase.grn.show', $grn->id)
                ->with('success', 'GRN updated successfully!');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error updating GRN: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * Submit for inspection (alias: inspect)
     */
    public function submit($id)
    {
        $this->authorize('purchase.grn.approve');
        
        $grn = GoodsReceiptNote::findOrFail($id);

        if ($grn->status !== 'DRAFT') {
            return back()->with('error', 'Only draft GRNs can be submitted.');
        }

        $grn->update(['status' => 'INSPECTING']);

        return back()->with('success', 'GRN submitted for inspection.');
    }

    /**
     * Start inspection (change status from DRAFT to INSPECTING)
     */
    public function inspect($id)
    {
        return $this->submit($id);
    }

    /**
     * Approve GRN and update stock
     */
    public function approve($id)
    {
        $this->authorize('purchase.grn.approve');
        
        $grn = GoodsReceiptNote::with(['items.product'])->findOrFail($id);

        if (!in_array($grn->status, ['DRAFT', 'INSPECTING'])) {
            return back()->with('error', 'Only draft/inspecting GRNs can be approved.');
        }

        if ($grn->stock_updated) {
            return back()->with('error', 'Stock already updated for this GRN.');
        }

        DB::beginTransaction();
        try {
            // Update stock for each item
            foreach ($grn->items as $item) {
                if ($item->accepted_qty <= 0) continue;

                $product = $item->product;
                
                // Fallback: if Eloquent returns null (SoftDeletes), try direct DB
                if (!$product && $item->product_id) {
                    $rawProduct = DB::table('products')->where('id', $item->product_id)->first();
                    if ($rawProduct) {
                        $product = (object) [
                            'id' => $rawProduct->id,
                            'name' => $rawProduct->name,
                            'track_inventory' => $rawProduct->track_inventory ?? true,
                            'is_batch_managed' => $rawProduct->is_batch_managed ?? false,
                            'unit_id' => $rawProduct->unit_id ?? null,
                        ];
                    }
                }
                
                if (!$product || !$product->track_inventory) continue;

                $lotId = null;

                // Handle lot/batch if product is batch managed
                if ($product->is_batch_managed && $item->lot_no) {
                    $lot = \Modules\Inventory\Models\Lot::firstOrCreate([
                        'product_id' => $product->id,
                        'lot_no' => $item->lot_no,
                    ], [
                        'variation_id' => $item->variation_id,
                        'batch_no' => $item->batch_no,
                        'manufacturing_date' => $item->manufacturing_date,
                        'expiry_date' => $item->expiry_date,
                        'purchase_price' => $item->rate,
                        'initial_qty' => $item->accepted_qty,
                        'status' => 'ACTIVE',
                    ]);

                    $lotId = $lot->id;
                    $item->lot_id = $lotId;
                    $item->save();
                }

                // Get variation_id - handle empty string as null
                $variationId = !empty($item->variation_id) ? $item->variation_id : null;

                // Get current stock before update (filter by variation_id too)
                $stockBeforeQuery = \Modules\Inventory\Models\StockLevel::where('product_id', $product->id)
                    ->where('warehouse_id', $grn->warehouse_id)
                    ->when($grn->rack_id, fn($q) => $q->where('rack_id', $grn->rack_id))
                    ->when($lotId, fn($q) => $q->where('lot_id', $lotId));
                
                if ($variationId) {
                    $stockBeforeQuery->where('variation_id', $variationId);
                } else {
                    $stockBeforeQuery->whereNull('variation_id');
                }
                $stockBefore = $stockBeforeQuery->sum('qty') ?? 0;

                // Update/Create stock level
                $stockLevel = \Modules\Inventory\Models\StockLevel::firstOrNew([
                    'product_id' => $product->id,
                    'variation_id' => $variationId,
                    'warehouse_id' => $grn->warehouse_id,
                    'rack_id' => $grn->rack_id,
                    'lot_id' => $lotId,
                ]);

                $stockLevel->unit_id = $item->unit_id ?? $product->unit_id;
                $stockLevel->qty = ($stockLevel->qty ?? 0) + $item->accepted_qty;
                $stockLevel->save();

                $stockAfter = $stockLevel->qty;
                
                // Also update product_variations.stock_qty if variation exists
                if ($variationId) {
                    if (\Illuminate\Support\Facades\Schema::hasColumn('product_variations', 'stock_qty')) {
                        $updated = \Illuminate\Support\Facades\DB::table('product_variations')
                            ->where('id', $variationId)
                            ->increment('stock_qty', $item->accepted_qty);
                        
                        \Log::info("GRN Stock Update - Variation: {$variationId}, Qty: {$item->accepted_qty}, Updated: {$updated}");
                    }
                } else {
                    // Update product.stock_qty if no variation
                    if (\Illuminate\Support\Facades\Schema::hasColumn('products', 'stock_qty')) {
                        \Illuminate\Support\Facades\DB::table('products')
                            ->where('id', $product->id)
                            ->increment('stock_qty', $item->accepted_qty);
                    }
                }

                // Create stock movement record
                $refNo = 'GRN-' . $grn->grn_number . '-' . str_pad($item->id, 3, '0', STR_PAD_LEFT);
                
                $movement = \Modules\Inventory\Models\StockMovement::create([
                    'reference_no' => $refNo,
                    'product_id' => $product->id,
                    'variation_id' => $variationId,
                    'warehouse_id' => $grn->warehouse_id,
                    'rack_id' => $grn->rack_id,
                    'lot_id' => $lotId,
                    'unit_id' => $item->unit_id ?? $product->unit_id,
                    'qty' => $item->accepted_qty,
                    'base_qty' => $item->accepted_qty, // Assuming same unit, adjust if needed
                    'stock_before' => $stockBefore,
                    'stock_after' => $stockAfter,
                    'purchase_price' => $item->rate,
                    'movement_type' => 'IN',
                    'reference_type' => 'PURCHASE',
                    'reference_id' => $grn->id,
                    'reason' => 'GRN Receipt: ' . $grn->grn_number,
                    'notes' => 'PO: ' . ($grn->purchaseOrder->po_number ?? '-'),
                    'created_by' => auth()->id(),
                ]);

                // Link movement to GRN item
                $item->stock_movement_id = $movement->id;
                $item->save();
            }

            // Update GRN status
            $grn->update([
                'status' => 'APPROVED',
                'stock_updated' => true,
                'approved_at' => now(),
                'approved_by' => auth()->id(),
            ]);

            // Update Purchase Order status
            $grn->updatePurchaseOrder();

            DB::commit();

            return back()->with('success', 'GRN approved and stock updated successfully!');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error approving GRN: ' . $e->getMessage());
        }
    }

    /**
     * Reject GRN
     */
    public function reject(Request $request, $id)
    {
        $this->authorize('purchase.grn.approve');
        
        $grn = GoodsReceiptNote::findOrFail($id);

        if (!in_array($grn->status, ['DRAFT', 'INSPECTING'])) {
            return back()->with('error', 'Cannot reject GRN in current status.');
        }

        $request->validate([
            'rejection_reason' => 'required|string|max:500',
        ]);

        $grn->update([
            'status' => 'REJECTED',
            'rejection_reason' => $request->rejection_reason,
        ]);

        return back()->with('success', 'GRN rejected.');
    }

    /**
     * Cancel GRN
     */
    public function cancel($id)
    {
        $grn = GoodsReceiptNote::findOrFail($id);

        if ($grn->stock_updated) {
            return back()->with('error', 'Cannot cancel GRN after stock update.');
        }

        if ($grn->status === 'APPROVED') {
            return back()->with('error', 'Cannot cancel approved GRN.');
        }

        $grn->update(['status' => 'CANCELLED']);

        return back()->with('success', 'GRN cancelled.');
    }

    /**
     * Generate PDF for GRN
     */
    public function pdf($id)
    {
        $grn = GoodsReceiptNote::with([
            'vendor', 'purchaseOrder', 'warehouse', 'rack',
            'items.product', 'items.variation', 'items.unit',
            'receiver', 'approver',
        ])->findOrFail($id);

        $settings = [
            'show_logo' => PurchaseSetting::getValue('pdf_show_logo', true),
            'show_signature' => PurchaseSetting::getValue('pdf_show_signature', true),
            'signature_text' => PurchaseSetting::getValue('pdf_signature_text', 'Authorized Signatory'),
            'footer_text' => PurchaseSetting::getValue('pdf_footer_text', ''),
        ];

        $company = [];
        if (class_exists('\App\Models\Option')) {
            $company = [
                'name' => \App\Models\Option::get('company_name', config('app.name')),
                'address' => \App\Models\Option::get('company_address', ''),
                'city' => \App\Models\Option::get('company_city', ''),
                'state' => \App\Models\Option::get('company_state', ''),
                'zip' => \App\Models\Option::get('company_zip', ''),
                'phone' => \App\Models\Option::get('company_phone', ''),
                'email' => \App\Models\Option::get('company_email', ''),
                'gst' => \App\Models\Option::get('company_gst', ''),
                'logo' => \App\Models\Option::get('company_logo', ''),
            ];
        }

        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('purchase::grn.pdf', compact('grn', 'settings', 'company'));
        $pdf->setPaper('a4', 'landscape');

        return $pdf->stream("GRN_{$grn->grn_number}.pdf");
    }

    /**
     * Delete GRN
     */
    public function destroy($id)
    {
        $this->authorize('purchase.grn.delete');
        
        $grn = GoodsReceiptNote::findOrFail($id);

        if ($grn->stock_updated) {
            if (request()->ajax()) {
                return response()->json(['success' => false, 'message' => 'Cannot delete GRN after stock update.'], 422);
            }
            return back()->with('error', 'Cannot delete GRN after stock update.');
        }

        $grn->items()->delete();
        $grn->delete();

        if (request()->ajax()) {
            return response()->json(['success' => true, 'message' => 'GRN deleted successfully.']);
        }

        return redirect()
            ->route('admin.purchase.grn.index')
            ->with('success', 'GRN deleted successfully.');
    }

    /**
     * Bulk delete GRNs
     */
    public function bulkDelete(Request $request)
    {
        $ids = $request->input('ids', []);

        if (empty($ids)) {
            return response()->json(['success' => false, 'message' => 'No items selected.'], 400);
        }

        // Check for stock updated GRNs
        $hasStockUpdated = GoodsReceiptNote::whereIn('id', $ids)->where('stock_updated', true)->exists();
        if ($hasStockUpdated) {
            return response()->json([
                'success' => false, 
                'message' => 'Cannot delete GRNs with updated stock.'
            ], 422);
        }

        GoodsReceiptNoteItem::whereIn('goods_receipt_note_id', $ids)->delete();
        GoodsReceiptNote::whereIn('id', $ids)->delete();

        return response()->json([
            'success' => true,
            'message' => count($ids) . ' GRN(s) deleted.'
        ]);
    }

    /**
     * Get PO items for GRN creation (AJAX)
     */
    public function getPOItems($poId)
    {
        $po = PurchaseOrder::with(['items.product.unit', 'items.product.tax', 'items.variation', 'items.unit', 'vendor'])
            ->findOrFail($poId);

        $items = $po->items->map(function($item) {
            $pendingQty = max(0, $item->qty - $item->received_qty);
            $product = $item->product;
            $variation = $item->variation;
            
            // Fallback: if Eloquent relationship returns null (SoftDeletes/scopes), try direct DB
            if (!$product && $item->product_id) {
                $rawProduct = DB::table('products')->where('id', $item->product_id)->first();
                if ($rawProduct) {
                    $rawUnit = $rawProduct->unit_id ? DB::table('units')->find($rawProduct->unit_id) : null;
                    $rawTax = $rawProduct->tax_id ? DB::table('taxes')->find($rawProduct->tax_id) : null;
                    
                    return [
                        'id' => $item->id,
                        'product_id' => $item->product_id,
                        'variation_id' => $item->variation_id,
                        'product' => [
                            'name' => $rawProduct->name,
                            'sku' => $rawProduct->sku ?? '',
                            'hsn_code' => $rawProduct->hsn_code ?? '',
                            'purchase_price' => $rawProduct->purchase_price ?? 0,
                            'sale_price' => $rawProduct->sale_price ?? 0,
                            'mrp' => $rawProduct->mrp ?? 0,
                            'tax_id' => $rawProduct->tax_id ?? null,
                            'tax_name' => $rawTax ? $rawTax->name : null,
                            'tax_rate' => $rawTax ? $rawTax->rate : 0,
                            'is_batch_managed' => (bool)($rawProduct->is_batch_managed ?? false),
                        ],
                        'variation' => $variation ? [
                            'id' => $variation->id,
                            'variation_name' => $variation->variation_name,
                            'sku' => $variation->sku,
                        ] : null,
                        'unit_id' => $item->unit_id ?? $rawProduct->unit_id,
                        'unit' => $item->unit ? [
                            'short_name' => $item->unit->short_name ?? $item->unit->name,
                            'name' => $item->unit->name,
                        ] : ($rawUnit ? [
                            'short_name' => $rawUnit->short_name ?? $rawUnit->name,
                            'name' => $rawUnit->name,
                        ] : null),
                        'ordered_qty' => $item->qty,
                        'received_qty' => $item->received_qty,
                        'pending_qty' => $pendingQty,
                        'rate' => $item->rate,
                    ];
                }
            }
            
            return [
                'id' => $item->id,
                'product_id' => $item->product_id,
                'variation_id' => $item->variation_id,
                'product' => $product ? [
                    'name' => $product->name,
                    'sku' => $product->sku,
                    'hsn_code' => $product->hsn_code,
                    'purchase_price' => $product->purchase_price,
                    'sale_price' => $product->sale_price,
                    'mrp' => $product->mrp,
                    'tax_id' => $product->tax_id,
                    'tax_name' => $product->tax ? $product->tax->name : null,
                    'tax_rate' => $product->tax ? $product->tax->rate : 0,
                    'is_batch_managed' => (bool)($product->is_batch_managed ?? false),
                ] : null,
                'variation' => $variation ? [
                    'id' => $variation->id,
                    'variation_name' => $variation->variation_name,
                    'sku' => $variation->sku,
                ] : null,
                'unit_id' => $item->unit_id,
                'unit' => $item->unit ? [
                    'short_name' => $item->unit->short_name ?? $item->unit->name,
                    'name' => $item->unit->name,
                ] : ($product && $product->unit ? [
                    'short_name' => $product->unit->short_name ?? $product->unit->name,
                    'name' => $product->unit->name,
                ] : null),
                'ordered_qty' => $item->qty,
                'received_qty' => $item->received_qty,
                'pending_qty' => $pendingQty,
                'rate' => $item->rate,
            ];
        })->filter(function($item) {
            return $item['pending_qty'] > 0;
        })->values();

        return response()->json([
            'success' => true,
            'po' => [
                'id' => $po->id,
                'po_number' => $po->po_number,
                'vendor_name' => $po->vendor->name ?? '-',
                'po_date' => $po->po_date->format('d M Y'),
            ],
            'items' => $items,
        ]);
    }

    /**
     * Get racks by warehouse (AJAX)
     */
    public function getRacks($warehouseId)
    {
        $racks = [];
        
        if (class_exists('\Modules\Inventory\Models\Rack')) {
            $racks = \Modules\Inventory\Models\Rack::where('warehouse_id', $warehouseId)
                ->where('is_active', true)
                ->orderBy('code')
                ->get(['id', 'code', 'name']);
        }

        return response()->json(['racks' => $racks]);
    }

    // ========================================================================
    // GSTR2 EXCEL EXPORT - matches PURCHASE_JAN_2026.xls format exactly
    // ========================================================================

    public function exportGstr2(Request $request)
    {
        $request->validate([
            'from_date' => 'required|date',
            'to_date'   => 'required|date|after_or_equal:from_date',
            'format'    => 'nullable|in:csv,xlsx,pdf',
        ]);

        $fromDate = $request->from_date;
        $toDate   = $request->to_date;
        $format   = $request->get('format', 'xlsx');

        if (!class_exists(\PhpOffice\PhpSpreadsheet\Spreadsheet::class)) {
            return back()->with('error', 'PhpSpreadsheet not installed. Run: composer require phpoffice/phpspreadsheet');
        }

        // ── Fetch approved GRNs ──
        $grns = GoodsReceiptNote::with(['vendor', 'items.product', 'items.variation', 'items.unit'])
            ->where('status', 'APPROVED')
            ->whereBetween('grn_date', [$fromDate, $toDate])
            ->orderBy('grn_date')
            ->orderBy('id')
            ->get();

        // ── Company config ──
        $companyName      = config('app.company_name', 'SARVAM SUPERMARKET');
        $companyAddress   = config('app.company_address', 'No.5, 5th MAIN ROAD, NAGPURA MAIN ROAD MAHALAKSHMIPURAM BANGALORE-560086');
        $companyGstin     = config('app.company_gstin', '29AURPM8556G3ZQ');
        $companyStateCode = substr($companyGstin, 0, 2);

        // ── Group data: each GRN = 1 invoice, items grouped by HSN ──
        $invoices = [];
        foreach ($grns as $grn) {
            $vendor = $grn->vendor;
            if (!$vendor) continue;

            $vendorGstin     = $vendor->gst_number ?? '';
            $vendorStateCode = strlen($vendorGstin) >= 2 ? substr($vendorGstin, 0, 2) : '';
            $isLocal         = ($vendorStateCode === $companyStateCode && !empty($vendorStateCode));

            $hsnGroups = [];
            foreach ($grn->items as $item) {
                $hsnCode = $item->product->hsn_code ?? 'NA';
                if (empty($hsnCode)) $hsnCode = 'NA';

                if (!isset($hsnGroups[$hsnCode])) {
                    $hsnGroups[$hsnCode] = [
                        'hsn_code' => $hsnCode, 'quantity' => 0, 'taxable_amount' => 0,
                        'sgst_rate' => 0, 'sgst_amount' => 0, 'cgst_rate' => 0, 'cgst_amount' => 0,
                        'igst_rate' => 0, 'igst_amount' => 0, 'total_gst' => 0,
                    ];
                }

                $qty             = (float) $item->accepted_qty;
                $rate            = (float) $item->rate;
                $discountPercent = (float) ($item->discount_percent ?? 0);
                $lineAmount      = $qty * $rate;
                $taxableAmount   = $lineAmount - ($lineAmount * $discountPercent / 100);
                $tax1Rate        = (float) ($item->tax_1_rate ?? 0);
                $tax2Rate        = (float) ($item->tax_2_rate ?? 0);

                if ($isLocal) {
                    $sgstRate = $tax1Rate; $cgstRate = $tax2Rate; $igstRate = 0;
                    $sgstAmt = round($taxableAmount * $sgstRate / 100, 2);
                    $cgstAmt = round($taxableAmount * $cgstRate / 100, 2);
                    $igstAmt = 0;
                } else {
                    $sgstRate = 0; $cgstRate = 0; $igstRate = $tax1Rate + $tax2Rate;
                    $sgstAmt = 0; $cgstAmt = 0;
                    $igstAmt = round($taxableAmount * $igstRate / 100, 2);
                }

                $hsnGroups[$hsnCode]['quantity']       += $qty;
                $hsnGroups[$hsnCode]['taxable_amount'] += $taxableAmount;
                $hsnGroups[$hsnCode]['sgst_rate']       = $sgstRate;
                $hsnGroups[$hsnCode]['sgst_amount']    += $sgstAmt;
                $hsnGroups[$hsnCode]['cgst_rate']       = $cgstRate;
                $hsnGroups[$hsnCode]['cgst_amount']    += $cgstAmt;
                $hsnGroups[$hsnCode]['igst_rate']       = $igstRate;
                $hsnGroups[$hsnCode]['igst_amount']    += $igstAmt;
                $hsnGroups[$hsnCode]['total_gst']      += ($sgstAmt + $cgstAmt + $igstAmt);
            }

            $invoiceTaxable  = array_sum(array_column($hsnGroups, 'taxable_amount'));
            $invoiceSgst     = array_sum(array_column($hsnGroups, 'sgst_amount'));
            $invoiceCgst     = array_sum(array_column($hsnGroups, 'cgst_amount'));
            $invoiceIgst     = array_sum(array_column($hsnGroups, 'igst_amount'));
            $invoiceTotalGst = $invoiceSgst + $invoiceCgst + $invoiceIgst;
            $invoiceValue    = round($invoiceTaxable + $invoiceTotalGst, 2);

            $invoices[] = [
                'vendor_name'  => strtoupper($vendor->name ?? ''),
                'vendor_gstin' => $vendorGstin,
                'invoice_date' => $grn->invoice_date ? $grn->invoice_date->format('d/m/Y') : $grn->grn_date->format('d/m/Y'),
                'invoice_no'   => $grn->invoice_number ?? $grn->grn_number,
                'invoice_value' => $invoiceValue,
                'supply_type'  => $isLocal ? 'Local' : 'Central',
                'invoice_type' => 'Inventory',
                'hsn_groups'   => array_values($hsnGroups),
                'total_qty'    => array_sum(array_column($hsnGroups, 'quantity')),
                'total_amount' => $invoiceTaxable,
                'total_taxable' => $invoiceTaxable,
                'total_sgst'   => $invoiceSgst,
                'total_cgst'   => $invoiceCgst,
                'total_igst'   => $invoiceIgst,
                'total_gst'    => $invoiceTotalGst,
            ];
        }

        // ══════════════════════════════════════════════════════════════
        // COMPUTE TOTALS (used by all formats)
        // ══════════════════════════════════════════════════════════════
        $b2bTotals = [
            'qty'     => array_sum(array_column($invoices, 'total_qty')),
            'amount'  => array_sum(array_column($invoices, 'total_amount')),
            'taxable' => array_sum(array_column($invoices, 'total_taxable')),
            'sgst'    => array_sum(array_column($invoices, 'total_sgst')),
            'cgst'    => array_sum(array_column($invoices, 'total_cgst')),
            'igst'    => array_sum(array_column($invoices, 'total_igst')),
            'gst'     => array_sum(array_column($invoices, 'total_gst')),
        ];

        $fileBase = 'GRN_GSTR2_' . date('Ymd', strtotime($fromDate)) . '_to_' . date('Ymd', strtotime($toDate));

        // ── CSV → skip Excel ──
        if ($format === 'csv') {
            return $this->gstr2DownloadCsv($invoices, $b2bTotals, $companyName, $companyAddress, $companyGstin, $fromDate, $toDate, $fileBase);
        }
        // ── PDF → skip Excel ──
        if ($format === 'pdf') {
            return $this->gstr2DownloadPdf($invoices, $b2bTotals, $companyName, $companyAddress, $companyGstin, $fromDate, $toDate, $fileBase);
        }

        // ══════════════════════════════════════════════════════════════
        // BUILD EXCEL — exact replica of PURCHASE_JAN_2026.xls
        // ══════════════════════════════════════════════════════════════

        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle('GST_DETAIL');

        // Default font: Calibri 11pt (same as original)
        $spreadsheet->getDefaultStyle()->getFont()->setName('Calibri')->setSize(11);

        // ── COLUMN WIDTHS (exact match from original .xls in character units) ──
        $widths = [
            'A' => 2.7,  'B' => 5.7,  'C' => 30.7, 'D' => 11.7, 'E' => 11.7,
            'F' => 11.7, 'G' => 11.7, 'H' => 11.7, 'I' => 11.7, 'J' => 8.7,
            'K' => 12.7, 'L' => 11.7, 'M' => 11.7, 'N' => 5.7,  'O' => 10.7,
            'P' => 5.7,  'Q' => 10.7, 'R' => 5.7,  'S' => 10.7, 'T' => 10.7,
            'U' => 10.7,
        ];
        foreach ($widths as $col => $w) {
            $sheet->getColumnDimension($col)->setWidth($w);
        }

        // ── Shorthand constants ──
        $hCenter = \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER;
        $hRight  = \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_RIGHT;
        $hLeft   = \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_LEFT;
        $vCenter = \PhpOffice\PhpSpreadsheet\Style\Alignment::VERTICAL_CENTER;
        $fillSolid = \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID;
        $borderThin = \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN;
        $numFmt = '#,##0.00';

        // ── ROW 1: Company Name (bold, centered, merged B1:U1) ──
        $sheet->setCellValue('B1', $companyName);
        $sheet->mergeCells('B1:U1');
        $sheet->getStyle('B1')->getFont()->setBold(true);
        $sheet->getStyle('B1')->getAlignment()->setHorizontal($hCenter);

        // ── ROW 2: Address (centered, merged B2:U2) ──
        $sheet->setCellValue('B2', $companyAddress);
        $sheet->mergeCells('B2:U2');
        $sheet->getStyle('B2')->getAlignment()->setHorizontal($hCenter);

        // ── ROW 3: GSTIN (centered, merged B3:U3) ──
        $sheet->setCellValue('B3', 'GSTIN : ' . $companyGstin);
        $sheet->mergeCells('B3:U3');
        $sheet->getStyle('B3')->getAlignment()->setHorizontal($hCenter);

        // ── ROW 4: Blank (merged B4:U4) ──
        $sheet->mergeCells('B4:U4');

        // ── ROW 5: Period title (bold, centered, merged B5:U5) ──
        $fromFmt = date('d/m/Y', strtotime($fromDate));
        $toFmt   = date('d/m/Y', strtotime($toDate));
        $sheet->setCellValue('B5', "GSTR2 DETAILS FOR THE PERIOD {$fromFmt} TO {$toFmt}");
        $sheet->mergeCells('B5:U5');
        $sheet->getStyle('B5')->getFont()->setBold(true);
        $sheet->getStyle('B5')->getAlignment()->setHorizontal($hCenter);

        // ══════════════════════════════════════════════════════════════
        // ROWS 6-7: Column Headers (2-row header with merged cells)
        // ══════════════════════════════════════════════════════════════

        // Row 6 text
        $sheet->setCellValue('B6', 'S.No');
        $sheet->setCellValue('C6', 'Desc');
        $sheet->setCellValue('D6', 'GSTIN');
        $sheet->setCellValue('E6', "Invoice\nDate");
        $sheet->setCellValue('F6', "Invoice\nNo.");
        $sheet->setCellValue('G6', "Invoice\nValue");
        $sheet->setCellValue('H6', "Local/\nCentral");
        $sheet->setCellValue('I6', "Invoice\nType");
        $sheet->setCellValue('J6', "HSN\nCode");
        $sheet->setCellValue('K6', 'Quantity');
        $sheet->setCellValue('L6', 'Amount');
        $sheet->setCellValue('M6', "Taxable\nAmount");
        $sheet->setCellValue('N6', 'SGST');
        $sheet->setCellValue('P6', 'CGST');
        $sheet->setCellValue('R6', 'IGST');
        $sheet->setCellValue('T6', 'Cess');
        $sheet->setCellValue('U6', 'Total GST');

        // Row 7 sub-headers
        $sheet->setCellValue('N7', '%age');
        $sheet->setCellValue('O7', 'Amount');
        $sheet->setCellValue('P7', '%age');
        $sheet->setCellValue('Q7', 'Amount');
        $sheet->setCellValue('R7', '%age');
        $sheet->setCellValue('S7', 'Amount');

        // Merge single-header columns across rows 6-7
        foreach (['B','C','D','E','F','G','H','I','J','K','L','M','T','U'] as $c) {
            $sheet->mergeCells("{$c}6:{$c}7");
        }
        // Merge tax group headers (row 6 only, spanning 2 cols)
        $sheet->mergeCells('N6:O6');
        $sheet->mergeCells('P6:Q6');
        $sheet->mergeCells('R6:S6');

        // Header style: green bg (#92D050), centered, thin borders, wrap text
        $sheet->getStyle('B6:U7')->applyFromArray([
            'alignment' => ['horizontal' => $hCenter, 'vertical' => $vCenter, 'wrapText' => true],
            'fill'      => ['fillType' => $fillSolid, 'startColor' => ['rgb' => '92D050']],
            'borders'   => ['allBorders' => ['borderStyle' => $borderThin]],
            'font'      => ['bold' => true, 'size' => 10],
        ]);
        $sheet->getRowDimension(6)->setRowHeight(30);
        $sheet->getRowDimension(7)->setRowHeight(18);

        // ══════════════════════════════════════════════════════════════
        // ROW 8: B2B / Other Taxable Invoices (summary)
        // ══════════════════════════════════════════════════════════════

        $b2bRow = 8;
        $sheet->setCellValue('C' . $b2bRow, ' B2B / Other Taxable Invoices');
        $sheet->getStyle('B' . $b2bRow . ':U' . $b2bRow)->getFont()->setBold(true);

        $this->gstr2FillNumericRow($sheet, $b2bRow, $b2bTotals['qty'], $b2bTotals['amount'],
            $b2bTotals['taxable'], 0, $b2bTotals['sgst'], 0, $b2bTotals['cgst'],
            0, $b2bTotals['igst'], 0, $b2bTotals['gst']);

        // Border for B2B row
        $sheet->getStyle("B{$b2bRow}:U{$b2bRow}")->getBorders()->getAllBorders()->setBorderStyle($borderThin);

        // ══════════════════════════════════════════════════════════════
        // DATA ROWS (starting row 9)
        // ══════════════════════════════════════════════════════════════

        $currentRow = $b2bRow + 1;
        $serialNo = 0;
        $dataStartRow = $currentRow;

        foreach ($invoices as $invoice) {
            $serialNo++;
            $isFirstHsn = true;

            foreach ($invoice['hsn_groups'] as $hsnIdx => $hsn) {
                if ($isFirstHsn) {
                    // First HSN row: show vendor info + invoice details
                    $sheet->setCellValue('B' . $currentRow, $serialNo . '.');
                    $sheet->getStyle('B' . $currentRow)->getAlignment()->setHorizontal($hRight);
                    $sheet->setCellValue('C' . $currentRow, $invoice['vendor_name']);
                    $sheet->setCellValue('D' . $currentRow, $invoice['vendor_gstin']);
                    $sheet->setCellValue('E' . $currentRow, $invoice['invoice_date']);
                    $sheet->setCellValue('F' . $currentRow, $invoice['invoice_no']);
                    $sheet->setCellValue('G' . $currentRow, $invoice['invoice_value']);
                    $sheet->getStyle('G' . $currentRow)->getNumberFormat()->setFormatCode($numFmt);
                    $sheet->setCellValue('H' . $currentRow, $invoice['supply_type']);
                    $sheet->setCellValue('I' . $currentRow, $invoice['invoice_type']);
                    $sheet->setCellValue('L' . $currentRow, round($hsn['taxable_amount'], 2));
                    $isFirstHsn = false;
                } else {
                    // Subsequent HSN rows: blank vendor/invoice, only HSN data
                    $sheet->setCellValue('L' . $currentRow, 0);
                }

                // HSN columns (always)
                $sheet->setCellValue('J' . $currentRow, $hsn['hsn_code']);
                $sheet->setCellValue('K' . $currentRow, $hsn['quantity']);
                $sheet->setCellValue('M' . $currentRow, round($hsn['taxable_amount'], 2));
                $sheet->setCellValue('N' . $currentRow, $hsn['sgst_rate']);
                $sheet->setCellValue('O' . $currentRow, round($hsn['sgst_amount'], 2));
                $sheet->setCellValue('P' . $currentRow, $hsn['cgst_rate']);
                $sheet->setCellValue('Q' . $currentRow, round($hsn['cgst_amount'], 2));
                $sheet->setCellValue('R' . $currentRow, $hsn['igst_rate']);
                $sheet->setCellValue('S' . $currentRow, round($hsn['igst_amount'], 2));
                $sheet->setCellValue('T' . $currentRow, 0);
                $sheet->setCellValue('U' . $currentRow, ($hsnIdx === 0) ? round($invoice['total_gst'], 2) : 0);

                // Format numeric columns
                $sheet->getStyle("K{$currentRow}:U{$currentRow}")->getNumberFormat()->setFormatCode($numFmt);
                $sheet->getStyle("K{$currentRow}:U{$currentRow}")->getAlignment()->setHorizontal($hRight);

                $currentRow++;
            }
        }

        // Border around all data rows
        $lastDataRow = $currentRow - 1;
        if ($lastDataRow >= $dataStartRow) {
            $sheet->getStyle("B{$dataStartRow}:U{$lastDataRow}")->getBorders()
                ->getAllBorders()->setBorderStyle($borderThin);
        }

        // ══════════════════════════════════════════════════════════════
        // FOOTER ROWS (ITC Reversal, Reverse Charges, Gross Total)
        // ══════════════════════════════════════════════════════════════

        $footerStart = $currentRow;

        // ITC Reversal
        $sheet->setCellValue('C' . $currentRow, ' ITC Reversal');
        $sheet->getStyle('C' . $currentRow)->getFont()->setBold(true);
        $this->gstr2FillNumericRow($sheet, $currentRow, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
        $currentRow++;

        // Tax Paid on Reverse Charges
        $sheet->setCellValue('C' . $currentRow, ' Tax Paid on Reverse Charges');
        $sheet->getStyle('C' . $currentRow)->getFont()->setBold(true);
        $this->gstr2FillNumericRow($sheet, $currentRow, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
        $currentRow++;

        // Tax Paid under Reverse Charge on Advance
        $sheet->setCellValue('C' . $currentRow, ' Tax Paid under Reverse Charge on Advance');
        $sheet->getStyle('C' . $currentRow)->getFont()->setBold(true);
        $this->gstr2FillNumericRow($sheet, $currentRow, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
        $currentRow++;

        // Gross Total (bold, with totals)
        $sheet->setCellValue('C' . $currentRow, ' Gross Total');
        $sheet->getStyle("B{$currentRow}:U{$currentRow}")->getFont()->setBold(true);
        $this->gstr2FillNumericRow($sheet, $currentRow, $b2bTotals['qty'], $b2bTotals['amount'],
            $b2bTotals['taxable'], 0, $b2bTotals['sgst'], 0, $b2bTotals['cgst'],
            0, $b2bTotals['igst'], 0, $b2bTotals['gst']);

        // Green bg for Gross Total row
        $sheet->getStyle("B{$currentRow}:U{$currentRow}")->getFill()
            ->setFillType($fillSolid)->getStartColor()->setRGB('C6EFCE');

        // Border around footer rows
        $sheet->getStyle("B{$footerStart}:U{$currentRow}")->getBorders()
            ->getAllBorders()->setBorderStyle($borderThin);

        // ── Freeze pane & print settings ──
        $sheet->freezePane('A8');
        $sheet->getPageSetup()
            ->setOrientation(\PhpOffice\PhpSpreadsheet\Worksheet\PageSetup::ORIENTATION_LANDSCAPE)
            ->setPaperSize(\PhpOffice\PhpSpreadsheet\Worksheet\PageSetup::PAPERSIZE_A4)
            ->setFitToWidth(1)
            ->setFitToHeight(0);

        // ── Download XLSX ──
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);

        return response()->streamDownload(function () use ($writer) {
            $writer->save('php://output');
        }, "{$fileBase}.xlsx", [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'Cache-Control' => 'max-age=0',
        ]);
    }

    /**
     * Helper: fill numeric columns K-U in a summary row
     */
    private function gstr2FillNumericRow($sheet, $row, $qty, $amount, $taxable, $sgstRate, $sgstAmt, $cgstRate, $cgstAmt, $igstRate, $igstAmt, $cess, $totalGst)
    {
        $sheet->setCellValue('K' . $row, round($qty, 2));
        $sheet->setCellValue('L' . $row, round($amount, 2));
        $sheet->setCellValue('M' . $row, round($taxable, 2));
        $sheet->setCellValue('N' . $row, round($sgstRate, 2));
        $sheet->setCellValue('O' . $row, round($sgstAmt, 2));
        $sheet->setCellValue('P' . $row, round($cgstRate, 2));
        $sheet->setCellValue('Q' . $row, round($cgstAmt, 2));
        $sheet->setCellValue('R' . $row, round($igstRate, 2));
        $sheet->setCellValue('S' . $row, round($igstAmt, 2));
        $sheet->setCellValue('T' . $row, round($cess, 2));
        $sheet->setCellValue('U' . $row, round($totalGst, 2));
        $sheet->getStyle("K{$row}:U{$row}")->getNumberFormat()->setFormatCode('#,##0.00');
        $sheet->getStyle("K{$row}:U{$row}")->getAlignment()
            ->setHorizontal(\PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_RIGHT);
    }

    /**
     * CSV Export — GSTR2 format
     */
    private function gstr2DownloadCsv($invoices, $b2bTotals, $companyName, $companyAddress, $companyGstin, $fromDate, $toDate, $fileBase)
    {
        $fromFmt = date('d/m/Y', strtotime($fromDate));
        $toFmt   = date('d/m/Y', strtotime($toDate));

        return response()->streamDownload(function () use ($invoices, $b2bTotals, $companyName, $companyAddress, $companyGstin, $fromFmt, $toFmt) {
            $f = fopen('php://output', 'w');

            fputcsv($f, [$companyName]);
            fputcsv($f, [$companyAddress]);
            fputcsv($f, ['GSTIN : ' . $companyGstin]);
            fputcsv($f, []);
            fputcsv($f, ["GSTR2 DETAILS FOR THE PERIOD {$fromFmt} TO {$toFmt}"]);

            fputcsv($f, ['S.No', 'Desc', 'GSTIN', 'Invoice Date', 'Invoice No.', 'Invoice Value',
                'Local/Central', 'Invoice Type', 'HSN Code', 'Quantity', 'Amount', 'Taxable Amount',
                'SGST %', 'SGST Amount', 'CGST %', 'CGST Amount', 'IGST %', 'IGST Amount', 'Cess', 'Total GST']);

            fputcsv($f, ['', 'B2B / Other Taxable Invoices', '', '', '', '', '', '', '',
                number_format($b2bTotals['qty'], 2, '.', ''), number_format($b2bTotals['amount'], 2, '.', ''),
                number_format($b2bTotals['taxable'], 2, '.', ''), '0.00', number_format($b2bTotals['sgst'], 2, '.', ''),
                '0.00', number_format($b2bTotals['cgst'], 2, '.', ''), '0.00', number_format($b2bTotals['igst'], 2, '.', ''),
                '0.00', number_format($b2bTotals['gst'], 2, '.', '')]);

            $sn = 0;
            foreach ($invoices as $inv) {
                $sn++;
                $first = true;
                foreach ($inv['hsn_groups'] as $hi => $hsn) {
                    $row = $first
                        ? [$sn . '.', $inv['vendor_name'], $inv['vendor_gstin'], $inv['invoice_date'], $inv['invoice_no'],
                           number_format($inv['invoice_value'], 2, '.', ''), $inv['supply_type'], $inv['invoice_type']]
                        : ['', '', '', '', '', '', '', ''];
                    $first = false;

                    $row[] = $hsn['hsn_code'];
                    $row[] = number_format($hsn['quantity'], 2, '.', '');
                    $row[] = number_format($hsn['taxable_amount'], 2, '.', '');
                    $row[] = number_format($hsn['taxable_amount'], 2, '.', '');
                    $row[] = number_format($hsn['sgst_rate'], 2, '.', '');
                    $row[] = number_format($hsn['sgst_amount'], 2, '.', '');
                    $row[] = number_format($hsn['cgst_rate'], 2, '.', '');
                    $row[] = number_format($hsn['cgst_amount'], 2, '.', '');
                    $row[] = number_format($hsn['igst_rate'], 2, '.', '');
                    $row[] = number_format($hsn['igst_amount'], 2, '.', '');
                    $row[] = '0.00';
                    $row[] = ($hi === 0) ? number_format($inv['total_gst'], 2, '.', '') : '0.00';
                    fputcsv($f, $row);
                }
            }

            $zeroRow = ['', '', '', '', '', '', '', '', '', '0.00', '0.00', '0.00', '0.00', '0.00', '0.00', '0.00', '0.00', '0.00', '0.00', '0.00'];
            foreach (['ITC Reversal', 'Tax Paid on Reverse Charges', 'Tax Paid under Reverse Charge on Advance'] as $lbl) {
                $zr = $zeroRow;
                $zr[1] = $lbl;
                fputcsv($f, $zr);
            }

            fputcsv($f, ['', 'Gross Total', '', '', '', '', '', '', '',
                number_format($b2bTotals['qty'], 2, '.', ''), number_format($b2bTotals['amount'], 2, '.', ''),
                number_format($b2bTotals['taxable'], 2, '.', ''), '0.00', number_format($b2bTotals['sgst'], 2, '.', ''),
                '0.00', number_format($b2bTotals['cgst'], 2, '.', ''), '0.00', number_format($b2bTotals['igst'], 2, '.', ''),
                '0.00', number_format($b2bTotals['gst'], 2, '.', '')]);

            fclose($f);
        }, "{$fileBase}.csv", [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$fileBase}.csv\"",
        ]);
    }

    /**
     * PDF Export — GSTR2 format (landscape A4)
     */
    private function gstr2DownloadPdf($invoices, $b2bTotals, $companyName, $companyAddress, $companyGstin, $fromDate, $toDate, $fileBase)
    {
        $fromFmt = date('d/m/Y', strtotime($fromDate));
        $toFmt   = date('d/m/Y', strtotime($toDate));
        $nf = function($v) { return number_format($v, 2); };

        $html = '<!DOCTYPE html><html><head><meta charset="utf-8">
        <style>
            @page { size: A4 landscape; margin: 10mm; }
            body { font-family: Arial, sans-serif; font-size: 8pt; margin: 0; }
            .hdr { text-align: center; margin-bottom: 8px; }
            .hdr h2 { margin: 0; font-size: 14pt; }
            .hdr p { margin: 2px 0; font-size: 9pt; }
            .hdr .per { font-weight: bold; font-size: 10pt; margin-top: 6px; }
            table { width: 100%; border-collapse: collapse; font-size: 7.5pt; }
            th, td { border: 1px solid #333; padding: 3px 4px; }
            th { background: #92D050; font-weight: bold; text-align: center; font-size: 7pt; }
            td { text-align: right; }
            td.l { text-align: left; }
            td.c { text-align: center; }
            .b { font-weight: bold; }
            .gr td { font-weight: bold; background: #C6EFCE; }
        </style></head><body>';

        $html .= '<div class="hdr"><h2>' . e($companyName) . '</h2><p>' . e($companyAddress) . '</p>';
        $html .= '<p>GSTIN : ' . e($companyGstin) . '</p>';
        $html .= '<p class="per">GSTR2 DETAILS FOR THE PERIOD ' . $fromFmt . ' TO ' . $toFmt . '</p></div>';

        $html .= '<table><thead><tr>';
        $html .= '<th rowspan="2">S.No</th><th rowspan="2">Desc</th><th rowspan="2">GSTIN</th>';
        $html .= '<th rowspan="2">Invoice<br>Date</th><th rowspan="2">Invoice<br>No.</th>';
        $html .= '<th rowspan="2">Invoice<br>Value</th><th rowspan="2">Local/<br>Central</th>';
        $html .= '<th rowspan="2">Invoice<br>Type</th><th rowspan="2">HSN<br>Code</th>';
        $html .= '<th rowspan="2">Qty</th><th rowspan="2">Amount</th><th rowspan="2">Taxable<br>Amt</th>';
        $html .= '<th colspan="2">SGST</th><th colspan="2">CGST</th><th colspan="2">IGST</th>';
        $html .= '<th rowspan="2">Cess</th><th rowspan="2">Total<br>GST</th>';
        $html .= '</tr><tr><th>%</th><th>Amt</th><th>%</th><th>Amt</th><th>%</th><th>Amt</th></tr></thead><tbody>';

        // B2B summary
        $html .= '<tr class="b"><td></td><td class="l b">B2B / Other Taxable Invoices</td>' . str_repeat('<td></td>', 7);
        $html .= '<td>' . $nf($b2bTotals['qty']) . '</td><td>' . $nf($b2bTotals['amount']) . '</td><td>' . $nf($b2bTotals['taxable']) . '</td>';
        $html .= '<td>0.00</td><td>' . $nf($b2bTotals['sgst']) . '</td><td>0.00</td><td>' . $nf($b2bTotals['cgst']) . '</td>';
        $html .= '<td>0.00</td><td>' . $nf($b2bTotals['igst']) . '</td><td>0.00</td><td>' . $nf($b2bTotals['gst']) . '</td></tr>';

        // Data rows
        $sn = 0;
        foreach ($invoices as $inv) {
            $sn++;
            $first = true;
            foreach ($inv['hsn_groups'] as $hi => $hsn) {
                $html .= '<tr>';
                if ($first) {
                    $html .= '<td class="c">' . $sn . '.</td><td class="l">' . e($inv['vendor_name']) . '</td>';
                    $html .= '<td class="l">' . e($inv['vendor_gstin']) . '</td><td class="c">' . $inv['invoice_date'] . '</td>';
                    $html .= '<td class="c">' . e($inv['invoice_no']) . '</td><td>' . $nf($inv['invoice_value']) . '</td>';
                    $html .= '<td class="c">' . $inv['supply_type'] . '</td><td class="c">' . $inv['invoice_type'] . '</td>';
                    $first = false;
                } else {
                    $html .= str_repeat('<td></td>', 8);
                }
                $html .= '<td class="c">' . e($hsn['hsn_code']) . '</td>';
                $html .= '<td>' . $nf($hsn['quantity']) . '</td><td>' . $nf($hsn['taxable_amount']) . '</td><td>' . $nf($hsn['taxable_amount']) . '</td>';
                $html .= '<td>' . $nf($hsn['sgst_rate']) . '</td><td>' . $nf($hsn['sgst_amount']) . '</td>';
                $html .= '<td>' . $nf($hsn['cgst_rate']) . '</td><td>' . $nf($hsn['cgst_amount']) . '</td>';
                $html .= '<td>' . $nf($hsn['igst_rate']) . '</td><td>' . $nf($hsn['igst_amount']) . '</td>';
                $html .= '<td>0.00</td><td>' . (($hi === 0) ? $nf($inv['total_gst']) : '0.00') . '</td></tr>';
            }
        }

        // Footer
        foreach (['ITC Reversal', 'Tax Paid on Reverse Charges', 'Tax Paid under Reverse Charge on Advance'] as $lbl) {
            $html .= '<tr class="b"><td></td><td class="l b">' . $lbl . '</td>' . str_repeat('<td></td>', 7) . str_repeat('<td>0.00</td>', 11) . '</tr>';
        }

        // Gross Total
        $html .= '<tr class="gr"><td></td><td class="l b">Gross Total</td>' . str_repeat('<td></td>', 7);
        $html .= '<td>' . $nf($b2bTotals['qty']) . '</td><td>' . $nf($b2bTotals['amount']) . '</td><td>' . $nf($b2bTotals['taxable']) . '</td>';
        $html .= '<td>0.00</td><td>' . $nf($b2bTotals['sgst']) . '</td><td>0.00</td><td>' . $nf($b2bTotals['cgst']) . '</td>';
        $html .= '<td>0.00</td><td>' . $nf($b2bTotals['igst']) . '</td><td>0.00</td><td>' . $nf($b2bTotals['gst']) . '</td></tr>';
        $html .= '</tbody></table></body></html>';

        // Try DOMPDF → Mpdf → HTML fallback
        if (class_exists(\Dompdf\Dompdf::class)) {
            $dompdf = new \Dompdf\Dompdf();
            $dompdf->setPaper('A4', 'landscape');
            $dompdf->loadHtml($html);
            $dompdf->render();
            return response($dompdf->output(), 200, [
                'Content-Type' => 'application/pdf',
                'Content-Disposition' => "attachment; filename=\"{$fileBase}.pdf\"",
            ]);
        }
        if (class_exists(\Mpdf\Mpdf::class)) {
            $mpdf = new \Mpdf\Mpdf(['orientation' => 'L', 'format' => 'A4', 'margin_left' => 5, 'margin_right' => 5, 'margin_top' => 5, 'margin_bottom' => 5]);
            $mpdf->WriteHTML($html);
            return response($mpdf->Output('', 'S'), 200, [
                'Content-Type' => 'application/pdf',
                'Content-Disposition' => "attachment; filename=\"{$fileBase}.pdf\"",
            ]);
        }

        // Fallback: HTML (user can print to PDF)
        return response($html, 200, [
            'Content-Type' => 'text/html',
            'Content-Disposition' => "attachment; filename=\"{$fileBase}.html\"",
        ]);
    }
}