<?php

namespace Modules\Purchase\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class VendorItem extends Model
{
    protected $fillable = [
        'vendor_id',
        'product_id',
        'variation_id',
        'unit_id',
        'vendor_sku',
        'purchase_price',
        'moq',
        'lead_days',
        'is_preferred',
        'is_active',
        'notes',
        'created_by',
    ];

    protected $casts = [
        'purchase_price' => 'decimal:2',
        'moq' => 'decimal:3',
        'is_preferred' => 'boolean',
        'is_active' => 'boolean',
    ];

    // ==================== RELATIONSHIPS ====================

    public function vendor(): BelongsTo
    {
        return $this->belongsTo(Vendor::class);
    }

    public function product(): BelongsTo
    {
        return $this->belongsTo(\Modules\Inventory\Models\Product::class);
    }

    public function variation(): BelongsTo
    {
        return $this->belongsTo(\Modules\Inventory\Models\ProductVariation::class);
    }

    public function unit(): BelongsTo
    {
        return $this->belongsTo(\Modules\Inventory\Models\Unit::class);
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(\App\Models\Admin::class, 'created_by');
    }

    // ==================== SCOPES ====================

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeForVendor($query, $vendorId)
    {
        return $query->where('vendor_id', $vendorId);
    }

    // ==================== ACCESSORS ====================

    public function getDisplayNameAttribute(): string
    {
        $name = $this->product->name ?? 'Unknown Product';
        if ($this->variation) {
            $name .= ' » ' . ($this->variation->variation_name ?? $this->variation->sku ?? '');
        }
        return $name;
    }
}
