<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Consolidated Task Module Migration
     * 
     * Tables Created:
     * 1. module_task_statuses        - Status master (Backlog, To Do, In Progress, etc.)
     * 2. module_tasks                - Main tasks table
     * 3. module_task_assignees       - Task ↔ Admin assignees (pivot)
     * 4. module_task_followers       - Task ↔ Admin followers (pivot)
     * 5. module_task_attachments     - File attachments
     * 6. module_task_comments        - Task comments
     * 7. module_task_items           - Products/Services in tasks
     * 8. module_task_checklists      - Checklist items
     * 9. module_task_time_logs       - Time tracking with GPS
     * 10. module_task_time_log_pauses - Pause/Resume events
     */
    public function up(): void
    {
        // ╔════════════════════════════════════════════════════════════════╗
        // ║  1. TASK STATUSES (Master Table)                               ║
        // ╚════════════════════════════════════════════════════════════════╝
        if (!Schema::hasTable('module_task_statuses')) {
            Schema::create('module_task_statuses', function (Blueprint $table) {
                $table->id();
                $table->string('label', 100);
                $table->string('color', 20)->default('#3b82f6');
                $table->integer('sort_order')->default(0);
                $table->boolean('show_by_default')->default(true);
                $table->boolean('is_completed')->default(false);
                $table->boolean('is_active')->default(true);
                
                $table->index('is_active');
                $table->index('is_completed');
            });

            // Insert default statuses
            DB::table('module_task_statuses')->insert([
                ['label' => 'Backlog', 'color' => '#9ca3af', 'sort_order' => 1, 'show_by_default' => true, 'is_completed' => false, 'is_active' => true],
                ['label' => 'To Do', 'color' => '#6b7280', 'sort_order' => 2, 'show_by_default' => true, 'is_completed' => false, 'is_active' => true],
                ['label' => 'In Progress', 'color' => '#3b82f6', 'sort_order' => 3, 'show_by_default' => true, 'is_completed' => false, 'is_active' => true],
                ['label' => 'Review', 'color' => '#f59e0b', 'sort_order' => 4, 'show_by_default' => true, 'is_completed' => false, 'is_active' => true],
                ['label' => 'Done', 'color' => '#10b981', 'sort_order' => 5, 'show_by_default' => true, 'is_completed' => true, 'is_active' => true],
                ['label' => 'Blocked', 'color' => '#ef4444', 'sort_order' => 6, 'show_by_default' => false, 'is_completed' => false, 'is_active' => true],
            ]);
        }

        // ╔════════════════════════════════════════════════════════════════╗
        // ║  2. TASKS (Main Table)                                         ║
        // ╚════════════════════════════════════════════════════════════════╝
        if (!Schema::hasTable('module_tasks')) {
            Schema::create('module_tasks', function (Blueprint $table) {
                $table->id();
                
                // Task Type & Related Reference
                $table->enum('task_type', ['general', 'project', 'service', 'contract', 'support'])->default('general');
                $table->unsignedBigInteger('project_id')->nullable();
                $table->unsignedBigInteger('service_id')->nullable();
                $table->unsignedBigInteger('contract_id')->nullable();
                $table->unsignedBigInteger('support_ticket_id')->nullable();
                $table->string('related_name', 255)->nullable()->comment('Stored name for reference');
                
                // Task Details
                $table->unsignedBigInteger('milestone_id')->nullable();
                $table->string('title', 191);
                $table->text('description')->nullable();
                $table->unsignedBigInteger('status_id')->nullable();
                $table->enum('priority', ['low', 'medium', 'high', 'critical'])->default('medium');
                
                // Dates
                $table->date('planned_start_date')->nullable();
                $table->date('due_date')->nullable();
                $table->timestamp('completed_at')->nullable();
                
                // Billing
                $table->decimal('estimated_hours', 8, 2)->nullable();
                $table->decimal('hourly_rate', 12, 2)->nullable();
                $table->boolean('is_billable')->default(true);
                
                // Recurring
                $table->boolean('is_recurring')->default(false);
                $table->string('recurring_frequency', 50)->nullable();
                $table->date('recurring_until')->nullable();
                
                // Hierarchy
                $table->unsignedBigInteger('parent_task_id')->nullable();
                $table->integer('sort_order')->default(0);
                
                // Audit
                $table->unsignedBigInteger('created_by')->nullable();
                $table->unsignedBigInteger('updated_by')->nullable();
                $table->timestamps();
                $table->softDeletes();
            });

            // Add indexes
            Schema::table('module_tasks', function (Blueprint $table) {
                $table->index('task_type');
                $table->index('project_id');
                $table->index('service_id');
                $table->index('contract_id');
                $table->index('milestone_id');
                $table->index('status_id');
                $table->index('priority');
                $table->index('due_date');
                $table->index('parent_task_id');
                $table->index('created_by');
            });
        }

        // ╔════════════════════════════════════════════════════════════════╗
        // ║  3. TASK ASSIGNEES (Pivot Table)                               ║
        // ╚════════════════════════════════════════════════════════════════╝
        if (!Schema::hasTable('module_task_assignees')) {
            Schema::create('module_task_assignees', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('task_id');
                $table->unsignedBigInteger('admin_id');
                $table->timestamps();
                
                $table->unique(['task_id', 'admin_id']);
                $table->index('task_id');
                $table->index('admin_id');
            });
        }

        // ╔════════════════════════════════════════════════════════════════╗
        // ║  4. TASK FOLLOWERS (Pivot Table)                               ║
        // ╚════════════════════════════════════════════════════════════════╝
        if (!Schema::hasTable('module_task_followers')) {
            Schema::create('module_task_followers', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('task_id');
                $table->unsignedBigInteger('admin_id');
                $table->timestamps();
                
                $table->unique(['task_id', 'admin_id']);
                $table->index('task_id');
                $table->index('admin_id');
            });
        }

        // ╔════════════════════════════════════════════════════════════════╗
        // ║  5. TASK ATTACHMENTS                                           ║
        // ╚════════════════════════════════════════════════════════════════╝
        if (!Schema::hasTable('module_task_attachments')) {
            Schema::create('module_task_attachments', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('task_id');
                $table->string('filename', 255);
                $table->string('file_path', 500);
                $table->unsignedBigInteger('file_size')->nullable();
                $table->unsignedBigInteger('uploaded_by')->nullable();
                $table->timestamp('created_at')->nullable();
                
                $table->index('task_id');
                $table->index('uploaded_by');
            });
        }

        // ╔════════════════════════════════════════════════════════════════╗
        // ║  6. TASK COMMENTS                                              ║
        // ╚════════════════════════════════════════════════════════════════╝
        if (!Schema::hasTable('module_task_comments')) {
            Schema::create('module_task_comments', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('task_id');
                $table->unsignedBigInteger('admin_id')->nullable();
                $table->text('comment');
                $table->boolean('portal_visible')->default(false);
                $table->timestamps();
                
                $table->index('task_id');
                $table->index('admin_id');
            });
        }

        // ╔════════════════════════════════════════════════════════════════╗
        // ║  7. TASK ITEMS (Products/Services for Billing)                 ║
        // ╚════════════════════════════════════════════════════════════════╝
        if (!Schema::hasTable('module_task_items')) {
            Schema::create('module_task_items', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('task_id');
                $table->unsignedBigInteger('product_id')->nullable();
                $table->decimal('quantity', 15, 2)->default(1);
                $table->decimal('unit_price', 15, 2)->default(0);
                $table->decimal('total_price', 15, 2)->default(0);
                $table->json('tax_ids')->nullable();
                $table->decimal('tax_amount', 15, 2)->default(0);
                $table->text('notes')->nullable();
                $table->timestamps();
                
                $table->index('task_id');
                $table->index('product_id');
            });
        }

        // ╔════════════════════════════════════════════════════════════════╗
        // ║  8. TASK CHECKLISTS                                            ║
        // ╚════════════════════════════════════════════════════════════════╝
        if (!Schema::hasTable('module_task_checklists')) {
            Schema::create('module_task_checklists', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('task_id');
                $table->string('title', 255);
                $table->boolean('is_completed')->default(false);
                $table->integer('sort_order')->default(0);
                $table->timestamps();
                
                $table->index('task_id');
            });
        }

        // ╔════════════════════════════════════════════════════════════════╗
        // ║  9. TASK TIME LOGS (Time Tracking with GPS)                    ║
        // ╚════════════════════════════════════════════════════════════════╝
        if (!Schema::hasTable('module_task_time_logs')) {
            Schema::create('module_task_time_logs', function (Blueprint $table) {
                $table->id();
                
                // Core Fields
                $table->unsignedBigInteger('task_id');
                $table->unsignedBigInteger('admin_id');
                $table->timestamp('start_time');
                $table->timestamp('end_time')->nullable();
                $table->integer('duration_minutes')->default(0);
                $table->text('description')->nullable();
                
                // Billing Fields
                $table->boolean('is_billable')->default(true);
                $table->boolean('is_billed')->default(false);
                $table->boolean('is_running')->default(false);
                $table->boolean('is_paused')->default(false);
                $table->decimal('hourly_rate', 12, 2)->nullable();
                $table->unsignedBigInteger('invoice_id')->nullable();
                
                // Pause Tracking
                $table->integer('total_pause_seconds')->default(0);
                
                // Start GPS & Location
                $table->string('ip_address', 45)->nullable()->comment('IPv4 or IPv6 address');
                $table->decimal('latitude', 10, 8)->nullable()->comment('GPS latitude');
                $table->decimal('longitude', 11, 8)->nullable()->comment('GPS longitude');
                $table->decimal('location_accuracy', 8, 2)->nullable()->comment('GPS accuracy in meters');
                $table->string('location_address', 500)->nullable()->comment('Reverse geocoded address');
                
                // End GPS & Location
                $table->string('end_ip_address', 45)->nullable();
                $table->decimal('end_latitude', 10, 8)->nullable();
                $table->decimal('end_longitude', 11, 8)->nullable();
                $table->decimal('end_location_accuracy', 8, 2)->nullable();
                $table->string('end_location_address', 500)->nullable();
                
                $table->timestamps();
            });

            // Add indexes
            Schema::table('module_task_time_logs', function (Blueprint $table) {
                // Core indexes
                $table->index('task_id');
                $table->index('admin_id');
                $table->index('is_running');
                $table->index('is_paused');
                $table->index('is_billable');
                $table->index('is_billed');
                $table->index('invoice_id');
                
                // Composite indexes
                $table->index(['task_id', 'admin_id'], 'idx_task_admin');
                $table->index(['admin_id', 'start_time'], 'idx_admin_starttime');
                $table->index(['is_running', 'admin_id'], 'idx_running_admin');
                
                // GPS search indexes
                $table->index(['latitude', 'longitude'], 'idx_gps_coordinates');
            });
        }

        // ╔════════════════════════════════════════════════════════════════╗
        // ║  10. TASK TIME LOG PAUSES (Pause/Resume Events)                ║
        // ╚════════════════════════════════════════════════════════════════╝
        if (!Schema::hasTable('module_task_time_log_pauses')) {
            Schema::create('module_task_time_log_pauses', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('time_log_id')->comment('Parent time log');
                
                // Pause Event
                $table->timestamp('paused_at')->comment('When timer was paused');
                $table->decimal('pause_latitude', 10, 8)->nullable()->comment('GPS lat at pause');
                $table->decimal('pause_longitude', 11, 8)->nullable()->comment('GPS lng at pause');
                $table->decimal('pause_accuracy', 8, 2)->nullable()->comment('GPS accuracy at pause');
                $table->string('pause_ip_address', 45)->nullable();
                
                // Resume Event (null if not yet resumed)
                $table->timestamp('resumed_at')->nullable()->comment('When timer was resumed');
                $table->decimal('resume_latitude', 10, 8)->nullable()->comment('GPS lat at resume');
                $table->decimal('resume_longitude', 11, 8)->nullable()->comment('GPS lng at resume');
                $table->decimal('resume_accuracy', 8, 2)->nullable()->comment('GPS accuracy at resume');
                $table->string('resume_ip_address', 45)->nullable();
                
                // Calculated pause duration
                $table->integer('pause_duration_seconds')->default(0)->comment('Duration of this pause');
                
                // Optional note
                $table->string('pause_reason')->nullable()->comment('Why paused (optional)');
                
                $table->timestamps();
                
                // Indexes
                $table->index('time_log_id');
                $table->index(['time_log_id', 'resumed_at']);
            });
        }
    }

    /**
     * Reverse the migrations (drop all tables in reverse order)
     */
    public function down(): void
    {
        Schema::dropIfExists('module_task_time_log_pauses');
        Schema::dropIfExists('module_task_time_logs');
        Schema::dropIfExists('module_task_checklists');
        Schema::dropIfExists('module_task_items');
        Schema::dropIfExists('module_task_comments');
        Schema::dropIfExists('module_task_attachments');
        Schema::dropIfExists('module_task_followers');
        Schema::dropIfExists('module_task_assignees');
        Schema::dropIfExists('module_tasks');
        Schema::dropIfExists('module_task_statuses');
    }
};