<?php

namespace Modules\Tasks\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Modules\Tasks\Models\Task;
use Modules\Tasks\Models\TaskStatus;
use Modules\Projects\Models\Project;
use Modules\Tasks\Models\TimeLog;
use Modules\Tasks\Models\TaskComment;
use Modules\Tasks\Models\TaskAttachment;
use Modules\Tasks\Models\TaskItem;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Exception;

class TaskController extends AdminController
{
    public function index()
    {
        $admin = $this->admin;
        
        // Base query with visibility filter
        $baseQuery = Task::visibleTo($admin);
        
        $stats = [
            'total' => (clone $baseQuery)->count(),
            'my_tasks' => Task::assignedTo($admin->id)->count(),
            'overdue' => (clone $baseQuery)->overdue()->count(),
            'completed' => (clone $baseQuery)->completed()->count(),
        ];
        
        $taskTypes = Task::TASK_TYPES;

        return $this->View('tasks::tasks.index', compact('stats', 'taskTypes'));
    }

    public function dataTable(Request $request): JsonResponse
    {
        try {
            $admin = $this->admin;
            $query = Task::with(['status', 'assignees', 'milestone'])
                ->visibleTo($admin); // Apply visibility filter

            // Search filter
            if ($search = $request->input('search')) {
                $query->search($search);
            }
            
            // Filter by task type
            if ($type = $request->input('task_type')) {
                $query->byType($type);
            }

            // Advanced filters
            if ($filters = $request->input('filters')) {
                $decoded = is_array($filters) ? $filters : json_decode($filters, true);
                foreach ($decoded ?? [] as $key => $value) {
                    if ($value !== '' && $value !== null) {
                        if ($key === 'status_id') {
                            $query->where('status_id', $value);
                        } elseif ($key === 'priority') {
                            $query->where('priority', $value);
                        } elseif ($key === 'project_id') {
                            $query->where('project_id', $value);
                        }
                    }
                }
            }

            // Sorting
            $query->orderBy(
                $request->input('sort', 'id'),
                $request->input('dir', 'desc')
            );

            // Pagination
            $data = $query->paginate($request->input('per_page', 15));

            // Map items with display fields and action URLs
            $items = collect($data->items())->map(function ($item) {
                // Add action URLs
                $item->_edit_url = route('admin.tasks.edit', $item->id);
                $item->_show_url = route('admin.tasks.show', $item->id);
                
                // Add display fields for frontend rendering
                $item->project_display = $item->project ? $item->project->title : '-';
                $item->status_display = $item->status ? $item->status->label : 'Unknown';
                $item->priority_display = ucfirst($item->priority);
                
                // Format due date
                $item->due_date_display = $item->due_date 
                    ? $item->due_date->format('M d, Y')
                    : '-';
                
                // Get first owner name
                $item->owner_display = $item->assignedTo 
                    ? $item->assignedTo->name 
                    : 'Unassigned';
                
                // Get owners count
                $item->owners_count = ($item->assigned_to ? 1 : 0);
                
                // Check if overdue (use accessor from model)
                $item->is_overdue = $item->is_overdue;
                
                return $item;
            });

            return response()->json([
                'data' => $items,
                'total' => $data->total(),
                'current_page' => $data->currentPage(),
                'last_page' => $data->lastPage(),
            ]);

        } catch (Exception $e) {
            Log::error('TaskController@dataTable Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to load tasks data',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
 * Get task statistics for AJAX refresh
 */
public function stats(): JsonResponse
{
    try {
        $stats = [
            'total' => Task::count(),
            'my_tasks' => Task::where('assigned_to', $this->admin->id)->count(),
            'overdue' => Task::overdue()->count(),
            'completed' => Task::completed()->count(),
        ];

        return response()->json($stats);

    } catch (Exception $e) {
        Log::error('TaskController@stats Error: ' . $e->getMessage());
        
        return response()->json([
            'success' => false,
            'message' => 'Failed to load stats'
        ], 500);
    }
}

    public function create(Request $request)
    {
        try {
            $statuses = TaskStatus::where('is_active', true)->orderBy('sort_order')->get();
            $admins = \App\Models\Admin::where('is_active', true)->orderBy('name')->get();
            
            // Task types
            $taskTypes = Task::TASK_TYPES;
            
            // Check which modules are available
            $modulesAvailable = [
                'project' => Schema::hasTable('projects'),
                'service' => Schema::hasTable('services'),
                'contract' => Schema::hasTable('contracts'),
                'support' => Schema::hasTable('support_tickets'),
            ];
            
            // Fetch products for items used section
            $products = collect([]);
            if (Schema::hasTable('products')) {
                $products = DB::table('products')->orderBy('name')->get();
            }
            
            // Fetch taxes for items
            $taxes = collect([]);
            if (Schema::hasTable('taxes')) {
                $taxes = DB::table('taxes')->orderBy('name')->get();
            }
            
            // Get pre-selected type and related ID from query parameter
            $selectedType = $request->query('type', 'general');
            $selectedRelatedId = $request->query('related_id');

            return $this->View('tasks::tasks.create', compact(
                'statuses', 'admins', 'taskTypes', 'modulesAvailable', 
                'products', 'taxes', 'selectedType', 'selectedRelatedId'
            ));

        } catch (Exception $e) {
            Log::error('TaskController@create Error: ' . $e->getMessage());
            return back()->with('error', 'Failed to load create form');
        }
    }

    /**
     * Get related items based on task type (AJAX)
     */
    public function getRelatedItems(Request $request): JsonResponse
    {
        $type = $request->input('type');
        $items = [];
        $available = false;
        $message = '';

        try {
            switch ($type) {
                case 'project':
                    if (Schema::hasTable('projects')) {
                        $available = true;
                        
                        $query = DB::table('projects');
                        
                        // Check which column exists for project name
                        $nameColumn = Schema::hasColumn('projects', 'title') ? 'title' : 'name';
                        
                        // Join with customers if account_id exists
                        if (Schema::hasTable('customers') && Schema::hasColumn('projects', 'account_id')) {
                            $query->leftJoin('customers', 'projects.account_id', '=', 'customers.id');
                            $query->select(
                                'projects.id',
                                DB::raw("CONCAT('#', projects.id, ' - ', COALESCE(projects.{$nameColumn}, 'N/A'), ' (', COALESCE(customers.company, customers.name, 'No Client'), ')') as name")
                            );
                        } else {
                            $query->select(
                                'projects.id',
                                DB::raw("CONCAT('#', projects.id, ' - ', COALESCE(projects.{$nameColumn}, 'N/A')) as name")
                            );
                        }
                        
                        // Check if deleted_at column exists
                        if (Schema::hasColumn('projects', 'deleted_at')) {
                            $query->whereNull('projects.deleted_at');
                        }
                        
                        $items = $query->orderBy('projects.id', 'desc')->get();
                        
                        if ($items->isEmpty()) {
                            $message = 'No projects found. Create a project first.';
                        }
                    } else {
                        $message = 'Projects module is not installed.';
                    }
                    break;

                case 'service':
                    if (Schema::hasTable('services')) {
                        $available = true;
                        
                        $query = DB::table('services');
                        
                        // Join with customers/clients table if exists
                        if (Schema::hasTable('customers') && Schema::hasColumn('services', 'client_id')) {
                            $query->leftJoin('customers', 'services.client_id', '=', 'customers.id');
                            $query->select(
                                'services.id',
                                DB::raw("CONCAT('#', services.id, ' - ', COALESCE(customers.company, customers.name, 'No Client'), ' - ', COALESCE(services.machine_name, 'N/A')) as name")
                            );
                        } else {
                            // No customer join, just show ID and machine name
                            $query->select(
                                'services.id',
                                DB::raw("CONCAT('#', services.id, ' - ', COALESCE(services.machine_name, 'N/A')) as name")
                            );
                        }
                        
                        // Check if deleted_at column exists
                        if (Schema::hasColumn('services', 'deleted_at')) {
                            $query->whereNull('services.deleted_at');
                        }
                        
                        $items = $query->orderBy('services.id', 'desc')->get();
                        
                        if ($items->isEmpty()) {
                            $message = 'No services found. Create a service first.';
                        }
                    } else {
                        $message = 'Services module is not installed.';
                    }
                    break;

                case 'contract':
                    if (Schema::hasTable('contracts')) {
                        $available = true;
                        
                        $query = DB::table('contracts');
                        
                        // Check which column exists for contract name
                        $nameColumn = 'subject';
                        if (Schema::hasColumn('contracts', 'subject')) {
                            $nameColumn = 'subject';
                        } elseif (Schema::hasColumn('contracts', 'title')) {
                            $nameColumn = 'title';
                        } elseif (Schema::hasColumn('contracts', 'name')) {
                            $nameColumn = 'name';
                        }
                        
                        // Join with customers if client_id exists
                        if (Schema::hasTable('customers') && Schema::hasColumn('contracts', 'client_id')) {
                            $query->leftJoin('customers', 'contracts.client_id', '=', 'customers.id');
                            $query->select(
                                'contracts.id',
                                DB::raw("CONCAT('#', contracts.id, ' - ', COALESCE(contracts.{$nameColumn}, 'N/A'), ' (', COALESCE(customers.company, customers.name, 'No Client'), ')') as name")
                            );
                        } else {
                            $query->select(
                                'contracts.id',
                                DB::raw("CONCAT('#', contracts.id, ' - ', COALESCE(contracts.{$nameColumn}, 'N/A')) as name")
                            );
                        }
                        
                        // Check if deleted_at column exists
                        if (Schema::hasColumn('contracts', 'deleted_at')) {
                            $query->whereNull('contracts.deleted_at');
                        }
                        
                        $items = $query->orderBy('contracts.id', 'desc')->get();
                        
                        if ($items->isEmpty()) {
                            $message = 'No contracts found. Create a contract first.';
                        }
                    } else {
                        $message = 'Contracts module is not installed.';
                    }
                    break;

                case 'support':
                    if (Schema::hasTable('support_tickets')) {
                        $available = true;
                        
                        $query = DB::table('support_tickets');
                        
                        // Build display name with ticket_key and subject
                        if (Schema::hasColumn('support_tickets', 'ticket_key')) {
                            $query->select(
                                'support_tickets.id',
                                DB::raw("CONCAT('[', COALESCE(support_tickets.ticket_key, support_tickets.id), '] ', COALESCE(support_tickets.subject, 'No Subject')) as name")
                            );
                        } else {
                            $query->select(
                                'support_tickets.id',
                                DB::raw("CONCAT('#', support_tickets.id, ' - ', COALESCE(support_tickets.subject, 'No Subject')) as name")
                            );
                        }
                        
                        // Check if deleted_at column exists
                        if (Schema::hasColumn('support_tickets', 'deleted_at')) {
                            $query->whereNull('support_tickets.deleted_at');
                        }
                        
                        $items = $query->orderBy('support_tickets.id', 'desc')->get();
                        
                        if ($items->isEmpty()) {
                            $message = 'No support tickets found.';
                        }
                    } else {
                        $message = 'Support module is not installed.';
                    }
                    break;

                case 'general':
                default:
                    $available = true;
                    $message = '';
                    break;
            }

            return response()->json([
                'success' => true,
                'available' => $available,
                'items' => $items,
                'message' => $message,
            ]);

        } catch (Exception $e) {
            Log::error('TaskController@getRelatedItems Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'available' => false,
                'items' => [],
                'message' => 'Error loading items',
            ]);
        }
    }

    public function store(Request $request)
    {
        // Build validation rules dynamically based on what tables exist
        $rules = [
            'task_type' => 'required|in:general,project,service,contract,support',
            'project_id' => 'nullable|integer',
            'service_id' => 'nullable|integer',
            'contract_id' => 'nullable|integer',
            'support_ticket_id' => 'nullable|integer',
            'related_name' => 'nullable|string|max:255',
            'title' => 'required|string|max:191',
            'description' => 'nullable|string',
            'priority' => 'required|in:low,medium,high,critical',
            'planned_start_date' => 'nullable|date',
            'due_date' => 'nullable|date',
            'estimated_hours' => 'nullable|numeric|min:0',
            'hourly_rate' => 'nullable|numeric|min:0',
            'is_billable' => 'boolean',
            'assignees' => 'nullable|array',
            'assignees.*' => 'exists:admins,id',
            'followers' => 'nullable|array',
            'followers.*' => 'exists:admins,id',
            'attachments' => 'nullable|array|max:3',
            'attachments.*' => 'file|max:10240|mimes:jpg,jpeg,png,gif,webp,pdf,doc,docx,xls,xlsx,csv,ppt,pptx,txt,zip,rar',
            'items' => 'nullable|array',
            'items.*.product_id' => 'nullable|exists:products,id',
            'items.*.quantity' => 'nullable|numeric|min:0.01',
            'items.*.unit_price' => 'nullable|numeric|min:0',
            'items.*.tax_ids' => 'nullable',
            'items.*.notes' => 'nullable|string|max:500',
        ];
        
        // Add status validation only if table exists
        if (Schema::hasTable('module_task_statuses')) {
            $rules['status_id'] = 'required|exists:module_task_statuses,id';
        } else {
            $rules['status_id'] = 'nullable|integer';
        }
        
        // Add milestone validation only if table exists
        if (Schema::hasTable('project_milestones')) {
            $rules['milestone_id'] = 'nullable|exists:project_milestones,id';
        } else {
            $rules['milestone_id'] = 'nullable|integer';
        }
        
        $validated = $request->validate($rules);

        DB::beginTransaction();
        try {
            $validated['created_by'] = $this->admin->id;
            $validated['updated_by'] = $this->admin->id;
            
            // Store related name for reference
            if ($validated['task_type'] !== 'general' && empty($validated['related_name'])) {
                $validated['related_name'] = $this->getRelatedName($validated);
            }
            
            $assignees = $validated['assignees'] ?? [];
            $followers = $validated['followers'] ?? [];
            $attachmentFiles = $request->file('attachments', []);
            $items = $validated['items'] ?? [];
            unset($validated['assignees'], $validated['followers'], $validated['attachments'], $validated['items']);
            
            $task = Task::create($validated);
            
            // Sync assignees
            if (!empty($assignees)) {
                $task->assignees()->sync($assignees);
            }
            
            // Sync followers (managers)
            if (!empty($followers)) {
                $task->followers()->sync($followers);
            }

            // Handle file uploads (max 3 files)
            if (!empty($attachmentFiles)) {
                $this->handleTaskAttachments($task, $attachmentFiles);
            }

            // Handle task items (products used)
            if (!empty($items)) {
                $this->handleTaskItems($task, $items);
            }

            DB::commit();

            return redirect()->route('admin.tasks.index')
                ->with('success', 'Task created successfully!');

        } catch (Exception $e) {
            DB::rollback();
            Log::error('TaskController@store Error: ' . $e->getMessage(), [
                'data' => $validated,
                'trace' => $e->getTraceAsString()
            ]);

            return back()->withInput()->with('error', 'Failed to create task: ' . $e->getMessage());
        }
    }

    /**
     * Handle task items (products used)
     */
    protected function handleTaskItems(Task $task, array $items): void
    {
        foreach ($items as $item) {
            if (empty($item['product_id']) || empty($item['quantity'])) {
                continue;
            }

            $unitPrice = $item['unit_price'] ?? 0;
            $quantity = $item['quantity'];
            $subtotal = $unitPrice * $quantity;
            
            // Handle tax calculation - parse tax_ids properly
            $taxIds = [];
            $rawTaxIds = $item['tax_ids'] ?? null;
            
            if (!empty($rawTaxIds)) {
                if (is_array($rawTaxIds)) {
                    $taxIds = array_filter($rawTaxIds); // Remove empty values
                } elseif (is_string($rawTaxIds) && $rawTaxIds !== '[]' && $rawTaxIds !== '') {
                    $decoded = json_decode($rawTaxIds, true);
                    if (is_array($decoded)) {
                        $taxIds = array_filter($decoded);
                    }
                }
            }
            
            // Convert all to integers
            $taxIds = array_map('intval', $taxIds);
            $taxIds = array_filter($taxIds); // Remove zeros
            
            $taxAmount = 0;
            if (!empty($taxIds) && Schema::hasTable('taxes')) {
                $taxes = DB::table('taxes')->whereIn('id', $taxIds)->get();
                foreach ($taxes as $tax) {
                    $taxAmount += ($subtotal * $tax->rate) / 100;
                }
            }
            
            $totalPrice = $subtotal + $taxAmount;

            TaskItem::create([
                'task_id' => $task->id,
                'product_id' => $item['product_id'],
                'quantity' => $quantity,
                'unit_price' => $unitPrice,
                'total_price' => $totalPrice,
                'tax_ids' => !empty($taxIds) ? json_encode(array_values($taxIds)) : null,
                'tax_amount' => $taxAmount,
                'notes' => $item['notes'] ?? null,
            ]);
        }
    }

    /**
     * Handle task file attachments (max 3 files)
     */
    protected function handleTaskAttachments(Task $task, array $files): void
    {
        $uploadCount = 0;
        $maxFiles = 3;

        foreach ($files as $file) {
            if ($uploadCount >= $maxFiles) {
                break;
            }

            if (!$file || !$file->isValid()) {
                continue;
            }

            try {
                $originalName = $file->getClientOriginalName();
                $extension = $file->getClientOriginalExtension();
                $fileName = pathinfo($originalName, PATHINFO_FILENAME);
                $uniqueName = $fileName . '_' . time() . '_' . uniqid() . '.' . $extension;

                $path = $file->storeAs(
                    'tasks/' . $task->id,
                    $uniqueName,
                    'public'
                );

                TaskAttachment::create([
                    'task_id' => $task->id,
                    'filename' => $originalName,
                    'file_path' => $path,
                    'file_size' => $file->getSize(),
                    'uploaded_by' => $this->admin->id,
                ]);

                $uploadCount++;

            } catch (Exception $e) {
                Log::error('Task attachment upload error: ' . $e->getMessage());
            }
        }
    }

    /**
     * Check if current user can access a specific task
     * Admins can access all tasks, Staff can only access assigned/followed/created tasks
     */
    protected function canAccessTask(Task $task): bool
    {
        $admin = $this->admin;
        
        // Full admins (is_admin=1) can access all tasks
        if ($admin->is_admin == 1) {
            return true;
        }
        
        // Check if user created the task
        if ($task->created_by == $admin->id) {
            return true;
        }
        
        // Check if user is assigned to or following the task
        $isAssignee = $task->assignees()->where('admin_id', $admin->id)->exists();
        $isFollower = $task->followers()->where('admin_id', $admin->id)->exists();
        
        return $isAssignee || $isFollower;
    }
    
    /**
     * Check if current user can edit a specific task
     */
    protected function canEditTask(Task $task): bool
    {
        $admin = $this->admin;
        
        // Full admins can edit all
        if ($admin->is_admin == 1) {
            return true;
        }
        
        // Staff can edit if they created it or are assigned/following
        if ($task->created_by == $admin->id) {
            return true;
        }
        if ($task->assignees()->where('admin_id', $admin->id)->exists()) {
            return true;
        }
        if ($task->followers()->where('admin_id', $admin->id)->exists()) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Check if current user can delete a specific task
     */
    protected function canDeleteTask(Task $task): bool
    {
        $admin = $this->admin;
        
        // Only admins can delete tasks
        if ($admin->is_admin == 1) {
            return true;
        }
        
        return false;
    }

    public function show($id)
    {
        try {
            // Build relations array based on what tables exist
            $relations = ['status', 'comments.admin', 'attachments', 'timeLogs.admin', 'timeLogs.pauses', 'assignees', 'followers'];
            
            if (Schema::hasTable('projects')) {
                $relations[] = 'project';
                $relations[] = 'project.customer';
                // Add project status if the relationship exists
                if (Schema::hasTable('project_statuses')) {
                    $relations[] = 'project.status';
                }
            }
            if (Schema::hasTable('project_milestones')) {
                $relations[] = 'milestone';
            }
            if (Schema::hasTable('services')) {
                $relations[] = 'service';
                if (Schema::hasTable('customers')) {
                    $relations[] = 'service.client';
                }
            }
            if (Schema::hasTable('contracts')) {
                $relations[] = 'contract';
                if (Schema::hasTable('customers')) {
                    $relations[] = 'contract.client';
                }
            }
            if (Schema::hasTable('support_tickets')) {
                $relations[] = 'supportTicket';
            }
            
            $task = Task::with($relations)->findOrFail($id);
            
            // Check if user can access this task
            if (!$this->canAccessTask($task)) {
                return redirect()->route('admin.tasks.index')
                    ->with('error', 'You do not have permission to view this task.');
            }

            return $this->View('tasks::tasks.show', compact('task'));

        } catch (Exception $e) {
            Log::error('TaskController@show Error: ' . $e->getMessage());
            return redirect()->route('admin.tasks.index')->with('error', 'Task not found');
        }
    }

    public function edit($id)
    {
        try {
            $task = Task::with(['attachments', 'items', 'assignees', 'followers', 'project', 'service', 'contract', 'supportTicket'])->findOrFail($id);
            
            // Check if user can edit this task
            if (!$this->canEditTask($task)) {
                return redirect()->route('admin.tasks.index')
                    ->with('error', 'You do not have permission to edit this task.');
            }
            
            $statuses = TaskStatus::where('is_active', true)->orderBy('sort_order')->get();
            
            // Task types
            $taskTypes = Task::TASK_TYPES;
            
            // Get projects if Projects module is installed
            $projects = collect([]);
            try {
                if (class_exists('Modules\Projects\Models\Project') && Schema::hasTable('projects')) {
                    $projects = Project::orderBy('title')->get();
                }
            } catch (\Exception $e) {
                // Projects module not available
            }
            
            $admins = \App\Models\Admin::where('is_active', true)->orderBy('name')->get();
            
            // Fetch products for items used section
            $products = collect([]);
            if (Schema::hasTable('products')) {
                $products = DB::table('products')->orderBy('name')->get();
            }
            
            // Fetch taxes for items
            $taxes = collect([]);
            if (Schema::hasTable('taxes')) {
                $taxes = DB::table('taxes')->orderBy('name')->get();
            }

            return $this->View('tasks::tasks.edit', compact('task', 'statuses', 'taskTypes', 'projects', 'admins', 'products', 'taxes'));

        } catch (Exception $e) {
            Log::error('TaskController@edit Error: ' . $e->getMessage() . ' Line: ' . $e->getLine());
            return redirect()->route('admin.tasks.index')->with('error', 'Failed to load task edit page');
        }
    }

    public function update(Request $request, $id)
    {
        $task = Task::findOrFail($id);
        
        // Check if user can edit this task
        if (!$this->canEditTask($task)) {
            return redirect()->route('admin.tasks.index')
                ->with('error', 'You do not have permission to update this task.');
        }

        $validated = $request->validate([
            'task_type' => 'required|in:general,project,service,contract,support',
            'project_id' => 'nullable|integer',
            'service_id' => 'nullable|integer',
            'contract_id' => 'nullable|integer',
            'support_ticket_id' => 'nullable|integer',
            'related_name' => 'nullable|string|max:255',
            'milestone_id' => 'nullable|integer',
            'title' => 'required|string|max:191',
            'description' => 'nullable|string',
            'status_id' => 'required|exists:module_task_statuses,id',
            'priority' => 'required|in:low,medium,high,critical',
            'planned_start_date' => 'nullable|date',
            'due_date' => 'nullable|date',
            'estimated_hours' => 'nullable|numeric|min:0',
            'hourly_rate' => 'nullable|numeric|min:0',
            'is_billable' => 'boolean',
            'assignees' => 'nullable|array',
            'assignees.*' => 'exists:admins,id',
            'followers' => 'nullable|array',
            'followers.*' => 'exists:admins,id',
            'attachments' => 'nullable|array|max:3',
            'attachments.*' => 'file|max:10240|mimes:jpg,jpeg,png,gif,webp,pdf,doc,docx,xls,xlsx,csv,ppt,pptx,txt,zip,rar',
            // Task Items validation
            'items' => 'nullable|array',
            'items.*.product_id' => 'required_with:items|exists:products,id',
            'items.*.quantity' => 'required_with:items|numeric|min:0.01',
            'items.*.unit_price' => 'nullable|numeric|min:0',
            'items.*.tax_ids' => 'nullable',
            'items.*.notes' => 'nullable|string|max:500',
        ]);

        DB::beginTransaction();
        try {
            $validated['updated_by'] = $this->admin->id;
            
            // Store related name for reference
            if ($validated['task_type'] !== 'general' && empty($validated['related_name'])) {
                $validated['related_name'] = $this->getRelatedName($validated);
            }
            
            $assignees = $validated['assignees'] ?? [];
            $followers = $validated['followers'] ?? [];
            $attachmentFiles = $request->file('attachments', []);
            $items = $validated['items'] ?? [];
            unset($validated['assignees'], $validated['followers'], $validated['attachments'], $validated['items']);
            
            $task->update($validated);
            
            // Sync assignees
            $task->assignees()->sync($assignees);
            
            // Sync followers (managers)
            $task->followers()->sync($followers);

            // Handle new file uploads (max 3 files)
            if (!empty($attachmentFiles)) {
                $this->handleTaskAttachments($task, $attachmentFiles);
            }

            // Handle task items (delete old, add new)
            $task->items()->delete();
            if (!empty($items)) {
                $this->handleTaskItems($task, $items);
            }

            DB::commit();

            return redirect()->route('admin.tasks.show', $task->id)
                ->with('success', 'Task updated successfully!');

        } catch (Exception $e) {
            DB::rollback();
            Log::error('TaskController@update Error: ' . $e->getMessage(), [
                'task_id' => $id,
                'trace' => $e->getTraceAsString()
            ]);

            return back()->withInput()->with('error', 'Failed to update task: ' . $e->getMessage());
        }
    }

    public function destroy($id)
    {
        try {
            $task = Task::findOrFail($id);
            
            // Check if user can delete this task
            if (!$this->canDeleteTask($task)) {
                if (request()->ajax()) {
                    return response()->json([
                        'success' => false, 
                        'message' => 'You do not have permission to delete this task.'
                    ], 403);
                }
                return redirect()->route('admin.tasks.index')
                    ->with('error', 'You do not have permission to delete this task.');
            }
            
            $task->delete();

            if (request()->ajax()) {
                return response()->json([
                    'success' => true, 
                    'message' => 'Task deleted successfully!'
                ]);
            }

            return redirect()->route('admin.tasks.index')
                ->with('success', 'Task deleted successfully!');

        } catch (Exception $e) {
            Log::error('TaskController@destroy Error: ' . $e->getMessage());

            if (request()->ajax()) {
                return response()->json([
                    'success' => false, 
                    'message' => 'Failed to delete task'
                ], 500);
            }

            return back()->with('error', 'Failed to delete task');
        }
    }

    public function bulkDelete(Request $request): JsonResponse
    {
        try {
            $ids = $request->input('ids', []);
            
            if (empty($ids)) {
                return response()->json([
                    'success' => false, 
                    'message' => 'No tasks selected'
                ], 400);
            }

            $deleted = Task::whereIn('id', $ids)->delete();

            return response()->json([
                'success' => true, 
                'message' => "{$deleted} task(s) deleted successfully!"
            ]);

        } catch (Exception $e) {
            Log::error('TaskController@bulkDelete Error: ' . $e->getMessage());

            return response()->json([
                'success' => false, 
                'message' => 'Failed to delete tasks'
            ], 500);
        }
    }

    public function bulkStatusUpdate(Request $request): JsonResponse
    {
        try {
            $ids = $request->input('ids', []);
            $statusId = $request->input('status_id');
            
            if (empty($ids) || !$statusId) {
                return response()->json([
                    'success' => false, 
                    'message' => 'Invalid request data'
                ], 400);
            }

            $updated = Task::whereIn('id', $ids)->update(['status_id' => $statusId]);

            return response()->json([
                'success' => true, 
                'message' => "{$updated} task(s) updated successfully!"
            ]);

        } catch (Exception $e) {
            Log::error('TaskController@bulkStatusUpdate Error: ' . $e->getMessage());

            return response()->json([
                'success' => false, 
                'message' => 'Failed to update tasks'
            ], 500);
        }
    }

    /**
     * ============================================================
     * TIMER METHODS WITH GPS TRACKING
     * ============================================================
     */

    /**
     * Start timer with GPS location tracking
     * Requires: latitude, longitude, location_accuracy from frontend
     */
    public function startTimer(Request $request, $id): JsonResponse
    {
        try {
            // Ensure admin is authenticated
            if (!$this->admin) {
                return response()->json([
                    'success' => false,
                    'message' => 'Authentication required. Please login again.'
                ], 401);
            }

            // Validate incoming GPS data (optional)
            $validated = $request->validate([
                'latitude' => 'nullable|numeric|between:-90,90',
                'longitude' => 'nullable|numeric|between:-180,180',
                'location_accuracy' => 'nullable|numeric|min:0',
            ]);

            $task = Task::findOrFail($id);
            
            // Check if user can access this task
            if (!$this->canAccessTask($task)) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to start timer on this task.'
                ], 403);
            }
            
            // Check if THIS TASK already has ANY time log (active or finished)
            // Only ONE time record allowed per task
            $existingTimer = TimeLog::where('task_id', $task->id)
                ->where('admin_id', $this->admin->id)
                ->first();
                
            if ($existingTimer) {
                // Check if it's still active (not finished)
                if (!$existingTimer->end_time) {
                    return response()->json([
                        'success' => false,
                        'message' => 'This task already has an active timer.',
                    ], 400);
                }
                // Already finished - cannot start again
                return response()->json([
                    'success' => false,
                    'message' => 'This task has already been completed. Time tracking is only allowed once per task.',
                ], 400);
            }

            // Create time log with GPS data
            $hourlyRate = null;
            try {
                if (isset($task->hourly_rate) && $task->hourly_rate) {
                    $hourlyRate = $task->hourly_rate;
                } elseif ($task->project && isset($task->project->hourly_rate)) {
                    $hourlyRate = $task->project->hourly_rate;
                }
            } catch (\Exception $e) {
                // Ignore hourly rate errors
            }
            
            $timeLog = TimeLog::create([
                'task_id' => $task->id,
                'admin_id' => $this->admin->id,
                'start_time' => now(),
                'is_running' => true,
                'is_paused' => false,
                'is_billable' => $task->is_billable ?? true,
                'hourly_rate' => $hourlyRate,
                'duration_minutes' => 0,
                'total_pause_seconds' => 0,
                // GPS & Location Data (optional)
                'ip_address' => $request->ip(),
                'latitude' => $validated['latitude'] ?? null,
                'longitude' => $validated['longitude'] ?? null,
                'location_accuracy' => $validated['location_accuracy'] ?? null,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Timer started successfully!',
                'data' => [
                    'timer_id' => $timeLog->id,
                    'start_time' => $timeLog->start_time->toIso8601String(),
                    'task_title' => $task->title,
                    'has_location' => $timeLog->has_location,
                    'location_string' => $timeLog->location_string,
                ]
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid GPS data provided',
                'errors' => $e->errors()
            ], 422);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Server Error',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Pause running timer
     * Records pause event with location
     */
    public function pauseTimer(Request $request, $id): JsonResponse
    {
        try {
            if (!$this->admin) {
                return response()->json([
                    'success' => false,
                    'message' => 'Authentication required'
                ], 401);
            }

            $task = Task::findOrFail($id);
            
            // Find running timer
            $timeLog = TimeLog::where('task_id', $task->id)
                ->where('admin_id', $this->admin->id)
                ->where('is_running', true)
                ->whereNull('end_time')
                ->first();
                
            if (!$timeLog) {
                return response()->json([
                    'success' => false,
                    'message' => 'No running timer found for this task'
                ], 404);
            }

            // Pause the timer with location
            $pause = $timeLog->pauseTimer([
                'latitude' => $request->input('latitude'),
                'longitude' => $request->input('longitude'),
                'accuracy' => $request->input('accuracy'),
                'ip_address' => $request->ip(),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Timer paused successfully!',
                'data' => [
                    'timer_id' => $timeLog->id,
                    'pause_id' => $pause->id,
                    'paused_at' => $pause->paused_at->toIso8601String(),
                    'pause_count' => $timeLog->pause_count,
                    'is_running' => false,
                    'is_paused' => true,
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to pause timer',
                'error' => config('app.debug') ? $e->getMessage() : 'Server Error'
            ], 500);
        }
    }

    /**
     * Resume paused timer
     * Records resume event with location
     */
    public function resumeTimer(Request $request, $id): JsonResponse
    {
        try {
            if (!$this->admin) {
                return response()->json([
                    'success' => false,
                    'message' => 'Authentication required'
                ], 401);
            }

            $task = Task::findOrFail($id);
            
            // Find paused timer
            $timeLog = TimeLog::where('task_id', $task->id)
                ->where('admin_id', $this->admin->id)
                ->where('is_paused', true)
                ->whereNull('end_time')
                ->first();
                
            if (!$timeLog) {
                return response()->json([
                    'success' => false,
                    'message' => 'No paused timer found for this task'
                ], 404);
            }

            // Resume timer with location (no restriction on other running timers)
            $timeLog->resumeTimer([
                'latitude' => $request->input('latitude'),
                'longitude' => $request->input('longitude'),
                'accuracy' => $request->input('accuracy'),
                'ip_address' => $request->ip(),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Timer resumed successfully!',
                'data' => [
                    'timer_id' => $timeLog->id,
                    'total_pause_seconds' => $timeLog->total_pause_seconds,
                    'pause_count' => $timeLog->pause_count,
                    'is_running' => true,
                    'is_paused' => false,
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to resume timer',
                'error' => config('app.debug') ? $e->getMessage() : 'Server Error'
            ], 500);
        }
    }

    /**
     * Finish timer and finalize time log
     * Calculates final duration (total time - pause time)
     */
    public function stopTimer(Request $request, $id): JsonResponse
    {
        try {
            if (!$this->admin) {
                return response()->json([
                    'success' => false,
                    'message' => 'Authentication required'
                ], 401);
            }

            $task = Task::findOrFail($id);
            
            // Find active timer (running or paused)
            $timeLog = TimeLog::where('task_id', $task->id)
                ->where('admin_id', $this->admin->id)
                ->whereNull('end_time')
                ->first();
                
            if (!$timeLog) {
                return response()->json([
                    'success' => false,
                    'message' => 'No active timer found for this task'
                ], 404);
            }

            // Finish the timer with end location
            $timeLog->finishTimer([
                'latitude' => $request->input('latitude'),
                'longitude' => $request->input('longitude'),
                'accuracy' => $request->input('accuracy'),
                'ip_address' => $request->ip(),
            ]);

            // Reload to get updated values
            $timeLog->refresh();

            // Calculate total seconds from start to end
            $totalSeconds = $timeLog->start_time->diffInSeconds($timeLog->end_time);
            
            // Get pause seconds
            $pauseSeconds = $timeLog->total_pause_seconds ?? 0;
            
            // Working seconds = total - pauses
            $workingSeconds = max(0, $totalSeconds - $pauseSeconds);

            // Format times as HH:MM:SS
            $formattedWorkingTime = $this->formatSecondsToTime($workingSeconds);
            $formattedTotalTime = $this->formatSecondsToTime($totalSeconds);
            $formattedPauseTime = $this->formatSecondsToTime($pauseSeconds);

            return response()->json([
                'success' => true,
                'message' => 'Timer finished successfully!',
                'data' => [
                    'timer_id' => $timeLog->id,
                    'working_seconds' => $workingSeconds,
                    'total_seconds' => $totalSeconds,
                    'pause_seconds' => $pauseSeconds,
                    'formatted_working_time' => $formattedWorkingTime,
                    'total_time' => $formattedTotalTime,
                    'pause_time' => $formattedPauseTime,
                    'pause_count' => $timeLog->pauses()->count(),
                    'start_time' => $timeLog->start_time->format('d M Y, h:i:s A'),
                    'end_time' => $timeLog->end_time->format('d M Y, h:i:s A'),
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to finish timer',
                'error' => config('app.debug') ? $e->getMessage() : 'Server Error'
            ], 500);
        }
    }

    /**
     * Format seconds to HH:MM:SS
     */
    private function formatSecondsToTime($seconds): string
    {
        $hours = floor($seconds / 3600);
        $minutes = floor(($seconds % 3600) / 60);
        $secs = $seconds % 60;
        return sprintf('%02d:%02d:%02d', $hours, $minutes, $secs);
    }

    /**
     * Format duration helper (kept for compatibility)
     */
    private function formatDuration($minutes): string
    {
        $hours = floor($minutes / 60);
        $mins = $minutes % 60;
        return sprintf('%02d:%02d:00', $hours, $mins);
    }

    /**
     * ============================================================
     * COMMENT & ATTACHMENT METHODS
     * ============================================================
     */

    public function addComment(Request $request, $id): JsonResponse
    {
        try {
            $task = Task::findOrFail($id);
            
            $validated = $request->validate([
                'comment' => 'required|string',
                'portal_visible' => 'boolean',
            ]);

            $comment = TaskComment::create([
                'task_id' => $task->id,
                'admin_id' => $this->admin->id,
                'comment' => $validated['comment'],
                'portal_visible' => $validated['portal_visible'] ?? false,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Comment added successfully!',
                'comment' => $comment->load('admin')
            ]);

        } catch (Exception $e) {
            Log::error('TaskController@addComment Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to add comment'
            ], 500);
        }
    }

    public function attachFile(Request $request, $id): JsonResponse
    {
        try {
            $task = Task::findOrFail($id);
            
            $request->validate([
                'file' => 'required|file|max:10240', // 10MB max
            ]);

            $file = $request->file('file');
            $path = $this->uploadFile($file, 'task-attachments', 'public');
            
            if (!$path) {
                return response()->json([
                    'success' => false, 
                    'message' => 'File upload failed'
                ], 500);
            }

            $attachment = TaskAttachment::create([
                'task_id' => $task->id,
                'filename' => $file->getClientOriginalName(),
                'file_path' => $path,
                'file_size' => $file->getSize(),
                'uploaded_by' => $this->admin->id,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'File uploaded successfully!',
                'attachment' => $attachment
            ]);

        } catch (Exception $e) {
            Log::error('TaskController@attachFile Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to upload file'
            ], 500);
        }
    }

    /**
     * Download task attachment
     */
    public function downloadAttachment($attachmentId)
    {
        try {
            $attachment = TaskAttachment::findOrFail($attachmentId);
            
            $filePath = storage_path('app/public/' . $attachment->file_path);
            
            if (!file_exists($filePath)) {
                return back()->with('error', 'File not found');
            }
            
            return response()->download($filePath, $attachment->filename);
            
        } catch (Exception $e) {
            Log::error('TaskController@downloadAttachment Error: ' . $e->getMessage());
            return back()->with('error', 'Failed to download file');
        }
    }

    /**
     * Preview/stream task attachment (for images)
     */
    public function previewAttachment($attachmentId)
    {
        try {
            $attachment = TaskAttachment::findOrFail($attachmentId);
            
            $filePath = storage_path('app/public/' . $attachment->file_path);
            
            if (!file_exists($filePath)) {
                abort(404, 'File not found');
            }
            
            $mimeType = mime_content_type($filePath);
            
            return response()->file($filePath, [
                'Content-Type' => $mimeType,
            ]);
            
        } catch (Exception $e) {
            Log::error('TaskController@previewAttachment Error: ' . $e->getMessage());
            abort(404, 'File not found');
        }
    }

    public function deleteAttachment($id, $attachmentId): JsonResponse
    {
        try {
            $task = Task::findOrFail($id);
            $attachment = TaskAttachment::where('task_id', $task->id)
                ->findOrFail($attachmentId);
                
            // Delete physical file
            $this->deleteFile($attachment->file_path, 'public');
            
            // Delete record
            $attachment->delete();

            return response()->json([
                'success' => true,
                'message' => 'Attachment deleted successfully!'
            ]);

        } catch (Exception $e) {
            Log::error('TaskController@deleteAttachment Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to delete attachment'
            ], 500);
        }
    }

    /**
     * Get related name based on task type
     */
    protected function getRelatedName(array $data): ?string
    {
        try {
            switch ($data['task_type']) {
                case 'project':
                    if (!empty($data['project_id']) && Schema::hasTable('projects')) {
                        $project = DB::table('projects')->where('id', $data['project_id'])->first();
                        return $project->title ?? null;
                    }
                    break;

                case 'service':
                    if (!empty($data['service_id']) && Schema::hasTable('services')) {
                        $service = DB::table('services')->where('id', $data['service_id'])->first();
                        return $service->name ?? null;
                    }
                    break;

                case 'contract':
                    if (!empty($data['contract_id']) && Schema::hasTable('contracts')) {
                        $contract = DB::table('contracts')->where('id', $data['contract_id'])->first();
                        return $contract->subject ?? null;
                    }
                    break;

                case 'support':
                    if (!empty($data['support_ticket_id']) && Schema::hasTable('support_tickets')) {
                        $ticket = DB::table('support_tickets')->where('id', $data['support_ticket_id'])->first();
                        return $ticket->subject ?? null;
                    }
                    break;
            }
        } catch (\Exception $e) {
            // Silently fail
        }

        return null;
    }
}