<?php

namespace Modules\Tasks\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;

class Task extends Model
{
    use SoftDeletes;

    protected $table = 'module_tasks';

    const TASK_TYPES = [
        'general' => 'General Task',
        'project' => 'Project Task',
        'service' => 'Service Task',
        'contract' => 'Contract Task',
        'support' => 'Support Ticket Task',
    ];

    protected $fillable = [
        'task_type',
        'project_id',
        'service_id',
        'contract_id',
        'support_ticket_id',
        'related_name',
        'milestone_id',
        'title',
        'description',
        'status_id',
        'priority',
        'planned_start_date',
        'due_date',
        'completed_at',
        'estimated_hours',
        'hourly_rate',
        'is_billable',
        'is_recurring',
        'recurring_frequency',
        'recurring_until',
        'parent_task_id',
        'sort_order',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'planned_start_date' => 'date',
        'due_date' => 'date',
        'completed_at' => 'datetime',
        'estimated_hours' => 'decimal:2',
        'hourly_rate' => 'decimal:2',
        'is_billable' => 'boolean',
        'is_recurring' => 'boolean',
        'recurring_until' => 'date',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime',
    ];

    // ==================== RELATIONSHIPS ====================

    public function project()
    {
        if (class_exists('Modules\Projects\Models\Project')) {
            return $this->belongsTo('Modules\Projects\Models\Project', 'project_id');
        }
        return $this->belongsTo(Task::class, 'project_id')->whereRaw('1 = 0');
    }

    public function service()
    {
        // Check if services table exists
        if (class_exists('App\Models\Service')) {
            return $this->belongsTo('App\Models\Service', 'service_id');
        }
        return $this->belongsTo(Task::class, 'service_id')->whereRaw('1 = 0');
    }

    public function contract()
    {
        // Check if contracts table exists
        if (class_exists('App\Models\Contract')) {
            return $this->belongsTo('App\Models\Contract', 'contract_id');
        }
        return $this->belongsTo(Task::class, 'contract_id')->whereRaw('1 = 0');
    }

    public function supportTicket()
    {
        // Check if support_tickets table exists
        if (class_exists('App\Models\SupportTicket')) {
            return $this->belongsTo('App\Models\SupportTicket', 'support_ticket_id');
        }
        // Try alternate model location
        if (class_exists('Modules\Support\Models\Ticket')) {
            return $this->belongsTo('Modules\Support\Models\Ticket', 'support_ticket_id');
        }
        return $this->belongsTo(Task::class, 'support_ticket_id')->whereRaw('1 = 0');
    }

    public function milestone()
    {
        if (class_exists('Modules\Projects\Models\Milestone')) {
            return $this->belongsTo('Modules\Projects\Models\Milestone', 'milestone_id');
        }
        return $this->belongsTo(Task::class, 'milestone_id')->whereRaw('1 = 0');
    }

    public function status()
    {
        return $this->belongsTo(TaskStatus::class, 'status_id');
    }

    public function creator()
    {
        return $this->belongsTo('App\Models\Admin', 'created_by');
    }

    public function updater()
    {
        return $this->belongsTo('App\Models\Admin', 'updated_by');
    }

    /**
     * Get all assignees for this task (multiple members)
     */
    public function assignees()
    {
        return $this->belongsToMany('App\Models\Admin', 'module_task_assignees', 'task_id', 'admin_id')
            ->withTimestamps();
    }

    /**
     * Get all followers/managers for this task (multiple members)
     */
    public function followers()
    {
        return $this->belongsToMany('App\Models\Admin', 'module_task_followers', 'task_id', 'admin_id')
            ->withTimestamps();
    }

    public function timeLogs()
    {
        return $this->hasMany(TimeLog::class, 'task_id');
    }

    public function comments()
    {
        return $this->hasMany(TaskComment::class, 'task_id')->orderBy('created_at', 'asc');
    }

    public function attachments()
    {
        return $this->hasMany(TaskAttachment::class, 'task_id');
    }

    public function items()
    {
        return $this->hasMany(TaskItem::class, 'task_id');
    }

    public function checklists()
    {
        return $this->hasMany(TaskChecklist::class, 'task_id')->orderBy('sort_order');
    }

    public function parentTask()
    {
        return $this->belongsTo(Task::class, 'parent_task_id');
    }

    public function subtasks()
    {
        return $this->hasMany(Task::class, 'parent_task_id');
    }

    // ==================== SCOPES ====================

    /**
     * Scope to filter tasks visible to a specific admin
     * Admin (is_admin=1) sees all tasks
     * Staff (is_admin=0) sees only assigned/followed/created tasks
     */
    public function scopeVisibleTo(Builder $query, $admin): Builder
    {
        // Check if user is a full admin (is_admin = 1)
        if ($admin->is_admin == 1) {
            return $query; // Admin sees all tasks
        }
        
        // Staff sees only tasks assigned to them, followed by them, or created by them
        return $query->where(function($q) use ($admin) {
            $q->whereHas('assignees', function ($sub) use ($admin) {
                $sub->where('admin_id', $admin->id);
            })->orWhereHas('followers', function ($sub) use ($admin) {
                $sub->where('admin_id', $admin->id);
            })->orWhere('created_by', $admin->id);
        });
    }

    /**
     * Scope to filter by assignee
     */
    public function scopeAssignedTo(Builder $query, $adminId): Builder
    {
        return $query->whereHas('assignees', function ($q) use ($adminId) {
            $q->where('admin_id', $adminId);
        });
    }

    public function scopeActive(Builder $query): Builder
    {
        return $query->whereHas('status', function ($q) {
            $q->where('show_by_default', true);
        });
    }

    public function scopeCompleted(Builder $query): Builder
    {
        return $query->whereHas('status', function ($q) {
            $q->where('is_completed', true);
        });
    }

    public function scopeIncomplete(Builder $query): Builder
    {
        return $query->whereHas('status', function ($q) {
            $q->where('is_completed', false);
        });
    }

    public function scopeBillable(Builder $query): Builder
    {
        return $query->where('is_billable', true);
    }

    public function scopeOverdue(Builder $query): Builder
    {
        return $query->where('due_date', '<', now())
            ->whereHas('status', function ($q) {
                $q->where('is_completed', false);
            });
    }

    public function scopeSearch(Builder $query, ?string $search): Builder
    {
        if (empty($search)) {
            return $query;
        }

        return $query->where(function ($q) use ($search) {
            $q->where('title', 'LIKE', "%{$search}%")
                ->orWhere('description', 'LIKE', "%{$search}%")
                ->orWhere('related_name', 'LIKE', "%{$search}%");
        });
    }

    public function scopeByType(Builder $query, $type): Builder
    {
        return $query->where('task_type', $type);
    }

    public function scopeByProject(Builder $query, $projectId): Builder
    {
        return $query->where('project_id', $projectId);
    }

    public function scopeByService(Builder $query, $serviceId): Builder
    {
        return $query->where('service_id', $serviceId);
    }

    public function scopeByContract(Builder $query, $contractId): Builder
    {
        return $query->where('contract_id', $contractId);
    }

    public function scopeByMilestone(Builder $query, $milestoneId): Builder
    {
        return $query->where('milestone_id', $milestoneId);
    }

    public function scopeByStatus(Builder $query, $statusId): Builder
    {
        return $query->where('status_id', $statusId);
    }

    public function scopeByPriority(Builder $query, $priority): Builder
    {
        return $query->where('priority', $priority);
    }

    // ==================== ACCESSORS ====================

    public function getTaskTypeLabelAttribute(): string
    {
        return self::TASK_TYPES[$this->task_type] ?? ucfirst($this->task_type);
    }

    public function getStatusLabelAttribute(): string
    {
        return $this->status ? $this->status->label : 'Unknown';
    }

    public function getStatusColorAttribute(): string
    {
        return $this->status ? $this->status->color : '#9ca3af';
    }

    public function getPriorityLabelAttribute(): string
    {
        $priorities = config('tasks.priorities', [
            'low' => 'Low',
            'medium' => 'Medium',
            'high' => 'High',
            'critical' => 'Critical',
        ]);
        return $priorities[$this->priority] ?? ucfirst($this->priority);
    }

    public function getPriorityColorAttribute(): string
    {
        $colors = [
            'low' => '#10b981',
            'medium' => '#3b82f6',
            'high' => '#f59e0b',
            'critical' => '#ef4444',
        ];
        return $colors[$this->priority] ?? '#9ca3af';
    }

    public function getIsOverdueAttribute(): bool
    {
        if (!$this->due_date) {
            return false;
        }
        return $this->due_date->isPast() && !$this->completed_at;
    }

    public function getIsCompletedAttribute(): bool
    {
        return $this->status && $this->status->is_completed;
    }

    public function getTotalTimeLoggedAttribute(): float
    {
        return $this->timeLogs()->sum('duration_minutes') / 60;
    }

    public function getTotalBillableTimeAttribute(): float
    {
        return $this->timeLogs()->where('is_billable', true)->sum('duration_minutes') / 60;
    }

    /**
     * Get assignee names as comma-separated string
     */
    public function getAssigneeNamesAttribute(): string
    {
        return $this->assignees->pluck('name')->implode(', ') ?: 'Unassigned';
    }

    /**
     * Check if this task has an active timer running
     */
    public function getHasActiveTimerAttribute(): bool
    {
        try {
            $adminId = null;
            if (auth()->guard('admin')->check()) {
                $adminId = auth()->guard('admin')->id();
            } elseif (auth()->check()) {
                $adminId = auth()->id();
            }
            
            if (!$adminId) {
                return false;
            }
            
            return $this->timeLogs()
                ->where('admin_id', $adminId)
                ->whereNull('end_time')
                ->exists();
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Check if this task's timer has been completed (finished)
     */
    public function getHasCompletedTimerAttribute(): bool
    {
        try {
            $adminId = null;
            if (auth()->guard('admin')->check()) {
                $adminId = auth()->guard('admin')->id();
            } elseif (auth()->check()) {
                $adminId = auth()->id();
            }
            
            if (!$adminId) {
                return false;
            }
            
            return $this->timeLogs()
                ->where('admin_id', $adminId)
                ->whereNotNull('end_time')
                ->exists();
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Get the active timer for this task
     */
    public function getActiveTimerAttribute()
    {
        try {
            $adminId = null;
            if (auth()->guard('admin')->check()) {
                $adminId = auth()->guard('admin')->id();
            } elseif (auth()->check()) {
                $adminId = auth()->id();
            }
            
            if (!$adminId) {
                return null;
            }
            
            return $this->timeLogs()
                ->where('admin_id', $adminId)
                ->whereNull('end_time')
                ->first();
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Get the completed timer for this task
     */
    public function getCompletedTimerAttribute()
    {
        try {
            $adminId = null;
            if (auth()->guard('admin')->check()) {
                $adminId = auth()->guard('admin')->id();
            } elseif (auth()->check()) {
                $adminId = auth()->id();
            }
            
            if (!$adminId) {
                return null;
            }
            
            return $this->timeLogs()
                ->where('admin_id', $adminId)
                ->whereNotNull('end_time')
                ->first();
        } catch (\Exception $e) {
            return null;
        }
    }
}
