<style>
/* ========================================
   Task Detail - Theme-Aware Styles
   ======================================== */

/* Page Animation */
@keyframes fadeIn {
    from { opacity: 0; transform: translateY(10px); }
    to { opacity: 1; transform: translateY(0); }
}

@keyframes slideDown {
    from { opacity: 0; transform: translateY(-10px); }
    to { opacity: 1; transform: translateY(0); }
}

.task-detail-page {
    padding: 2rem;
    animation: fadeIn 0.4s ease;
}

/* Page Header */
.page-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 2rem;
    flex-wrap: wrap;
    gap: 1.5rem;
}

.page-title-section {
    display: flex;
    align-items: center;
    gap: 1.5rem;
    flex: 1;
    min-width: 0;
}

.btn-back {
    width: 40px;
    height: 40px;
    display: flex;
    align-items: center;
    justify-content: center;
    background: #ffffff;
    border: 1px solid #e5e7eb;
    border-radius: 0.5rem;
    text-decoration: none;
    color: #111827;
    transition: all 0.3s ease;
    flex-shrink: 0;
}

.btn-back:hover {
    background: #3b82f6;
    border-color: #3b82f6;
    color: white;
    transform: translateY(-2px);
}

.btn-back svg {
    width: 20px;
    height: 20px;
}

.title-content {
    flex: 1;
    min-width: 0;
}

.hidden {
    display: none !important;
}

.page-title {
    font-size: 1.25rem;
    font-weight: 700;
    color: #111827;
    margin: 0 0 0.25rem 0;
    word-wrap: break-word;
}

.page-subtitle {
    font-size: 0.875rem;
    color: #6b7280;
    margin: 0;
    display: flex;
    align-items: center;
    gap: 0.25rem;
}

.page-subtitle svg {
    width: 14px;
    height: 14px;
}

/* Action Buttons */
.action-buttons {
    display: flex;
    gap: 1rem;
}

.btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 10px 20px;
    border-radius: 0.5rem;
    font-size: 1rem;
    font-weight: 600;
    text-decoration: none;
    transition: all 0.3s ease;
    cursor: pointer;
    border: none;
}

.btn svg {
    width: 18px;
    height: 18px;
}

.btn-primary {
    background: #3b82f6;
    color: white;
}

.btn-primary:hover {
    background: #2563eb;
    transform: translateY(-2px);
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
}

.btn-secondary {
    background: #ffffff;
    border: 1px solid #e5e7eb;
    color: #111827;
}

.btn-secondary:hover {
    background: #f3f4f6;
    transform: translateY(-2px);
}

/* Timer Button Container */
.timer-button-wrapper {
    position: relative;
}

/* Timer Modal */
.timer-modal {
    position: absolute;
    top: calc(100% + 12px);
    right: 0;
    background: #ffffff;
    border: 1px solid #e5e7eb;
    border-radius: 0.5rem;
    padding: 1.5rem;
    min-width: 320px;
    box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1);
    z-index: 1000;
    animation: slideDown 0.3s ease;
}

.timer-modal::before {
    content: '';
    position: absolute;
    top: -8px;
    right: 20px;
    width: 0;
    height: 0;
    border-left: 8px solid transparent;
    border-right: 8px solid transparent;
    border-bottom: 8px solid #e5e7eb;
}

.timer-modal::after {
    content: '';
    position: absolute;
    top: -7px;
    right: 21px;
    width: 0;
    height: 0;
    border-left: 7px solid transparent;
    border-right: 7px solid transparent;
    border-bottom: 7px solid #ffffff;
}

.timer-modal.hidden {
    display: none;
}

/* Pulse animation for running timer */
@keyframes pulse {
    0%, 100% { opacity: 1; transform: scale(1); }
    50% { opacity: 0.7; transform: scale(1.05); }
}

/* Timer Display */
.timer-display {
    text-align: center;
    margin-bottom: 1.5rem;
}

.timer-label {
    font-size: 0.875rem;
    color: #6b7280;
    margin-bottom: 0.5rem;
    font-weight: 500;
}

.timer-time {
    font-size: 48px;
    font-weight: 700;
    color: #3b82f6;
    font-family: 'Courier New', monospace;
    letter-spacing: 2px;
}

@keyframes pulse {
    0%, 100% { opacity: 1; }
    50% { opacity: 0.6; }
}

.timer-time.running {
    animation: pulse 2s ease-in-out infinite;
}

.timer-time.paused {
    color: #f59e0b;
}

/* Timer Info */
.timer-info {
    background: #f3f4f6;
    border-radius: 0.375rem;
    padding: 1rem;
    margin-bottom: 1.5rem;
}

.timer-info-item {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-size: 0.875rem;
    color: #6b7280;
    margin-bottom: 0.25rem;
}

.timer-info-item:last-child {
    margin-bottom: 0;
}

.timer-info-item svg {
    width: 14px;
    height: 14px;
    color: #10b981;
}

/* Timer Controls */
.timer-controls {
    display: flex;
    gap: 0.5rem;
}

.timer-btn {
    flex: 1;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.25rem;
    padding: 10px 16px;
    border-radius: 0.375rem;
    font-size: 0.875rem;
    font-weight: 600;
    border: none;
    cursor: pointer;
    transition: all 0.3s ease;
}

.timer-btn svg {
    width: 16px;
    height: 16px;
}

.timer-btn-pause {
    background: #fef3c7;
    color: #f59e0b;
}

.timer-btn-pause:hover {
    background: #f59e0b;
    color: white;
}

.timer-btn-resume {
    background: #d1fae5;
    color: #10b981;
}

.timer-btn-resume:hover {
    background: #10b981;
    color: white;
}

.timer-btn-stop {
    background: #fee2e2;
    color: #ef4444;
}

.timer-btn-stop:hover {
    background: #ef4444;
    color: white;
}

/* Loading State */
.timer-loading {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 1rem;
    padding: 2rem;
}

@keyframes spin {
    from { transform: rotate(0deg); }
    to { transform: rotate(360deg); }
}

.spinner {
    width: 40px;
    height: 40px;
    border: 3px solid #e5e7eb;
    border-top-color: #3b82f6;
    border-radius: 50%;
    animation: spin 0.8s linear infinite;
}

.timer-loading-text {
    font-size: 0.875rem;
    color: #6b7280;
}

/* Success Message */
.timer-success {
    text-align: center;
    padding: 1.5rem;
}

.success-icon {
    width: 64px;
    height: 64px;
    margin: 0 auto 1rem;
    color: #10b981;
}

.success-message {
    font-size: 0.9375rem;
    font-weight: 600;
    color: #111827;
    margin-bottom: 0.5rem;
}

.success-detail {
    font-size: 0.875rem;
    color: #6b7280;
}

/* Error Message */
.timer-error {
    background: #fee2e2;
    border: 1px solid #ef4444;
    border-radius: 0.375rem;
    padding: 1rem;
    margin-bottom: 1.5rem;
}

.error-message {
    display: flex;
    align-items: flex-start;
    gap: 0.5rem;
    font-size: 0.875rem;
    color: #ef4444;
}

.error-message svg {
    width: 18px;
    height: 18px;
    flex-shrink: 0;
    margin-top: 2px;
}

/* Info Cards */
.info-card {
    background: #ffffff;
    border: 1px solid #e5e7eb;
    border-radius: 0.5rem;
    padding: 2rem;
    margin-bottom: 2rem;
    transition: all 0.3s ease;
}

.info-card:hover {
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
}

.card-title {
    font-size: 1.125rem;
    font-weight: 600;
    color: #111827;
    margin: 0 0 1.5rem 0;
}

.info-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1.5rem;
}

.info-item {
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
}

.info-label {
    font-size: 0.75rem;
    font-weight: 600;
    color: #6b7280;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.info-value {
    font-size: 0.9375rem;
    color: #111827;
    font-weight: 500;
}

/* Priority Badge */
.priority-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.25rem;
    padding: 6px 12px;
    border-radius: 12px;
    font-size: 0.875rem;
    font-weight: 600;
}

.priority-badge svg {
    width: 14px;
    height: 14px;
}

.priority-low {
    background: #d1fae5;
    color: #10b981;
}

.priority-medium {
    background: #dbeafe;
    color: #3b82f6;
}

.priority-high {
    background: #fef3c7;
    color: #f59e0b;
}

.priority-critical {
    background: #fee2e2;
    color: #ef4444;
}

/* Status Badge */
.status-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.25rem;
    padding: 6px 12px;
    border-radius: 12px;
    font-size: 0.875rem;
    font-weight: 600;
    background: #dbeafe;
    color: #3b82f6;
}

.status-badge svg {
    width: 14px;
    height: 14px;
}

/* Owner Tags */
.owner-tags {
    display: flex;
    flex-wrap: wrap;
    gap: 0.5rem;
}

.owner-tag {
    display: inline-flex;
    align-items: center;
    gap: 0.25rem;
    padding: 8px 16px;
    background: #dbeafe;
    border-radius: 20px;
    font-size: 0.875rem;
    font-weight: 500;
    color: #3b82f6;
}

.owner-tag svg {
    width: 14px;
    height: 14px;
}

/* Description */
.description-content {
    margin-top: 1.5rem;
    padding-top: 1.5rem;
    border-top: 1px solid #e5e7eb;
}

.description-text {
    font-size: 1rem;
    color: #6b7280;
    line-height: 1.6;
    white-space: pre-wrap;
    word-wrap: break-word;
}

/* Responsive */
@media (max-width: 768px) {
    .page-header {
        flex-direction: column;
        align-items: flex-start;
    }
    
    .action-buttons {
        width: 100%;
    }
    
    .btn {
        flex: 1;
        justify-content: center;
    }
    
    .timer-modal {
        right: 0;
        left: 0;
        min-width: auto;
    }
    
    .info-grid {
        grid-template-columns: 1fr;
    }
}

/* ========== ATTACHMENTS STYLES ========== */
.attachments-grid {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.attachment-item {
    display: flex;
    align-items: center;
    gap: 1rem;
    padding: 1rem;
    background: #f3f4f6;
    border: 1px solid #e5e7eb;
    border-radius: 0.375rem;
    transition: all 0.2s ease;
}

.attachment-item:hover {
    border-color: #3b82f6;
    box-shadow: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
}

.attachment-preview {
    width: 60px;
    height: 60px;
    border-radius: 0.375rem;
    overflow: hidden;
    cursor: pointer;
    flex-shrink: 0;
    transition: transform 0.2s ease;
}

.attachment-preview:hover {
    transform: scale(1.05);
}

.attachment-preview img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.attachment-icon {
    width: 48px;
    height: 48px;
    display: flex;
    align-items: center;
    justify-content: center;
    background: #dbeafe;
    border-radius: 0.375rem;
    color: #3b82f6;
    flex-shrink: 0;
}

.attachment-icon svg {
    width: 24px;
    height: 24px;
}

.attachment-icon.file-pdf {
    background: #fef2f2;
    color: #dc2626;
}

.attachment-icon.file-word {
    background: #eff6ff;
    color: #2563eb;
}

.attachment-icon.file-excel {
    background: #f0fdf4;
    color: #16a34a;
}

.attachment-icon.file-archive {
    background: #fefce8;
    color: #ca8a04;
}

.attachment-icon.file-image {
    background: #fdf4ff;
    color: #c026d3;
}

.attachment-info {
    flex: 1;
    min-width: 0;
}

.attachment-name {
    font-size: 1rem;
    font-weight: 600;
    color: #111827;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
    margin-bottom: 4px;
}

.attachment-meta {
    display: flex;
    align-items: center;
    gap: 1rem;
    font-size: 0.875rem;
    color: #9ca3af;
}

.attachment-actions {
    display: flex;
    gap: 0.5rem;
}

.attachment-btn {
    width: 36px;
    height: 36px;
    display: flex;
    align-items: center;
    justify-content: center;
    background: #ffffff;
    border: 1px solid #e5e7eb;
    border-radius: 0.375rem;
    color: #6b7280;
    cursor: pointer;
    transition: all 0.2s ease;
    text-decoration: none;
}

.attachment-btn svg {
    width: 18px;
    height: 18px;
}

.attachment-btn.download:hover {
    background: #3b82f6;
    border-color: #3b82f6;
    color: white;
}

.attachment-btn.delete:hover {
    background: #ef4444;
    border-color: #ef4444;
    color: white;
}

/* Image Preview Modal */
.image-modal {
    display: none;
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.9);
    z-index: 9999;
    justify-content: center;
    align-items: center;
    padding: 20px;
}

.image-modal.active {
    display: flex;
}

.image-modal img {
    max-width: 90%;
    max-height: 90%;
    border-radius: 0.375rem;
    box-shadow: 0 20px 60px rgba(0, 0, 0, 0.5);
}

.image-modal-close {
    position: absolute;
    top: 20px;
    right: 20px;
    width: 44px;
    height: 44px;
    background: white;
    border: none;
    border-radius: 50%;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: transform 0.2s ease;
}

.image-modal-close:hover {
    transform: scale(1.1);
}

.image-modal-close svg {
    width: 24px;
    height: 24px;
}
</style>

<div class="task-detail-page">
    {{-- Page Header --}}
    <div class="page-header">
        <div class="page-title-section">
            <a href="{{ route('admin.tasks.index') }}" class="btn-back">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M15 19l-7-7 7-7"/>
                </svg>
            </a>
            <div class="title-content">
                <h1 class="page-title">{{ $task->title }}</h1>
                <p class="page-subtitle">
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M7 7h.01M7 3h5c.512 0 1.024.195 1.414.586l7 7a2 2 0 010 2.828l-7 7a2 2 0 01-2.828 0l-7-7A1.994 1.994 0 013 12V7a4 4 0 014-4z"/>
                    </svg>
                    Task #{{ $task->id }}
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24" style="width: 4px; height: 4px;">
                        <circle cx="12" cy="12" r="2" fill="currentColor"/>
                    </svg>
                    {{ $task->project?->title ?? 'No Project' }}
                </p>
            </div>
        </div>

        <div class="action-buttons">
            <a href="{{ route('admin.tasks.edit', $task->id) }}" class="btn btn-secondary">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"/>
                </svg>
                Edit Task
            </a>
            
            {{-- Timer Button with Modal --}}
            <div class="timer-button-wrapper">
                @if($task->has_completed_timer)
                    {{-- Timer Already Completed - Show View Button --}}
                    <button id="timerToggleBtn" class="btn btn-success" style="background: #10b981;">
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                        </svg>
                        <span id="timerToggleBtnText">View Time Record</span>
                    </button>
                @elseif($task->has_active_timer)
                    {{-- Timer Active - Show View Timer Button --}}
                    <button id="timerToggleBtn" class="btn btn-primary">
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                        </svg>
                        <span id="timerToggleBtnText">View Timer</span>
                    </button>
                @else
                    {{-- No Timer - Show Start Button --}}
                    <button id="timerToggleBtn" class="btn btn-primary">
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                        </svg>
                        <span id="timerToggleBtnText">Start Timer</span>
                    </button>
                @endif

                {{-- Timer Modal --}}
                <div id="timerModal" class="timer-modal hidden">
                    {{-- Loading State --}}
                    <div id="timerLoading" class="timer-loading hidden">
                        <div class="spinner"></div>
                        <div class="timer-loading-text">Getting location...</div>
                    </div>

                    {{-- Error State --}}
                    <div id="timerError" class="timer-error hidden">
                        <div class="error-message">
                            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                            </svg>
                            <span id="timerErrorText"></span>
                        </div>
                    </div>

                    {{-- Completed Timer Display (for viewing completed record) --}}
                    @if($task->has_completed_timer && $task->completed_timer)
                    <div id="timerCompleted">
                        <div class="timer-display">
                            <div class="timer-label" style="color: #10b981;">✓ Time Record Completed</div>
                        </div>
                        <div class="timer-info" style="text-align: left;">
                            <div class="timer-info-item">
                                <strong>Started:</strong> {{ $task->completed_timer->start_time->format('d M Y, h:i:s A') }}
                            </div>
                            <div class="timer-info-item">
                                <strong>Finished:</strong> {{ $task->completed_timer->end_time->format('d M Y, h:i:s A') }}
                            </div>
                            <div class="timer-info-item">
                                <strong>Working Time:</strong> 
                                @php
                                    $workingSec = $task->completed_timer->start_time->diffInSeconds($task->completed_timer->end_time) - ($task->completed_timer->total_pause_seconds ?? 0);
                                    $h = floor($workingSec / 3600);
                                    $m = floor(($workingSec % 3600) / 60);
                                    $s = $workingSec % 60;
                                @endphp
                                {{ sprintf('%02d:%02d:%02d', $h, $m, $s) }}
                            </div>
                            @if($task->completed_timer->total_pause_seconds > 0)
                            <div class="timer-info-item">
                                <strong>Paused:</strong>
                                @php
                                    $pauseSec = $task->completed_timer->total_pause_seconds;
                                    $ph = floor($pauseSec / 3600);
                                    $pm = floor(($pauseSec % 3600) / 60);
                                    $ps = $pauseSec % 60;
                                @endphp
                                {{ sprintf('%02d:%02d:%02d', $ph, $pm, $ps) }}
                                ({{ $task->completed_timer->pauses ? $task->completed_timer->pauses->count() : 0 }} pause(s))
                            </div>
                            @endif
                            @if($task->completed_timer->has_location)
                            <div class="timer-info-item">
                                <strong>Start Location:</strong>
                                <a href="{{ $task->completed_timer->google_maps_url }}" target="_blank" style="color: #3b82f6;">
                                    View on Map
                                </a>
                            </div>
                            @endif
                            @if($task->completed_timer->has_end_location)
                            <div class="timer-info-item">
                                <strong>End Location:</strong>
                                <a href="{{ $task->completed_timer->end_google_maps_url }}" target="_blank" style="color: #3b82f6;">
                                    View on Map
                                </a>
                            </div>
                            @endif
                        </div>
                    </div>
                    @endif

                    {{-- Active Timer Display --}}
                    <div id="timerActive" class="hidden">
                        <div class="timer-display">
                            <div class="timer-label" id="timerStatusLabel">Timer Running</div>
                            <div id="timerTimeDisplay" class="timer-time running">
                                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24" style="width: 32px; height: 32px; animation: pulse 2s infinite;">
                                    <circle cx="12" cy="12" r="10" stroke-width="2"/>
                                    <polyline points="12,6 12,12 16,14"/>
                                </svg>
                            </div>
                        </div>

                        <div class="timer-info">
                            <div class="timer-info-item">
                                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                                </svg>
                                <span id="timerStartTime">Started at --:--</span>
                            </div>
                            <div class="timer-info-item">
                                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M10 9v6m4-6v6"/>
                                </svg>
                                <span id="timerPauseCount">0 pauses</span>
                            </div>
                            <div class="timer-info-item">
                                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z"/>
                                </svg>
                                <span id="timerLocation">Location recorded</span>
                            </div>
                        </div>

                        <div class="timer-controls">
                            {{-- Pause Button (visible when running) --}}
                            <button id="pauseBtn" class="timer-btn timer-btn-pause">
                                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M10 9v6m4-6v6m7-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                                </svg>
                                Pause
                            </button>
                            
                            {{-- Resume Button (visible when paused) --}}
                            <button id="resumeBtn" class="timer-btn timer-btn-resume hidden">
                                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M14.752 11.168l-3.197-2.132A1 1 0 0010 9.87v4.263a1 1 0 001.555.832l3.197-2.132a1 1 0 000-1.664z"/>
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                                </svg>
                                Resume
                            </button>
                            
                            {{-- Finish Button --}}
                            <button id="stopBtn" class="timer-btn timer-btn-stop">
                                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                                </svg>
                                Finish
                            </button>
                        </div>
                    </div>

                    {{-- Success State --}}
                    <div id="timerSuccess" class="timer-success hidden">
                        <svg class="success-icon" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                        </svg>
                        <div class="success-message">Timer Finished!</div>
                        <div class="success-detail" id="successDetail">
                            <div>Working Time: <span id="workingTime">00:00:00</span></div>
                            <div style="font-size: 12px; color: #9ca3af; margin-top: 4px;">
                                Total: <span id="totalTime">00:00:00</span> | 
                                Paused: <span id="pauseTime">00:00:00</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    {{-- Task Details Card --}}
    <div class="info-card">
        <h3 class="card-title">Task Details</h3>
        <div class="info-grid">
            <div class="info-item">
                <div class="info-label">Status</div>
                <div class="info-value">
                    <span class="status-badge" style="background: {{ $task->status->color }}20; color: {{ $task->status->color }};">
                        <svg fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"/>
                        </svg>
                        {{ $task->status->label }}
                    </span>
                </div>
            </div>

            <div class="info-item">
                <div class="info-label">Priority</div>
                <div class="info-value">
                    <span class="priority-badge priority-{{ $task->priority }}">
                        @if($task->priority === 'critical')
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"/>
                        </svg>
                        @elseif($task->priority === 'high')
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M13 10V3L4 14h7v7l9-11h-7z"/>
                        </svg>
                        @elseif($task->priority === 'medium')
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                        </svg>
                        @else
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                        </svg>
                        @endif
                        {{ ucfirst($task->priority) }}
                    </span>
                </div>
            </div>

            @if($task->due_date)
            <div class="info-item">
                <div class="info-label">Due Date</div>
                <div class="info-value">{{ $task->due_date->format('M d, Y') }}</div>
            </div>
            @endif

            @if($task->estimated_hours)
            <div class="info-item">
                <div class="info-label">Estimated Hours</div>
                <div class="info-value">{{ $task->estimated_hours }}h</div>
            </div>
            @endif
        </div>

        @if($task->description)
        <div class="description-content">
            <div class="info-label">Description</div>
            <div class="description-text">{{ $task->description }}</div>
        </div>
        @endif
    </div>

    {{-- Related Entity Card (Project/Service/Contract/Support) --}}
    @php
        // Auto-detect task_type if not set but has related ID
        $detectedTaskType = $task->task_type ?? 'general';
        if ($detectedTaskType == 'general' || !$detectedTaskType) {
            if ($task->project_id) $detectedTaskType = 'project';
            elseif ($task->service_id) $detectedTaskType = 'service';
            elseif ($task->contract_id) $detectedTaskType = 'contract';
            elseif ($task->support_ticket_id) $detectedTaskType = 'support';
        }
    @endphp
    @if($detectedTaskType !== 'general')
    <div class="info-card">
        <h3 class="card-title">
            @if($detectedTaskType == 'project')
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24" style="width: 20px; height: 20px; margin-right: 8px; color: #3b82f6;">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M3 7v10a2 2 0 002 2h14a2 2 0 002-2V9a2 2 0 00-2-2h-6l-2-2H5a2 2 0 00-2 2z"/>
                </svg>
                Project Details
            @elseif($detectedTaskType == 'service')
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24" style="width: 20px; height: 20px; margin-right: 8px; color: #10b981;">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z"/>
                    <path stroke-linecap="round" stroke-linejoin="round" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/>
                </svg>
                Service Details
            @elseif($detectedTaskType == 'contract')
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24" style="width: 20px; height: 20px; margin-right: 8px; color: #f59e0b;">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/>
                </svg>
                Contract Details
            @elseif($detectedTaskType == 'support')
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24" style="width: 20px; height: 20px; margin-right: 8px; color: #8b5cf6;">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M18.364 5.636l-3.536 3.536m0 5.656l3.536 3.536M9.172 9.172L5.636 5.636m3.536 9.192l-3.536 3.536M21 12a9 9 0 11-18 0 9 9 0 0118 0zm-5 0a4 4 0 11-8 0 4 4 0 018 0z"/>
                </svg>
                Support Ticket Details
            @endif
        </h3>
        
        <div class="info-grid">
            {{-- Task Type --}}
            <div class="info-item">
                <div class="info-label">Task Type</div>
                <div class="info-value">
                    <span class="badge" style="background: #dbeafe; color: #3b82f6; padding: 4px 12px; border-radius: 20px; font-weight: 600;">
                        {{ ucfirst($detectedTaskType) }} Task
                    </span>
                </div>
            </div>
            
            @if($detectedTaskType == 'project')
                {{-- Project Info --}}
                <div class="info-item">
                    <div class="info-label">Project</div>
                    <div class="info-value">
                        @if($task->project)
                            <a href="{{ route('admin.projects.show', $task->project_id) }}" style="color: #3b82f6; text-decoration: none; font-weight: 600;">
                                #{{ $task->project_id }} - {{ $task->project->title ?? $task->project->name ?? $task->related_name }}
                            </a>
                        @else
                            {{ $task->related_name ?? 'Project #'.$task->project_id }}
                        @endif
                    </div>
                </div>
                
                @if($task->project)
                    @if($task->project->status ?? null)
                    <div class="info-item">
                        <div class="info-label">Project Status</div>
                        <div class="info-value">
                            @php
                                $projectStatus = $task->project->status;
                                $statusColor = '#3b82f6';
                                if (is_object($projectStatus)) {
                                    $statusLabel = $projectStatus->label ?? $projectStatus->name ?? 'Unknown';
                                    $statusColor = $projectStatus->color ?? '#3b82f6';
                                } elseif (is_array($projectStatus)) {
                                    $statusLabel = $projectStatus['label'] ?? $projectStatus['name'] ?? 'Unknown';
                                    $statusColor = $projectStatus['color'] ?? '#3b82f6';
                                } else {
                                    $statusLabel = ucfirst($projectStatus);
                                }
                            @endphp
                            <span style="background: {{ $statusColor }}20; color: {{ $statusColor }}; padding: 4px 12px; border-radius: 20px; font-weight: 600; font-size: 12px;">
                                {{ $statusLabel }}
                            </span>
                        </div>
                    </div>
                    @endif
                    
                    @if($task->project->customer ?? null)
                    <div class="info-item">
                        <div class="info-label">Customer</div>
                        <div class="info-value">
                            <a href="{{ route('admin.customers.show', $task->project->customer->id ?? 0) }}" style="color: #111827; text-decoration: none;">
                                {{ $task->project->customer->company ?? $task->project->customer->name ?? 'N/A' }}
                            </a>
                        </div>
                    </div>
                    @endif
                    
                    @if($task->project->start_date ?? null)
                    <div class="info-item">
                        <div class="info-label">Project Start Date</div>
                        <div class="info-value">{{ \Carbon\Carbon::parse($task->project->start_date)->format('M d, Y') }}</div>
                    </div>
                    @endif
                    
                    @if($task->project->deadline ?? $task->project->due_date ?? $task->project->end_date ?? null)
                    <div class="info-item">
                        <div class="info-label">Project Deadline</div>
                        <div class="info-value">
                            @php
                                $deadline = $task->project->deadline ?? $task->project->due_date ?? $task->project->end_date;
                                $deadlineDate = \Carbon\Carbon::parse($deadline);
                                $isOverdue = $deadlineDate->isPast();
                            @endphp
                            <span style="{{ $isOverdue ? 'color: #ef4444;' : '' }}">
                                {{ $deadlineDate->format('M d, Y') }}
                                @if($isOverdue)
                                    <span style="font-size: 11px;">(Overdue)</span>
                                @endif
                            </span>
                        </div>
                    </div>
                    @endif
                    
                    @if(isset($task->project->progress))
                    <div class="info-item">
                        <div class="info-label">Project Progress</div>
                        <div class="info-value">
                            <div style="display: flex; align-items: center; gap: 10px;">
                                <div style="flex: 1; height: 8px; background: #e5e7eb; border-radius: 4px; overflow: hidden;">
                                    <div style="width: {{ $task->project->progress }}%; height: 100%; background: #10b981; border-radius: 4px;"></div>
                                </div>
                                <span style="font-weight: 600; min-width: 40px;">{{ $task->project->progress }}%</span>
                            </div>
                        </div>
                    </div>
                    @endif
                    
                    @if($task->project->billing_type ?? null)
                    <div class="info-item">
                        <div class="info-label">Billing Type</div>
                        <div class="info-value">{{ ucfirst(str_replace('_', ' ', $task->project->billing_type)) }}</div>
                    </div>
                    @endif
                    
                    @if($task->project->hourly_rate ?? null)
                    <div class="info-item">
                        <div class="info-label">Hourly Rate</div>
                        <div class="info-value">₹{{ number_format($task->project->hourly_rate, 2) }}</div>
                    </div>
                    @endif
                    
                    @if($task->project->total_budget ?? $task->project->project_cost ?? null)
                    <div class="info-item">
                        <div class="info-label">Project Budget</div>
                        <div class="info-value">₹{{ number_format($task->project->total_budget ?? $task->project->project_cost, 2) }}</div>
                    </div>
                    @endif
                    
                    @if($task->project->description ?? null)
                    <div class="info-item" style="grid-column: 1 / -1;">
                        <div class="info-label">Project Description</div>
                        <div class="info-value" style="color: #6b7280; font-size: 13px; line-height: 1.5;">
                            {{ \Illuminate\Support\Str::limit(strip_tags($task->project->description), 200) }}
                        </div>
                    </div>
                    @endif
                @endif
                
            @elseif($detectedTaskType == 'service')
                {{-- Service Info --}}
                <div class="info-item">
                    <div class="info-label">Service</div>
                    <div class="info-value">
                        @if($task->service)
                            <a href="{{ route('admin.services.show', $task->service_id) }}" style="color: #10b981; text-decoration: none; font-weight: 600;">
                                #{{ $task->service_id }} - {{ $task->service->name ?? $task->related_name }}
                            </a>
                        @else
                            {{ $task->related_name ?? 'Service #'.$task->service_id }}
                        @endif
                    </div>
                </div>
                
                @if($task->service)
                    @if($task->service->price ?? null)
                    <div class="info-item">
                        <div class="info-label">Service Price</div>
                        <div class="info-value">{{ number_format($task->service->price, 2) }}</div>
                    </div>
                    @endif
                @endif
                
            @elseif($detectedTaskType == 'contract')
                {{-- Contract Info --}}
                <div class="info-item">
                    <div class="info-label">Contract</div>
                    <div class="info-value">
                        @if($task->contract)
                            <a href="{{ route('admin.contracts.show', $task->contract_id) }}" style="color: #f59e0b; text-decoration: none; font-weight: 600;">
                                #{{ $task->contract_id }} - {{ $task->contract->subject ?? $task->related_name }}
                            </a>
                        @else
                            {{ $task->related_name ?? 'Contract #'.$task->contract_id }}
                        @endif
                    </div>
                </div>
                
                @if($task->contract)
                    @if($task->contract->contract_value ?? null)
                    <div class="info-item">
                        <div class="info-label">Contract Value</div>
                        <div class="info-value">{{ number_format($task->contract->contract_value, 2) }}</div>
                    </div>
                    @endif
                    
                    @if($task->contract->start_date ?? null)
                    <div class="info-item">
                        <div class="info-label">Contract Period</div>
                        <div class="info-value">
                            {{ \Carbon\Carbon::parse($task->contract->start_date)->format('M d, Y') }}
                            @if($task->contract->end_date ?? null)
                                - {{ \Carbon\Carbon::parse($task->contract->end_date)->format('M d, Y') }}
                            @endif
                        </div>
                    </div>
                    @endif
                @endif
                
            @elseif($detectedTaskType == 'support')
                {{-- Support Ticket Info --}}
                <div class="info-item">
                    <div class="info-label">Support Ticket</div>
                    <div class="info-value">
                        @if($task->supportTicket)
                            <a href="{{ route('admin.support.tickets.show', $task->support_ticket_id) }}" style="color: #8b5cf6; text-decoration: none; font-weight: 600;">
                                [{{ $task->supportTicket->ticket_key ?? $task->support_ticket_id }}] {{ $task->supportTicket->subject ?? $task->related_name }}
                            </a>
                        @else
                            {{ $task->related_name ?? 'Ticket #'.$task->support_ticket_id }}
                        @endif
                    </div>
                </div>
                
                @if($task->supportTicket)
                    @if($task->supportTicket->status_id ?? $task->supportTicket->status ?? null)
                    <div class="info-item">
                        <div class="info-label">Ticket Status</div>
                        <div class="info-value">
                            @php
                                $ticketStatus = $task->supportTicket->status ?? null;
                                $statusName = 'Unknown';
                                if (is_object($ticketStatus)) {
                                    $statusName = $ticketStatus->name ?? $ticketStatus->label ?? 'Unknown';
                                } elseif (is_string($ticketStatus)) {
                                    $statusName = ucfirst($ticketStatus);
                                } elseif ($task->supportTicket->status_id) {
                                    $statusName = 'Status #' . $task->supportTicket->status_id;
                                }
                            @endphp
                            <span style="background: #8b5cf620; color: #8b5cf6; padding: 4px 12px; border-radius: 20px; font-weight: 600; font-size: 12px;">
                                {{ $statusName }}
                            </span>
                        </div>
                    </div>
                    @endif
                    
                    @if($task->supportTicket->priority_id ?? $task->supportTicket->priority ?? null)
                    <div class="info-item">
                        <div class="info-label">Ticket Priority</div>
                        <div class="info-value">
                            @php
                                $ticketPriority = $task->supportTicket->priority ?? null;
                                $priorityName = 'Unknown';
                                if (is_object($ticketPriority)) {
                                    $priorityName = $ticketPriority->name ?? $ticketPriority->label ?? 'Unknown';
                                } elseif (is_string($ticketPriority)) {
                                    $priorityName = ucfirst($ticketPriority);
                                } elseif ($task->supportTicket->priority_id) {
                                    $priorityName = 'Priority #' . $task->supportTicket->priority_id;
                                }
                            @endphp
                            {{ $priorityName }}
                        </div>
                    </div>
                    @endif
                    
                    @if($task->supportTicket->department_id ?? null)
                    <div class="info-item">
                        <div class="info-label">Department</div>
                        <div class="info-value">
                            @php
                                $department = $task->supportTicket->department ?? null;
                                $deptName = 'Unknown';
                                if (is_object($department)) {
                                    $deptName = $department->name ?? 'Unknown';
                                } elseif ($task->supportTicket->department_id) {
                                    $deptName = 'Department #' . $task->supportTicket->department_id;
                                }
                            @endphp
                            {{ $deptName }}
                        </div>
                    </div>
                    @endif
                    
                    @if($task->supportTicket->customer_id ?? null)
                    <div class="info-item">
                        <div class="info-label">Customer</div>
                        <div class="info-value">
                            {{ $task->supportTicket->name ?? $task->supportTicket->email ?? 'Customer #'.$task->supportTicket->customer_id }}
                        </div>
                    </div>
                    @endif
                    
                    @if($task->supportTicket->created_at ?? null)
                    <div class="info-item">
                        <div class="info-label">Ticket Created</div>
                        <div class="info-value">{{ \Carbon\Carbon::parse($task->supportTicket->created_at)->format('M d, Y h:i A') }}</div>
                    </div>
                    @endif
                    
                    @if($task->supportTicket->message ?? null)
                    <div class="info-item" style="grid-column: 1 / -1;">
                        <div class="info-label">Ticket Message</div>
                        <div class="info-value" style="color: #6b7280; font-size: 13px; line-height: 1.5;">
                            {{ \Illuminate\Support\Str::limit(strip_tags($task->supportTicket->message), 200) }}
                        </div>
                    </div>
                    @endif
                @endif
            @endif
        </div>
    </div>
    @endif

    {{-- Assigned To Card --}}
    @if($task->assignees && $task->assignees->count() > 0)
    <div class="info-card">
        <h3 class="card-title">Assigned To</h3>
        <div class="owner-tags">
            @foreach($task->assignees as $owner)
            <div class="owner-tag">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"/>
                </svg>
                {{ $owner->name }}
            </div>
            @endforeach
        </div>
    </div>
    @endif

    {{-- Attachments Card --}}
    @if($task->attachments && $task->attachments->count() > 0)
    <div class="info-card">
        <h3 class="card-title">
            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24" style="width: 20px; height: 20px; margin-right: 8px;">
                <path stroke-linecap="round" stroke-linejoin="round" d="M15.172 7l-6.586 6.586a2 2 0 102.828 2.828l6.414-6.586a4 4 0 00-5.656-5.656l-6.415 6.585a6 6 0 108.486 8.486L20.5 13"/>
            </svg>
            Attachments ({{ $task->attachments->count() }})
        </h3>
        <div class="attachments-grid">
            @foreach($task->attachments as $attachment)
            <div class="attachment-item">
                @if($attachment->is_image)
                <div class="attachment-preview" onclick="openImageModal('{{ route('admin.tasks.preview-attachment', $attachment->id) }}')">
                    <img src="{{ route('admin.tasks.preview-attachment', $attachment->id) }}" alt="{{ $attachment->filename }}">
                </div>
                @else
                <div class="attachment-icon {{ $attachment->file_icon }}">
                    @if($attachment->file_icon == 'file-pdf')
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M7 21h10a2 2 0 002-2V9.414a1 1 0 00-.293-.707l-5.414-5.414A1 1 0 0012.586 3H7a2 2 0 00-2 2v14a2 2 0 002 2z"/>
                    </svg>
                    @elseif($attachment->file_icon == 'file-image')
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"/>
                    </svg>
                    @elseif($attachment->file_icon == 'file-archive')
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M5 8h14M5 8a2 2 0 110-4h14a2 2 0 110 4M5 8v10a2 2 0 002 2h10a2 2 0 002-2V8m-9 4h4"/>
                    </svg>
                    @else
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/>
                    </svg>
                    @endif
                </div>
                @endif
                <div class="attachment-info">
                    <div class="attachment-name" title="{{ $attachment->filename }}">{{ $attachment->filename }}</div>
                    <div class="attachment-meta">
                        <span class="attachment-size">{{ $attachment->file_size_human }}</span>
                        <span class="attachment-date">{{ $attachment->created_at->format('M d, Y') }}</span>
                    </div>
                </div>
                <div class="attachment-actions">
                    <a href="{{ route('admin.tasks.download-attachment', $attachment->id) }}" class="attachment-btn download" title="Download">
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-4l-4 4m0 0l-4-4m4 4V4"/>
                        </svg>
                    </a>
                    <button type="button" class="attachment-btn delete" title="Delete" onclick="deleteAttachment({{ $attachment->id }})">
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/>
                        </svg>
                    </button>
                </div>
            </div>
            @endforeach
        </div>
    </div>
    @endif
</div>

{{-- Image Preview Modal --}}
<div class="image-modal" id="imageModal" onclick="closeImageModal()">
    <button class="image-modal-close" onclick="closeImageModal()">
        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" d="M6 18L18 6M6 6l12 12"/>
        </svg>
    </button>
    <img src="" alt="Preview" id="modalImage">
</div>

<script>
// ============================================================
// TIMER APP - COMPLETELY REWRITTEN WITH PROPER STATE MANAGEMENT
// ============================================================

const TimerApp = {
    // Application State
    taskId: {{ $task->id }},
    timerId: null,
    startTime: null,
    elapsedSeconds: 0,
    intervalId: null,
    isPaused: false,
    pauseCount: 0,
    isCompleted: false,
    
    // DOM Elements Cache
    elements: {
        toggleBtn: document.getElementById('timerToggleBtn'),
        toggleBtnText: document.getElementById('timerToggleBtnText'),
        modal: document.getElementById('timerModal'),
        loading: document.getElementById('timerLoading'),
        error: document.getElementById('timerError'),
        errorText: document.getElementById('timerErrorText'),
        active: document.getElementById('timerActive'),
        success: document.getElementById('timerSuccess'),
        timeDisplay: document.getElementById('timerTimeDisplay'),
        statusLabel: document.getElementById('timerStatusLabel'),
        startTimeDisplay: document.getElementById('timerStartTime'),
        pauseCountDisplay: document.getElementById('timerPauseCount'),
        locationDisplay: document.getElementById('timerLocation'),
        successDetail: document.getElementById('successDetail'),
        workingTime: document.getElementById('workingTime'),
        totalTime: document.getElementById('totalTime'),
        pauseTime: document.getElementById('pauseTime'),
        pauseBtn: document.getElementById('pauseBtn'),
        resumeBtn: document.getElementById('resumeBtn'),
        stopBtn: document.getElementById('stopBtn'),
    },

    // ==================== INITIALIZATION ====================
    
    init() {
        console.log('🚀 Timer App Initializing...');
        this.bindEvents();
        this.checkForActiveTimer();
        console.log('✅ Timer App Ready');
    },

    bindEvents() {
        // Main button click
        this.elements.toggleBtn.addEventListener('click', () => this.handleToggleClick());
        
        // Timer control buttons - only if they exist
        if (this.elements.pauseBtn) {
            this.elements.pauseBtn.addEventListener('click', () => this.pauseTimer());
        }
        if (this.elements.resumeBtn) {
            this.elements.resumeBtn.addEventListener('click', () => this.resumeTimer());
        }
        if (this.elements.stopBtn) {
            this.elements.stopBtn.addEventListener('click', () => this.finishTimer());
        }
        
        // Click outside to close
        document.addEventListener('click', (e) => {
            const isOutsideClick = !this.elements.toggleBtn.contains(e.target) && 
                                   !this.elements.modal.contains(e.target);
            const modalIsOpen = !this.elements.modal.classList.contains('hidden');
            
            if (isOutsideClick && modalIsOpen) {
                this.hideModal();
            }
        });
    },

    // Check if there's an existing timer on page load
    checkForActiveTimer() {
        const hasActiveTimer = {{ $task->has_active_timer ? 'true' : 'false' }};
        const hasCompletedTimer = {{ $task->has_completed_timer ? 'true' : 'false' }};
        
        if (hasCompletedTimer) {
            // Timer already completed - mark as completed
            this.isCompleted = true;
            console.log('✅ Timer already completed for this task');
        } else if (hasActiveTimer) {
            const timerData = @json($task->active_timer);
            
            if (timerData) {
                console.log('📌 Found active timer:', timerData);
                
                // Set state from database
                this.timerId = timerData.id;
                this.startTime = new Date(timerData.start_time);
                this.isPaused = timerData.is_paused || false;
                this.pauseCount = timerData.pause_count || 0;
                
                console.log('✅ Timer state loaded:', {
                    isPaused: this.isPaused,
                    pauseCount: this.pauseCount
                });
            }
        }
    },

    // ==================== MODAL CONTROL ====================
    
    handleToggleClick() {
        console.log('🖱️ Toggle button clicked, timerId:', this.timerId);
        
        if (this.isCompleted) {
            // Timer already completed - just show the modal with completed info
            console.log('✅ Showing completed timer record');
            this.showModal();
            // The completed info is rendered by Blade, no JS needed
        } else if (this.timerId) {
            // Timer exists and active - show it in the modal
            console.log('👁️ Showing existing timer');
            this.showModal();
            this.renderActiveTimer();
        } else {
            // No timer - start a new one
            console.log('▶️ Starting new timer');
            this.showModal();
            this.startNewTimer();
        }
    },

    showModal() {
        console.log('📂 Opening modal');
        this.elements.modal.classList.remove('hidden');
    },

    hideModal() {
        console.log('📁 Closing modal');
        this.elements.modal.classList.add('hidden');
        if (!this.isCompleted) {
            this.hideAllStates();
        }
    },

    // ==================== STATE MANAGEMENT (CRITICAL!) ====================
    
    // Hide ALL states before showing one
    hideAllStates() {
        if (this.elements.loading) this.elements.loading.classList.add('hidden');
        if (this.elements.error) this.elements.error.classList.add('hidden');
        this.elements.active.classList.add('hidden');
        this.elements.success.classList.add('hidden');
    },

    showLoadingState(message) {
        console.log('⏳ Loading:', message);
        this.hideAllStates(); // ← CRITICAL: Hide everything first
        this.elements.loading.querySelector('.timer-loading-text').textContent = message;
        this.elements.loading.classList.remove('hidden');
    },

    showErrorState(message) {
        console.log('❌ Error:', message);
        this.hideAllStates(); // ← CRITICAL: Hide everything first
        this.elements.errorText.textContent = message;
        this.elements.error.classList.remove('hidden');
    },

    showSuccessState() {
        console.log('✅ Success state shown');
        this.hideAllStates();
        this.elements.success.classList.remove('hidden');
    },

    showActiveTimerState() {
        console.log('⏱️ Showing active timer');
        this.hideAllStates(); // ← CRITICAL: Hide everything first
        this.elements.active.classList.remove('hidden');
    },

    // ==================== TIMER ACTIONS ====================
    
async startNewTimer() {
        this.showLoadingState('Getting your location...');

        try {
            // Get GPS location with retry for accuracy
            const position = await this.getGPSLocation();
            const accuracy = Math.round(position.coords.accuracy);
            console.log(`✅ Location obtained: ${position.coords.latitude}, ${position.coords.longitude} (±${accuracy}m)`);
            
            this.showLoadingState(`Location found (±${accuracy}m). Starting timer...`);

            // Send to backend
            const response = await fetch(`/admin/tasks/${this.taskId}/start-timer`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}',
                    'Accept': 'application/json'
                },
                body: JSON.stringify({
                    latitude: position.coords.latitude,
                    longitude: position.coords.longitude,
                    location_accuracy: position.coords.accuracy
                })
            });

            // Check if response is actually JSON
            const contentType = response.headers.get('content-type');
            if (!contentType || !contentType.includes('application/json')) {
                const text = await response.text();
                console.error('❌ Server returned HTML instead of JSON:', text.substring(0, 200));
                throw new Error('Server error: Expected JSON response but got HTML. Please check if you are logged in.');
            }

            const data = await response.json();
            console.log('📦 Server response:', data);

            if (!response.ok || !data.success) {
                throw new Error(data.message || 'Failed to start timer');
            }
            
            // Update state
            this.timerId = data.data.timer_id;
            this.startTime = new Date(data.data.start_time);
            this.locationAccuracy = accuracy;

            // Update UI
            this.elements.toggleBtnText.textContent = 'View Timer';
            this.renderActiveTimer();

            console.log('✅ Timer started successfully');

        } catch (error) {
            console.error('❌ Start timer error:', error);
            this.showErrorState(error.message);
        }
    },

    // ==================== UI RENDERING ====================
    
    renderActiveTimer() {
        this.showActiveTimerState();

        // Update start time display
        if (this.startTime) {
            const timeStr = this.startTime.toLocaleTimeString('en-US', {
                hour: '2-digit',
                minute: '2-digit',
                hour12: true
            });
            const dateStr = this.startTime.toLocaleDateString('en-US', {
                month: 'short',
                day: 'numeric'
            });
            this.elements.startTimeDisplay.textContent = `Started: ${dateStr} at ${timeStr}`;
        }

        // Update status and button visibility based on paused state
        if (this.isPaused) {
            this.elements.statusLabel.textContent = 'Timer Paused';
            this.elements.timeDisplay.classList.remove('running');
            this.elements.timeDisplay.classList.add('paused');
            this.elements.pauseBtn.classList.add('hidden');
            this.elements.resumeBtn.classList.remove('hidden');
        } else {
            this.elements.statusLabel.textContent = 'Timer Running';
            this.elements.timeDisplay.classList.add('running');
            this.elements.timeDisplay.classList.remove('paused');
            this.elements.pauseBtn.classList.remove('hidden');
            this.elements.resumeBtn.classList.add('hidden');
        }

        // Update pause count
        this.elements.pauseCountDisplay.textContent = `${this.pauseCount || 0} pause(s)`;
        
        // Location is already recorded at start
        this.elements.locationDisplay.textContent = 'Location recorded';
    },

    // ==================== PAUSE TIMER ====================
    
    async pauseTimer() {
        console.log('⏸️ Pausing timer...');
        this.showLoadingState('Getting location...');

        try {
            // Get current location for pause
            let location = {};
            try {
                const position = await this.getGPSLocation();
                location = {
                    latitude: position.coords.latitude,
                    longitude: position.coords.longitude,
                    accuracy: position.coords.accuracy
                };
                console.log('📍 Pause location:', location);
            } catch (locError) {
                console.warn('⚠️ Could not get pause location:', locError.message);
            }

            this.showLoadingState('Pausing timer...');

            const response = await fetch(`/admin/tasks/${this.taskId}/pause-timer`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}',
                    'Accept': 'application/json'
                },
                body: JSON.stringify(location)
            });

            const data = await response.json();

            if (!response.ok || !data.success) {
                throw new Error(data.message || 'Failed to pause timer');
            }

            // Update state
            this.isPaused = true;
            this.pauseCount = data.data.pause_count || (this.pauseCount + 1);

            // Update UI
            this.renderActiveTimer();

            console.log('✅ Timer paused');

        } catch (error) {
            console.error('❌ Pause error:', error);
            this.showErrorState(error.message);
        }
    },

    // ==================== RESUME TIMER ====================
    
    async resumeTimer() {
        console.log('▶️ Resuming timer...');
        this.showLoadingState('Getting location...');

        try {
            // Get current location for resume
            let location = {};
            try {
                const position = await this.getGPSLocation();
                location = {
                    latitude: position.coords.latitude,
                    longitude: position.coords.longitude,
                    accuracy: position.coords.accuracy
                };
                console.log('📍 Resume location:', location);
            } catch (locError) {
                console.warn('⚠️ Could not get resume location:', locError.message);
            }

            this.showLoadingState('Resuming timer...');

            const response = await fetch(`/admin/tasks/${this.taskId}/resume-timer`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}',
                    'Accept': 'application/json'
                },
                body: JSON.stringify(location)
            });

            const data = await response.json();

            if (!response.ok || !data.success) {
                throw new Error(data.message || 'Failed to resume timer');
            }

            // Update state
            this.isPaused = false;

            // Update UI
            this.renderActiveTimer();

            console.log('✅ Timer resumed');

        } catch (error) {
            console.error('❌ Resume error:', error);
            this.showErrorState(error.message);
        }
    },

    // ==================== FINISH TIMER ====================
    
    async finishTimer() {
        if (!confirm('Finish this timer and save the time log?')) {
            return;
        }

        console.log('🏁 Finishing timer...');
        this.showLoadingState('Getting end location...');

        try {
            // Get end location
            let endLocation = {};
            try {
                const position = await this.getGPSLocation();
                endLocation = {
                    latitude: position.coords.latitude,
                    longitude: position.coords.longitude,
                    accuracy: position.coords.accuracy
                };
                console.log('📍 End location captured:', endLocation);
            } catch (locError) {
                console.warn('⚠️ Could not get end location:', locError.message);
            }

            this.showLoadingState('Finishing timer...');

            const response = await fetch(`/admin/tasks/${this.taskId}/stop-timer`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}',
                    'Accept': 'application/json'
                },
                body: JSON.stringify(endLocation)
            });

            const data = await response.json();

            if (!response.ok || !data.success) {
                throw new Error(data.message || 'Failed to finish timer');
            }

            // Reset state
            this.timerId = null;
            this.startTime = null;
            this.isPaused = false;
            this.pauseCount = 0;

            // Update button
            this.elements.toggleBtnText.textContent = 'Start Timer';

            // Show success with detailed info
            this.elements.workingTime.textContent = data.data.formatted_working_time || '00:00:00';
            this.elements.totalTime.textContent = data.data.total_time || '00:00:00';
            this.elements.pauseTime.textContent = data.data.pause_time || '00:00:00';
            this.showSuccessState();

            console.log('✅ Timer finished:', data.data);

            // Auto-close and reload
            setTimeout(() => {
                this.hideModal();
                window.location.reload();
            }, 3000);

        } catch (error) {
            console.error('❌ Finish error:', error);
            this.showErrorState(error.message);
        }
    },

    // Keep old method name for compatibility
    async stopTimer() {
        return this.finishTimer();
    },

    // ==================== GPS LOCATION ====================
    
    getGPSLocation() {
        return new Promise((resolve, reject) => {
            if (!navigator.geolocation) {
                reject(new Error('GPS not supported by your browser'));
                return;
            }

            console.log('📍 Requesting GPS location...');
            
            navigator.geolocation.getCurrentPosition(
                (position) => {
                    const accuracy = position.coords.accuracy;
                    console.log('✅ GPS position obtained:', {
                        lat: position.coords.latitude,
                        lng: position.coords.longitude,
                        accuracy: accuracy + ' meters'
                    });
                    
                    // Warn if accuracy is poor (> 500 meters usually means IP/WiFi location)
                    if (accuracy > 500) {
                        console.warn('⚠️ Low GPS accuracy! Location may be approximate.');
                        console.warn('For better accuracy: Enable GPS on mobile, or go outdoors.');
                    }
                    
                    resolve(position);
                },
                (error) => {
                    let message = 'Location access denied';
                    switch(error.code) {
                        case error.PERMISSION_DENIED:
                            message = 'GPS permission denied. Please enable location access in your browser settings.';
                            break;
                        case error.POSITION_UNAVAILABLE:
                            message = 'GPS unavailable. Please check your device location services.';
                            break;
                        case error.TIMEOUT:
                            message = 'GPS timeout. Taking too long to get location. Please try again.';
                            break;
                        default:
                            message = 'GPS error: ' + error.message;
                    }
                    console.error('❌ GPS error (code ' + error.code + '):', message);
                    reject(new Error(message));
                },
                {
                    enableHighAccuracy: true,  // Use GPS hardware, not WiFi/Cell tower
                    timeout: 30000,            // Wait up to 30 seconds
                    maximumAge: 0              // NEVER use cached location - always get fresh
                }
            );
        });
    }
};

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    TimerApp.init();
});

// ========== IMAGE MODAL FUNCTIONS ==========
function openImageModal(src) {
    const modal = document.getElementById('imageModal');
    const img = document.getElementById('modalImage');
    img.src = src;
    modal.classList.add('active');
}

function closeImageModal() {
    const modal = document.getElementById('imageModal');
    modal.classList.remove('active');
}

// Close modal with Escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeImageModal();
    }
});

// ========== DELETE ATTACHMENT ==========
function deleteAttachment(attachmentId) {
    if (!confirm('Are you sure you want to delete this attachment?')) {
        return;
    }
    
    fetch(`/admin/tasks/{{ $task->id }}/attachment/${attachmentId}`, {
        method: 'DELETE',
        headers: {
            'X-CSRF-TOKEN': '{{ csrf_token() }}',
            'Content-Type': 'application/json',
            'Accept': 'application/json'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Reload page to refresh attachments
            window.location.reload();
        } else {
            alert(data.message || 'Failed to delete attachment');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Failed to delete attachment');
    });
}
</script>