<style>
/* ========================================
   Time Logs - Theme-Aware (GUIDE COMPLIANT)
   ======================================== */

/* Page Animation */
.timelogs-page {
    padding: 2rem;
    animation: fadeIn 0.4s ease;
}

@keyframes fadeIn {
    from { opacity: 0; transform: translateY(10px); }
    to { opacity: 1; transform: translateY(0); }
}

/* Slide In Animation */
@keyframes slideIn {
    from { opacity: 0; transform: translateX(-20px); }
    to { opacity: 1; transform: translateX(0); }
}

/* Pulse Animation (for running timers) */
@keyframes pulse {
    0%, 100% { transform: scale(1); opacity: 1; }
    50% { transform: scale(1.05); opacity: 0.8; }
}

/* ===== PAGE HEADER ===== */
.page-header {
    background: #3b82f6;
    padding: 1.5rem 2rem;
    border-radius: 0.5rem;
    margin-bottom: 1.5rem;
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
    transition: all 0.3s ease;
}

.header-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: 1.5rem;
}

.header-left h1 {
    color: white;
    font-size: 1.25rem;
    font-weight: 700;
    margin: 0 0 4px 0;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    letter-spacing: -0.02em;
}

.header-left h1 svg {
    width: 20px;
    height: 20px;
}

.header-left p {
    color: rgba(255, 255, 255, 0.9);
    font-size: 0.875rem;
    margin: 0;
}

/* Search Bar */
.header-search {
    flex: 1;
    max-width: 320px;
    position: relative;
}

.header-search svg {
    position: absolute;
    left: 12px;
    top: 50%;
    transform: translateY(-50%);
    width: 14px;
    height: 14px;
    color: rgba(255, 255, 255, 0.7);
    pointer-events: none;
    transition: all 0.3s ease;
}

.header-search input {
    width: 100%;
    padding: 8px 12px 8px 32px;
    border-radius: 0.375rem;
    border: 1px solid rgba(255, 255, 255, 0.3);
    font-size: 0.875rem;
    background: rgba(255, 255, 255, 0.15);
    color: white;
    transition: all 0.3s ease;
}

.header-search input::placeholder {
    color: rgba(255, 255, 255, 0.7);
}

.header-search input:focus {
    outline: none;
    background: white;
    color: #1a202c !important;
    border-color: white;
}

.header-search input:focus + svg {
    color: #3b82f6;
}

/* ===== STATS CARDS ===== */
.stats-grid {
    display: grid;
    grid-template-columns: repeat(4, 1fr);
    gap: 1rem;
    margin-bottom: 1.5rem;
}

.stat-card {
    background: #ffffff;
    border: 1px solid #e5e7eb;
    border-radius: 0.5rem;
    padding: 1rem 1.5rem;
    display: flex;
    align-items: center;
    gap: 1rem;
    transition: all 0.3s ease;
    animation: slideIn 0.3s ease;
}

.stat-card:nth-child(1) { animation-delay: 0.1s; }
.stat-card:nth-child(2) { animation-delay: 0.2s; }
.stat-card:nth-child(3) { animation-delay: 0.3s; }
.stat-card:nth-child(4) { animation-delay: 0.4s; }

.stat-card:hover {
    transform: translateY(-2px) scale(1.02);
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
}

.stat-icon {
    width: 36px;
    height: 36px;
    border-radius: 0.375rem;
    display: flex;
    align-items: center;
    justify-content: center;
    flex-shrink: 0;
    transition: all 0.3s ease;
}

.stat-card:hover .stat-icon {
    animation: pulse 1s ease infinite;
}

.stat-icon svg {
    width: 18px;
    height: 18px;
    color: white;
}

.stat-icon-primary { background: #3b82f6; }
.stat-icon-success { background: #10b981; }
.stat-icon-warning { background: #f59e0b; }

.stat-content h3 {
    font-size: 1.25rem;
    font-weight: 700;
    color: #111827;
    margin: 0;
    transition: all 0.3s ease;
}

.stat-card:hover .stat-content h3 {
    transform: scale(1.05);
}

.stat-content p {
    font-size: 0.75rem;
    color: #9ca3af;
    margin: 2px 0 0 0;
    text-transform: uppercase;
    letter-spacing: 0.05em;
    font-weight: 600;
}

/* ===== FILTERS BAR ===== */
.filters-card {
    background: #ffffff;
    border: 1px solid #e5e7eb;
    padding: 1.5rem;
    border-radius: 0.5rem;
    margin-bottom: 1.5rem;
    box-shadow: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
    animation: fadeIn 0.4s ease 0.5s backwards;
}

.filters-row {
    display: flex;
    gap: 1rem;
    align-items: center;
    flex-wrap: wrap;
}

.quick-filters {
    display: flex;
    gap: 0.5rem;
}

.filter-btn {
    padding: 8px 14px;
    border: 1px solid #e5e7eb;
    background: #ffffff;
    color: #6b7280;
    border-radius: 0.375rem;
    font-size: 0.875rem;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
}

.filter-btn:hover {
    border-color: #3b82f6;
    color: #3b82f6;
    transform: translateY(-1px);
}

.filter-btn.active {
    background: #3b82f6;
    color: white;
    border-color: #3b82f6;
    box-shadow: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
}

.per-page {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    margin-left: auto;
}

.per-page label {
    font-size: 0.875rem;
    color: #6b7280;
    font-weight: 600;
}

.per-page select {
    padding: 8px 12px;
    border: 1px solid #d1d5db;
    background: #ffffff;
    color: #111827;
    border-radius: 0.375rem;
    font-size: 0.875rem;
    cursor: pointer;
    transition: all 0.3s ease;
}

/* ===== TABLE CARD ===== */
.table-card {
    background: #ffffff;
    border: 1px solid #e5e7eb;
    border-radius: 0.5rem;
    padding: 2rem;
    box-shadow: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
    animation: fadeIn 0.4s ease 0.6s backwards;
}

.table-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1.5rem;
    padding-bottom: 1rem;
    border-bottom: 1px solid #e5e7eb;
}

.table-title {
    font-size: 1.125rem;
    font-weight: 700;
    color: #111827;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.table-title svg {
    width: 18px;
    height: 18px;
    color: #3b82f6;
}

.pagination-info {
    font-size: 0.875rem;
    color: #9ca3af;
    font-weight: 600;
}

/* ===== DATA TABLE ===== */
.data-table {
    width: 100%;
    border-collapse: separate;
    border-spacing: 0;
}

.data-table thead th {
    background: #ffffff;
    padding: 12px 1rem;
    text-align: left;
    font-size: 0.875rem;
    font-weight: 600;
    color: #6b7280;
    border-bottom: 2px solid #e5e7eb;
    white-space: nowrap;
    cursor: pointer;
    transition: all 0.3s ease;
}

.data-table thead th:hover {
    background: #ffffff;
    color: #3b82f6;
}

.data-table thead th.sortable::after {
    content: '⇅';
    margin-left: 4px;
    opacity: 0.3;
    font-size: 10px;
}

.data-table thead th.sorting-asc::after {
    content: '↑';
    opacity: 1;
    color: #3b82f6;
}

.data-table thead th.sorting-desc::after {
    content: '↓';
    opacity: 1;
    color: #3b82f6;
}

.data-table tbody tr {
    transition: all 0.3s ease;
    border-bottom: 1px solid #e5e7eb;
    animation: fadeIn 0.3s ease;
}

.data-table tbody tr:hover {
    background: #ffffff;
    transform: scale(1.005);
}

.data-table tbody td {
    padding: 14px 1rem;
    font-size: 1rem;
    color: #111827;
}

/* ===== DURATION BADGE ===== */
.duration-badge {
    display: inline-flex;
    align-items: center;
    gap: 4px;
    padding: 4px 10px;
    background: #dbeafe;
    border-radius: 12px;
    font-size: 0.75rem;
    font-weight: 600;
    color: #3b82f6;
    transition: all 0.3s ease;
}

.duration-badge:hover {
    transform: scale(1.05);
}

.duration-badge svg {
    width: 12px;
    height: 12px;
}

/* ===== STATUS BADGE ===== */
.status-badge {
    display: inline-flex;
    align-items: center;
    gap: 4px;
    padding: 4px 10px;
    border-radius: 12px;
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
    transition: all 0.3s ease;
}

.status-badge:hover {
    transform: scale(1.05);
}

.status-badge.running {
    background: #fef3c7;
    color: #f59e0b;
}

.status-badge.running svg {
    animation: pulse 2s ease-in-out infinite;
}

.status-badge.completed {
    background: #d1fae5;
    color: #10b981;
}

.status-badge svg {
    width: 12px;
    height: 12px;
}

/* ===== LOCATION DISPLAY ===== */
.location-cell {
    display: flex;
    flex-direction: column;
    gap: 4px;
}

.location-ip {
    display: flex;
    align-items: center;
    gap: 4px;
    font-size: 0.75rem;
    color: #9ca3af;
}

.location-ip svg {
    width: 12px;
    height: 12px;
}

.gps-link {
    display: inline-flex;
    align-items: center;
    gap: 4px;
    padding: 4px 8px;
    background: #d1fae5;
    border-radius: 0.25rem;
    font-size: 0.75rem;
    color: #10b981;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.2s ease;
}

.gps-link:hover {
    background: #10b981;
    color: white;
    transform: translateY(-1px);
}

.gps-link svg {
    width: 12px;
    height: 12px;
}

.no-location {
    font-size: 0.875rem;
    color: #9ca3af;
    font-style: italic;
}

/* ===== ACTION BUTTONS ===== */
.actions {
    display: flex;
    gap: 4px;
}

.btn-icon {
    width: 30px;
    height: 30px;
    border: none;
    border-radius: 0.375rem;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-icon svg {
    width: 14px;
    height: 14px;
}

.btn-icon:hover {
    transform: translateY(-1px) scale(1.1);
}

.btn-view {
    background: #dbeafe;
    color: #3b82f6;
}

.btn-view:hover {
    background: #3b82f6;
    color: white;
}

.btn-delete {
    background: #fee2e2;
    color: #ef4444;
}

.btn-delete:hover {
    background: #ef4444;
    color: white;
}

/* ===== PAGINATION ===== */
.pagination {
    display: flex;
    justify-content: center;
    align-items: center;
    gap: 0.5rem;
    margin-top: 2rem;
    padding-top: 1.5rem;
    border-top: 1px solid #e5e7eb;
}

.page-btn {
    padding: 8px 14px;
    border: 1px solid #e5e7eb;
    background: #ffffff;
    color: #6b7280;
    border-radius: 0.375rem;
    font-size: 0.875rem;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
}

.page-btn:hover:not(:disabled) {
    border-color: #3b82f6;
    color: #3b82f6;
    transform: translateY(-1px);
}

.page-btn:disabled {
    opacity: 0.4;
    cursor: not-allowed;
}

.page-btn.active {
    background: #3b82f6;
    color: white;
    border-color: #3b82f6;
    box-shadow: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
}

/* ===== EMPTY STATE ===== */
.empty-state {
    text-align: center;
    padding: 64px 20px;
    animation: fadeIn 0.4s ease;
}

.empty-state svg {
    width: 64px;
    height: 64px;
    color: #9ca3af;
    margin-bottom: 16px;
    animation: pulse 2s ease infinite;
}

.empty-state h3 {
    font-size: 1.125rem;
    color: #6b7280;
    margin-bottom: 8px;
    font-weight: 600;
}

/* ===== RESPONSIVE ===== */
@media (max-width: 992px) {
    .header-content {
        flex-direction: column;
        align-items: stretch;
    }
    
    .header-search {
        max-width: 100%;
    }
    
    .stats-grid {
        grid-template-columns: repeat(2, 1fr);
    }
}

@media (max-width: 768px) {
    .timelogs-page {
        padding: 1rem;
    }
    
    .stats-grid {
        grid-template-columns: 1fr;
    }
}
</style>

<div class="timelogs-page">
    <!-- Page Header -->
    <div class="page-header">
        <div class="header-content">
            <div class="header-left">
                <h1>
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                    </svg>
                    Time Logs
                </h1>
                <p>Track and manage time entries</p>
            </div>
            
            <div class="header-search">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/>
                </svg>
                <input type="text" id="searchInput" placeholder="Search time logs...">
            </div>
        </div>
    </div>

    <!-- Stats Cards -->
    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-icon stat-icon-primary">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"/>
                </svg>
            </div>
            <div class="stat-content">
                <h3 id="totalEntries">{{ number_format($stats['total_entries']) }}</h3>
                <p>Total Entries</p>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon stat-icon-primary">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"/>
                </svg>
            </div>
            <div class="stat-content">
                <h3 id="myEntries">{{ number_format($stats['my_entries']) }}</h3>
                <p>My Entries</p>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon stat-icon-warning">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                </svg>
            </div>
            <div class="stat-content">
                <h3 id="totalHours">{{ number_format($stats['total_hours'], 1) }}h</h3>
                <p>Total Hours</p>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon stat-icon-success">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                </svg>
            </div>
            <div class="stat-content">
                <h3 id="billableHours">{{ number_format($stats['billable_hours'], 1) }}h</h3>
                <p>Billable Hours</p>
            </div>
        </div>
    </div>

    <!-- Filters -->
    <div class="filters-card">
        <div class="filters-row">
            <div class="quick-filters">
                <button class="filter-btn active" data-filter="all" onclick="applyFilter(this, 'all')">All</button>
                <button class="filter-btn" data-filter="my_logs" onclick="applyFilter(this, 'my_logs')">My Logs</button>
                <button class="filter-btn" data-filter="running" onclick="applyFilter(this, 'running')">Running</button>
                <button class="filter-btn" data-filter="billable" onclick="applyFilter(this, 'billable')">Billable</button>
            </div>

            <div class="per-page">
                <label>Show:</label>
                <select id="perPageSelect" onchange="changePerPage(this.value)">
                    <option value="15" selected>15</option>
                    <option value="25">25</option>
                    <option value="50">50</option>
                    <option value="100">100</option>
                </select>
            </div>
        </div>
    </div>

    <!-- Time Logs Table -->
    <div class="table-card">
        <div class="table-header">
            <h3 class="table-title">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M4 6h16M4 10h16M4 14h16M4 18h16"/>
                </svg>
                All Time Logs
            </h3>
            <div class="pagination-info" id="paginationInfo">Showing 0-0 of 0 logs</div>
        </div>

        <table class="data-table">
            <thead>
                <tr>
                    <th style="width: 40px;">
                        <input type="checkbox" id="selectAll" onchange="toggleSelectAll(this)">
                    </th>
                    <th class="sortable" data-column="id">ID</th>
                    <th class="sortable" data-column="task_id">Task</th>
                    <th class="sortable" data-column="admin_id">Member</th>
                    <th class="sortable" data-column="start_time">Start Time</th>
                    <th class="sortable" data-column="duration_minutes">Duration</th>
                    <th>Location</th>
                    <th class="sortable" data-column="is_running">Status</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody id="logsBody">
                <!-- Populated via AJAX -->
            </tbody>
        </table>

        <!-- Pagination -->
        <div class="pagination" id="pagination" style="display: none;">
            <!-- Populated dynamically -->
        </div>
    </div>
</div>

<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script>
let currentPage = 1;
let perPage = 15;
let totalRecords = 0;
let lastPage = 1;
let sortColumn = 'id';
let sortDirection = 'desc';
let selectedIds = [];
let currentFilter = 'all';

$(document).ready(function() {
    loadLogs();

    // Search with debounce
    let searchTimeout;
    $('#searchInput').on('keyup', function() {
        clearTimeout(searchTimeout);
        searchTimeout = setTimeout(() => {
            currentPage = 1;
            loadLogs();
        }, 500);
    });

    // Column sorting
    $('.sortable').on('click', function() {
        const column = $(this).data('column');
        
        if (sortColumn === column) {
            sortDirection = sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            sortColumn = column;
            sortDirection = 'asc';
        }

        $('.sortable').removeClass('sorting-asc sorting-desc');
        $(this).addClass('sorting-' + sortDirection);

        loadLogs();
    });
});

function loadLogs() {
    const filters = {
        search: $('#searchInput').val(),
        filter_type: currentFilter,
        page: currentPage,
        per_page: perPage,
        sort: sortColumn,
        dir: sortDirection
    };

    $.ajax({
        url: '{{ route("admin.timelogs.data") }}',
        type: 'GET',
        data: filters,
        success: function(response) {
            renderLogs(response.data);
            updatePagination(response);
        },
        error: function(xhr) {
            alert('Error: ' + (xhr.responseJSON?.message || 'Unknown error'));
        }
    });
}

function renderLogs(logs) {
    const tbody = $('#logsBody');
    tbody.empty();

    if (logs.length === 0) {
        tbody.append(`
            <tr>
                <td colspan="9">
                    <div class="empty-state">
                        <svg fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                        </svg>
                        <h3>No Time Logs Found</h3>
                        <p>No time logs match your criteria</p>
                    </div>
                </td>
            </tr>
        `);
        return;
    }

    logs.forEach(log => {
        // Format duration - use server-calculated formatted duration for accuracy
        const durationDisplay = log._formatted_duration || '00:00:00';
        
        // Status badge
        const statusBadge = log.is_running 
            ? `<span class="status-badge running">
                <svg fill="currentColor" viewBox="0 0 20 20">
                    <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM9.555 7.168A1 1 0 008 8v4a1 1 0 001.555.832l3-2a1 1 0 000-1.664l-3-2z" clip-rule="evenodd"/>
                </svg>
                Running
            </span>`
            : `<span class="status-badge completed">
                <svg fill="currentColor" viewBox="0 0 20 20">
                    <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"/>
                </svg>
                Completed
            </span>`;

        // Location display - Show Start and End locations
        let locationDisplay = '<div class="location-cell">';
        
        // Start Location
        if (log._has_location) {
            locationDisplay += `
                <div style="display: flex; align-items: center; gap: 4px; margin-bottom: 4px;">
                    <span style="color: #10b981; font-size: 10px; font-weight: 600;">START:</span>
                    <a href="${log._google_maps_url}" target="_blank" class="gps-link" title="Start: ${log._location_string}" style="font-size: 11px;">
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24" style="width: 12px; height: 12px;">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z"/>
                        </svg>
                        View
                    </a>
                </div>`;
        } else {
            locationDisplay += `<div style="font-size: 10px; color: #9ca3af; margin-bottom: 4px;"><span style="color: #10b981;">START:</span> No GPS</div>`;
        }
        
        // End Location
        if (log._has_end_location) {
            const endMapsUrl = `https://www.google.com/maps?q=${log.end_latitude},${log.end_longitude}`;
            const endLocationStr = `${log.end_latitude}, ${log.end_longitude}`;
            locationDisplay += `
                <div style="display: flex; align-items: center; gap: 4px;">
                    <span style="color: #ef4444; font-size: 10px; font-weight: 600;">END:</span>
                    <a href="${endMapsUrl}" target="_blank" class="gps-link" title="End: ${endLocationStr}" style="font-size: 11px;">
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24" style="width: 12px; height: 12px;">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z"/>
                        </svg>
                        View
                    </a>
                </div>`;
        } else if (!log.is_running) {
            locationDisplay += `<div style="font-size: 10px; color: #9ca3af;"><span style="color: #ef4444;">END:</span> No GPS</div>`;
        } else {
            locationDisplay += `<div style="font-size: 10px; color: #f59e0b;"><span style="color: #ef4444;">END:</span> Running...</div>`;
        }
        
        locationDisplay += '</div>';

        const row = `
            <tr>
                <td><input type="checkbox" class="log-checkbox" value="${log.id}" onchange="toggleSelection(this)"></td>
                <td><strong>#${log.id}</strong></td>
                <td>${log.task ? log.task.title : '-'}</td>
                <td>${log.admin ? log.admin.name : '-'}</td>
                <td>${log.start_time ? new Date(log.start_time).toLocaleString() : '-'}</td>
                <td>
                    <span class="duration-badge">
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                        </svg>
                        ${durationDisplay}
                    </span>
                </td>
                <td>${locationDisplay}</td>
                <td>${statusBadge}</td>
                <td>
                    <div class="actions">
                        <a href="${log._show_url}" class="btn-icon btn-view" title="View">
                            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/>
                                <path stroke-linecap="round" stroke-linejoin="round" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"/>
                            </svg>
                        </a>
                        <button class="btn-icon btn-delete" onclick="deleteLog(${log.id})" title="Delete">
                            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/>
                            </svg>
                        </button>
                    </div>
                </td>
            </tr>
        `;
        tbody.append(row);
    });
}

function updatePagination(response) {
    totalRecords = response.total || 0;
    currentPage = response.current_page || 1;
    lastPage = response.last_page || 1;

    const startRecord = totalRecords === 0 ? 0 : ((currentPage - 1) * perPage) + 1;
    const endRecord = Math.min(currentPage * perPage, totalRecords);
    $('#paginationInfo').text(`Showing ${startRecord}-${endRecord} of ${totalRecords} logs`);

    const pagination = $('#pagination');
    
    if (lastPage <= 1) {
        pagination.hide();
        return;
    }

    pagination.show().empty();

    // Previous button
    pagination.append(`
        <button class="page-btn" onclick="goToPage(${currentPage - 1})" ${currentPage === 1 ? 'disabled' : ''}>
            Previous
        </button>
    `);

    // Page numbers
    const maxPages = 7;
    let startPage = Math.max(1, currentPage - Math.floor(maxPages / 2));
    let endPage = Math.min(lastPage, startPage + maxPages - 1);

    if (endPage - startPage < maxPages - 1) {
        startPage = Math.max(1, endPage - maxPages + 1);
    }

    if (startPage > 1) {
        pagination.append(`<button class="page-btn" onclick="goToPage(1)">1</button>`);
        if (startPage > 2) pagination.append(`<span style="padding: 0 8px;">...</span>`);
    }

    for (let i = startPage; i <= endPage; i++) {
        pagination.append(`
            <button class="page-btn ${i === currentPage ? 'active' : ''}" onclick="goToPage(${i})">
                ${i}
            </button>
        `);
    }

    if (endPage < lastPage) {
        if (endPage < lastPage - 1) pagination.append(`<span style="padding: 0 8px;">...</span>`);
        pagination.append(`<button class="page-btn" onclick="goToPage(${lastPage})">${lastPage}</button>`);
    }

    // Next button
    pagination.append(`
        <button class="page-btn" onclick="goToPage(${currentPage + 1})" ${currentPage === lastPage ? 'disabled' : ''}>
            Next
        </button>
    `);
}

function goToPage(page) {
    if (page < 1 || page > lastPage) return;
    currentPage = page;
    loadLogs();
}

function changePerPage(value) {
    perPage = parseInt(value);
    currentPage = 1;
    loadLogs();
}

function applyFilter(button, type) {
    $('.filter-btn').removeClass('active');
    $(button).addClass('active');
    
    currentFilter = type;
    currentPage = 1;
    loadLogs();
}

function toggleSelection(checkbox) {
    const id = parseInt(checkbox.value);
    if (checkbox.checked) {
        selectedIds.push(id);
    } else {
        selectedIds = selectedIds.filter(item => item !== id);
    }
}

function toggleSelectAll(checkbox) {
    $('.log-checkbox').prop('checked', checkbox.checked);
    selectedIds = checkbox.checked ? 
        $('.log-checkbox').map(function() { return parseInt(this.value); }).get() : [];
}

function deleteLog(logId) {
    if (!confirm('Delete this time log?')) return;

    $.ajax({
        url: `/admin/timelogs/${logId}`,
        type: 'DELETE',
        headers: {'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')},
        success: function(response) {
            if (response.success) {
                alert(response.message);
                loadLogs();
                updateStats();
            }
        },
        error: function(xhr) {
            alert('Delete failed: ' + (xhr.responseJSON?.message || 'Error'));
        }
    });
}

function updateStats() {
    $.ajax({
        url: '{{ route("admin.timelogs.stats") }}',
        type: 'GET',
        success: function(stats) {
            $('#totalEntries').text(stats.total_entries.toLocaleString());
            $('#myEntries').text(stats.my_entries.toLocaleString());
            $('#totalHours').text(stats.total_hours.toFixed(1) + 'h');
            $('#billableHours').text(stats.billable_hours.toFixed(1) + 'h');
        }
    });
}
</script>